extensions [table matrix]

globals [reference-system lefty-system righty-system
  ;; **slider** individual-percent ; how much individual turtles vary from their reference systems
  ; Each group, lefties and righties, has a mean for their systems, individual-percent
  ; is used to build in individual variation.
  ; leaders-talk-with-other-leaders ;; This is a counter, for counting the 
  ; number of times a leader of one breed talks to a leader of the other breed.
  ; We have decided for now not to tract this, but are leaving in the code
  ; to do so, if we choose to uncomment it.
  ;; **slider** dasd the standard deviation, applied to generating individual examples
  current-vowel
  lefty-confusions
  righty-confusions
  myticks
  dateandtime
  
  bob carol ted alice
  bob_under_score
]
breed [lefties lefty]
breed [righties righty]

turtles-own [exemplars ;; The history list of what the turtle has heard or thinks it has heard,
  ;; of length count-exemplars (slider on the Interface)
  ;; exemplars, a turtles variable, holds a table whose keys are the vowel IDs and
  ;; whose values are lists of 2-tuple lists, of first and second formant pairs.
  ;; After initialization, we assume that the ordering of the pairs is chronological,
  ;; with the oldest at the front, the youngest at the end.
  exemplar-means 
  exemplar-ratios 
  leader? ;; Is this turtle a leader?
  category-list ;; Is a list of pairs (lists) in which the first member is the signal
  ; received and the second is the guess. So, when they agree, the agent has 
  ; guessed correctly, and if not, not.
  radius-mean ; Gets the value lefty-radius-mean, if the turtle is a lefty, and 
  ; righty-radius mean, if the turtle is a righty. These values are given as sliders
  ; on the interface.
  myneighbors ; other turtles in-radius get-a-radius
] 

to setup
  clear-all
  if file-exists? "run-pretties.txt"
  [file-delete "run-pretties.txt"]
  if file-exists? "run-uglies.txt"
  [file-delete "run-uglies.txt"]
  if file-exists? "exemplar-means.csv"
  [file-delete "exemplar-means.csv"]
  if file-exists? "confusions.csv"
  [file-delete "confusions.csv"]
  make-reference-system
  set lefty-system make-group-system(reference-system)(percentshift)("down")
  set righty-system make-group-system(reference-system)(percentshift)("up")
  if (Scenarios = "User Interface")
    [
    set lefty-system make-group-system(reference-system)(percentshift)("down")
    set righty-system make-group-system(reference-system)(percentshift)("up")
    ]
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;; end of "User Interface"  ;;;;;;;;;;;;;;;;;;;;;
;  if (Scenarios = "Zero Model")
;    [
;  set individual-percent 0.05
;  set percentshift 0.0
;  set lefty-system make-group-system(reference-system)(percentshift)("down")
;  set righty-system make-group-system(reference-system)(percentshift)("up")
;  ;set LeaderNetworks false
;  set numberofleftyleaders 0
;  set numberofrightyleaders 0
;  set lefty-radius-mean 10
;  set righty-radius-mean 10
;    ]
 
  ;; Create confusion tables, one for lefties, one for righties
  let num-vowels length table:keys reference-system
  set lefty-confusions matrix:make-constant num-vowels num-vowels 0
;  show matrix:pretty-print-text lefty-confusions
;  show matrix:dimensions lefty-confusions
  set righty-confusions matrix:make-constant num-vowels num-vowels 0

  ;; Create the lefties on the left and the righties on the  ... right!
  ;; We are aware that unless we change the world size from the default, 
  ;; there are more righties than lefties.
  ask patches with [pxcor < 0] [sprout-lefties 1 [set radius-mean lefty-radius-mean]
    ]
  ask patches with [pxcor >= 0] [sprout-righties 1  [set radius-mean righty-radius-mean]
    ]

  ask lefties [set exemplars make-some-exemplars(count-exemplars)(lefty-system)(individual-percent)
    make-exemplar-means
    set shape "square"
    ;set color newcolor(mean exemplars)
    ]
  ;; ok to here
    
  ask righties [set exemplars make-some-exemplars(count-exemplars)(righty-system)(individual-percent)
    make-exemplar-means
    set shape "square"
    ;set color newcolor(mean exemplars)
    ]
  
  ask turtles [
    make-exemplar-ratios
    set leader? false
    set category-list []
    let lowhi set-color
    set color scale-color green get-vowel-value(VowelFollowed)(FormantFollowed) first lowhi last lowhi
    ]
  ask turtles [
    set myneighbors other turtles in-radius get-a-radius
  ]
  create-leaders
  ask lefties  with [leader? = true] 
   [create-links-with other n-of ((random lefty-leader-connectivity) + 1) lefties with [leader? = true]]
  ask righties  with [leader? = true] 
   [create-links-with other n-of ((random righty-leader-connectivity) + 1) righties with [leader? = true]]
  
  reset-ticks
  if file-exists? "exemplar-means.csv"
  [file-delete "exemplar-means.csv"]
  ;; Print the headers for the file:
  file-open "exemplar-means.csv"
  ;file-print (word "+++++ " ticks " +++++") 
  let toPrint (word "tick,Vowel,mean_turtles_first,mean_turtles_second,var_turtles_first,var_turtles_second")
  set toPrint (word toPrint ",mean_lefties_first,mean_lefties_second,mean_righties_first,mean_righties_second,variance_lefties_first")
  set toPrint (word toPrint ",variance_lefties_second,variance_righties_first,variance_righties_second") 
  set toPrint (word toPrint ",mean_leaders_lefties_first,mean_leaders_lefties_second,mean_leaders_righties_first,mean_leaders_righties_second")
  set toPrint (word toPrint ",var_leaders_lefties_first,var_leaders_lefties_second,var_leaders_righties_first,var_leaders_righties_second")
  file-print toPrint
  file-close-all
  ;; Now write the data:
  record-breed-formant-stats
  
  set dateandtime date-and-time
  
end ;; End of setup
;;;;;;;;;;;;;;;;;;;; end of setup ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to set-scenarios
;  if (Scenarios = "Zero Model")
;    [
;  set individual-percent 0.05
;  set percentshift 0.0
;;  set lefty-system make-group-system(reference-system)(percentshift)("down")
;;  set righty-system make-group-system(reference-system)(percentshift)("up")
;  ;set LeaderNetworks false
;  set numberofleftyleaders 0
;  set numberofrightyleaders 0
;  set lefty-radius-mean 10
;  set righty-radius-mean 10
;    ]
  if (Scenarios = "Base Model")
  [set reps 40000
   set numberofleftyleaders 80
   set numberofrightyleaders 80
   set leaderschange true
   set dasd 4.50
   set count-exemplars 100
   set leader-effect round (100 * 60 / count-exemplars)
   set leader-leader-effect false
   set pleaderleader 0

   set lefty-radius-mean 5
   set righty-radius-mean 5
   set lefty-leader-connectivity 5
   set righty-leader-connectivity 5

   set percentshift 0.10
   set bigotry 0
   set individual-percent 0.05
  ]
end


;;;;;;;;;;;;;;; end of set-scenarios ;;;;;;;;;;;;;;;;;;;;;;;

to create-leaders
  let lefty-leaders n-of numberofleftyleaders lefties
  ask lefty-leaders [set shape "dot"
    set leader? true]
  let righty-leaders n-of numberofrightyleaders righties
  ask righty-leaders [set shape "dot"
    set leader? true]
end


to go
  ask turtles [
   if (leaderschange  or leader? = false) [   
     let withwhom nobody
     ifelse (leader? = true and random-float 1 <= pleaderleader) ; and leaderstalkleaders)
      [set withwhom one-of link-neighbors]
      [ifelse ((random-float 1) < bigotry)
        [let mybreed [breed] of self 
        set withwhom one-of myneighbors with [breed = mybreed]]
        [set withwhom one-of myneighbors]]
  if withwhom != nobody [ ;; So we have somebody
   ; Initialize utterace
   let utterance 0 
   ;; Ask the interlocutor to say something and you record it:
   ask withwhom [set utterance utter-vowel] ;; Key: this is the interlocutor speaking.
   ;; utterance is now a 3-tuple list. First is the actual vowel, second is the first formant, third is the second formant
   map-utterance-to-vowel(utterance)
   ;; Find out what vowel you think you heard. Get the ID
   let daVowel get-last-mapped-vowel
   set current-vowel daVowel
   ;; instance is just the formants, of what you heard. It's a 2-list, first and second formants.
   ;; daVowel is what you think you heard.
   let instance (list item 1 utterance item 2 utterance)
   ;; add-exemplar-instance(current-vowel)(instance)
   ;; Now the new basic logic==>
   ;; (1)
   ;; (not leader) and (interlocutor is a leader) and (you have the same breed)
   if (leader? = false and [leader? = true] of withwhom and [breed] of withwhom = [breed] of self)
    [ask self [repeat leader-effect [add-exemplar-instance(current-vowel)(instance)]] ]  
   ;; (2)  
   ;; (leader) and (interlocutor is a leader) and (you have the same breed)           
   if (leader? = true and [leader? = true] of withwhom and [breed] of withwhom = [breed] of self)
    [ifelse leader-leader-effect = true
     ;; Leaders act as break on other leaders:
    [ask self [repeat leader-effect [add-exemplar-instance(current-vowel)(instance)]] ]   
   ;; Else do a normal accommation, not a leader accommodation, since you are a leader.
   ;; Immediately below: leader on leader becomes normal accommodation:
    [ask self [add-exemplar-instance(current-vowel)(instance)]  ]
    ]
   ;; (3)
   ;; (not leader) and (interlocutor is not a leader)
   if (leader? = false and [leader? = false] of withwhom)  
   ;; You are both not leaders, do a normal accommodation
    [ask self [add-exemplar-instance(current-vowel)(instance)]] 
   ;; (4) 
   ;; (I am a leader) and ((interlocutor is not a leader) or ((is a leader) and (of a different breed from me)))  
   if (leader? = true and (([leader? = false] of withwhom) or ([leader? = true] of withwhom and [breed] of withwhom != [breed] of self)))
   ;; If your are a leader and either (a) your interlocutor is not a leader or (b) it is, but it's a leader of another breed
   ;; then to a normal adjustment/accommodation
    [ask self [add-exemplar-instance(current-vowel)(instance)]]
  
  ;; Update the stats.
  make-new-vowel-mean(current-vowel)
  make-new-vowel-ratio(current-vowel)
  ] ;; if withwhom != nobody
   ] ;; if (leaderschange  or leader? = false)
  ] ;; end of ask turtles
  set myticks (myticks + 1)
  ;print myticks
;] ;; end of repeat
  let lowhi set-color
  ask turtles [set color scale-color green get-vowel-value(VowelFollowed)(FormantFollowed) first lowhi last lowhi]
;  ifelse myticks mod 10 = 0
;  [tick
;   set myticks 0
;   ;print "ticking"
;   ]
;  [;print "not ticking"
;    ]
;  set myticks (myticks + 1)
;  if ticks > 5
;  [let file (word date-and-time ".csv")
;    export-all-plots file
;    print "Stopping."
;    stop]
   tick
   set dateandtime date-and-time
   if ticks mod 100 = 0 [
   file-open "run-pretties.txt"
   file-print (word "+++++ " ticks " +++++")
   file-print matrix:pretty-print-text lefty-confusions
   file-print matrix:pretty-print-text righty-confusions
   file-close-all
   file-open "confusions.csv"
   ;file-print (word "+++++" ticks "+++++")
   ;file-print (word "******** lefty-confusions ***********")
   dumpmatrixcsv lefty-confusions "left"
   ;file-print (word "******** righty-confusions ***********")
   dumpmatrixcsv righty-confusions "right"
   file-close-all
   record-uglies
   ;; Now record the exemplar means for lefties and righties.
   record-breed-formant-stats
   ]
end ;; End of go

to-report set-color
 let da-formant-value get-formant-value(VowelFollowed)(FormantFollowed)
 let displacement (0.2 + percentshift)
 let lowscale da-formant-value / 2 ;(1 - displacement) * da-formant-value
 let hiscale da-formant-value * 1.5 ;(1 + displacement) * da-formant-value
 report (list lowscale hiscale)
end

to add-exemplar-instance [key pair]
  let daExemplars table:get exemplars key
  set daExemplars but-first daExemplars
  set daExemplars lput pair daExemplars
  table:put exemplars key daExemplars
end

;to test
;;  set bob make-group-system(reference-system)(0.05)("up")
;;  ;show bob
;;  ;show uniform-interval(100)(0.05)
;;  set lefty-system make-group-system(reference-system)(0.05)("down")
;;  show lefty-system
;
;  let daResult 0
;  ask one-of turtles [set daResult utter-vowel]
;  show daResult
;  print "Yo!"
;  ask one-of turtles [map-utterance-to-vowel(daResult)
;     
;    show category-list
;    ;; Find out what vowel you think you heard. Get the ID
;    let daVowel get-last-mapped-vowel
;    show daVowel
;    let daExemplars table:get exemplars daVowel
;    show daExemplars
;    set daExemplars but-first daExemplars
;    set daExemplars lput (list item 1 daResult item 2 daResult) daExemplars
;    show daExemplars
;    table:put exemplars daVowel daExemplars
;    show table:get exemplars daVowel
;    show table:get exemplar-means daVowel
;    show table:get exemplar-ratios daVowel
;    make-new-vowel-mean(daVowel)
;    make-new-vowel-ratio(daVowel)
;    show table:get exemplar-means daVowel
;    show table:get exemplar-ratios daVowel
;    ]
;  ;; update as appropriate
;  
;end

to-report get-last-mapped-vowel
  report last last category-list
end


to map-utterance-to-vowel [daUtterance]
  ;; New version. Looks at ratio of formants.
  ;; In the turtle context. The turtle is listening to the interlocutor.
  ;; The turtle needs to guess which vowel it heard. It then maps and
  ;; records daUtterance.
  ;; daUtterance is [<speaker's vowel> <utterance first formant> <utterance second formant>]
  let daFirst item 1 daUtterance
  let daSecond item 2 dautterance
  let perceptual-ratio (daSecond / daFirst)
  let distances []
  foreach table:keys exemplar-ratios [
   set distances lput abs((table:get exemplar-ratios ?) - perceptual-ratio) distances
  ]
  let daMin min distances
  let daMinValue (position daMin distances + 1)
  set category-list lput (list item 0 daUtterance daMinValue) category-list
  ;;; Steve on 2014-11-20:
  if length category-list > 5 ; 2015-02-06: A bit puzzling but seems innocuous. Have to limit it somehow.
  [set category-list but-first category-list]
  
  let speaker-vowel-index (item 0 daUtterance - 1)
  let hearer-vowel-index (daMinValue - 1)
  ifelse breed = lefties
  [matrix:set lefty-confusions speaker-vowel-index hearer-vowel-index (1 + matrix:get lefty-confusions speaker-vowel-index hearer-vowel-index) ]
  [matrix:set righty-confusions speaker-vowel-index hearer-vowel-index (1 + matrix:get righty-confusions speaker-vowel-index hearer-vowel-index)]
end

to-report utter-vowel
  ; To allow utterance alternatives in the future:
  ;if concept-method = "hybrid"
  ;[report utter-vowel-hybrid]
  ;if concept-method = "pure exemplar"
  ;[report utter-vowel-pure-exemplar]
  report utter-vowel-hybrid
end

to pp
  print matrix:pretty-print-text lefty-confusions
  print matrix:pretty-print-text righty-confusions
end

to-report utter-vowel-hybrid
  ;; For turtles.
  ;; Reports a list whose first value is the key of the chosen
  ;; vowel, whose second value is its perturbed first formant,
  ;; and whose third value is its perturbed second formant.
  ;;
  ;; Find out how many vowels there are.
  let num-vowels length table:keys exemplar-means
  ;; Pick a vowel at random, uniformly.
  let the-vowel ((random num-vowels) + 1)
  let daFormants table:get exemplar-means the-vowel
  let daFirst first daFormants
  let daSecond last daFormants
  let daFirstPerturbed max (list 0.1 random-normal daFirst dasd)
  let daSecondPerturbed max (list 0.1 random-normal daSecond dasd)
  report (list the-vowel daFirstPerturbed daSecondPerturbed)
end

to-report utter-vowel-pure-exemplar
  ;; Find out how many vowels there are.
  let num-vowels length table:keys exemplar-means
  ;; Pick a vowel at random, uniformly.
  let the-vowel ((random num-vowels) + 1)
  let daFormants get-pure-exemplar the-vowel 10
  
  let daFirst first daFormants
  let daSecond last daFormants
  let daFirstPerturbed max (list 0.1 random-normal daFirst dasd)
  let daSecondPerturbed max (list 0.1 random-normal daSecond dasd)
  report (list the-vowel daFirstPerturbed daSecondPerturbed)
end



to-report get-a-radius
  report (random-exponential radius-mean) + 1
end

to-report mean-exemplars
  ;; sok 2015-06-19 This reporter no longer works and is isolated, 
  ;; not used.
  let lefty-mean-accum []
  ask lefties [
    set lefty-mean-accum fput mean exemplars lefty-mean-accum
  ]
  let righty-mean-accum []
  ask righties [
    set righty-mean-accum fput mean exemplars righty-mean-accum
  ]
  report (list mean lefty-mean-accum variance lefty-mean-accum 
     mean righty-mean-accum variance righty-mean-accum)
  
end

to-report newcolor [myscore]
  report  max (list 0 (((floor (myscore / 10 )) * 10) + 5))
end

to make-reference-system
  set reference-system table:make
  table:put reference-system 1 (list 299 2190)
  table:put reference-system 2 (list 364 2260)
  table:put reference-system 3 (list 481 1980)
  table:put reference-system 4 (list 750 1250)
  table:put reference-system 5 (list 377 1400)
  table:put reference-system 6 (list 290 630)
  table:put reference-system 7 (list 392 690)
  table:put reference-system 8 (list 502 790)
end



to-report  make-group-system [ref daPercent upordown]
  let new-system table:make
  foreach table:keys ref [
    let daEntry table:get ref ?
    let first-formant first daEntry 
    let second-formant last daEntry
;    let new-first-formant uniform-interval(first-formant)(daPercent)
;    let new-second-formant uniform-interval(second-formant)(daPercent)
    let new-first-formant fixed-offset(first-formant)(daPercent)(upordown)
    let new-second-formant fixed-offset(second-formant)(daPercent)(upordown)
    table:put new-system ? (list new-first-formant new-second-formant)    
    
  ]
  report new-system
end

to-report fixed-offset [formant daPercent upordown]
  if upordown = "down"
   [report formant * (1 - daPercent)]
  if upordown = "up"
   [report formant * (1 + daPercent)]
end
  
to-report uniform-interval [daCenter daPercent]
  ; daPercent should be a low number, such as 0.1
  let low daCenter - (daCenter * daPercent)
  let high daCenter + (daCenter * daPercent)
  report (random-float (high - low))  + (daCenter - (daCenter * daPercent))
end

to-report make-some-exemplars [n ref-table daPercent]
  ;let toreport []
  let my-table table:make
  foreach table:keys ref-table [
    let daList []
    let daEntry table:get ref-table ?
    let first-formant first daEntry 
    let second-formant last daEntry
    let new-first-formant 0
    let new-second-formant 0
    repeat n [
    set new-first-formant uniform-interval(first-formant)(daPercent)
    set new-second-formant uniform-interval(second-formant)(daPercent)    
    set daList fput (list new-first-formant new-second-formant) daList
    ]
    table:put my-table ? daList    
  ]
  report my-table
end

to make-exemplar-means
  set exemplar-means table:make
  foreach table:keys exemplars [
  make-new-vowel-mean(?)
  ]
end

to make-new-vowel-mean [daVowel]
  let llist []
  let hlist []
  let theItems table:get exemplars daVowel
  let n length theItems
  foreach n-values n [?] [
    set llist fput (first item ? theItems) llist
    set hlist fput (last item ? theItems) hlist
  ]
  table:put exemplar-means daVowel (list mean llist mean hlist)
end

to make-exemplar-ratios
  set exemplar-ratios table:make
  foreach table:keys exemplar-means [
    make-new-vowel-ratio(?)
  ] 
end

to make-new-vowel-ratio [daVowel]
;  foreach table:keys exemplar-means [
;    let daValues table:get exemplar-means ?
;    let daFirst first daValues
;    let daSecond last daValues
;    table:put exemplar-ratios ? (daSecond / daFirst)
;  ]
    let daValues table:get exemplar-means daVowel
    let daFirst first daValues
    let daSecond last daValues
    table:put exemplar-ratios daVowel (daSecond / daFirst)
end

;to debug
;  foreach table:keys exemplar-means [
;    print ?
;  ]
;end

to-report get-vowel-variance [LorR vNum fNum]
  ;; LorR should be lefties or righties
  ;; vNum is 1, 2, ... 8 the vowel number
  ;; fNum should be 1, 2, whatever, the formant number
  let daList []
  ask LorR  [
   set daList lput (item (fNum - 1) table:get exemplar-means vNum) daList 
  ]
  report variance daList
end

to-report get-vowel-value [vNum fNum]
  ;; vNum is 1, 2, ... 8 the vowel number
  ;; fNum should be 1, 2, whatever, the formant number
  report item (fNum - 1) table:get exemplar-means vNum
end

to-report get-vowel-means [LorR vNum fNum]
  ;; LorR should be lefties or righties
  ;; vNum is 1, 2, ... 8 the vowel number
  ;; fNum should be 1, 2, whatever, the formant number
  let daList []
  ask LorR  [
   set daList lput (item (fNum - 1) table:get exemplar-means vNum) daList 
  ]

;  let daList []
;  ifelse LorR = "lefties"
;  [ ask lefties [let daPair table:get exemplar-means daVowel
;      let daFirst first daPair
;      set daList lput daFirst daList
;  ]
;  ]
;  [
;    ask righties [let daPair table:get exemplar-means daVowel
;      let daFirst first daPair
;      set daList lput daFirst daList
;  ]
;  ]
    report mean daList
end

to-report get-vowel-means-first [LorR daVowel]
  let daList []
  ifelse LorR = "lefties"
  [ ask lefties [let daPair table:get exemplar-means daVowel
      let daFirst first daPair
      set daList lput daFirst daList
  ]
  ]
  [
    ask righties [let daPair table:get exemplar-means daVowel
      let daFirst first daPair
      set daList lput daFirst daList
  ]
  ]
    report mean daList
end

to-report get-vowel-means-second [LorR daVowel]
  let daList []
  ifelse LorR = "lefties"
  [ ask lefties [let daPair table:get exemplar-means daVowel
      let daSecond last daPair
      set daList lput daSecond daList
  ]
  ]
  [
    ask righties [let daPair table:get exemplar-means daVowel
      let daSecond last daPair
      set daList lput daSecond daList
  ]
  ]
    report mean daList
end

;to test-leader-exemplar-mean
;  print leader-exemplar-mean(lefties)(4)(1)
;end

to-report leader-exemplar-mean [LorR vNum fNum]
  ;; LorR should be lefties or righties
  ;; fNum should be 1, 2, whatever
  let daValues []
  ask LorR with [leader? = true] [
   set daValues lput (item (fNum - 1) table:get exemplar-means vNum) daValues 
  ]
  report mean daValues
end

to-report leader-exemplar-variance [LorR vNum fNum]
  ;; LorR should be lefties or righties
  ;; fNum should be 1, 2, whatever
  let daValues []
  ask LorR with [leader? = true] [
   set daValues lput (item (fNum - 1) table:get exemplar-means vNum) daValues 
  ]
  report variance daValues
end

to-report get-formant-value [daVowel daFormant]
  let daPair table:get reference-system daVowel
  report item (daFormant - 1) daPair
end

to record-breed-formant-stats 
;print mean [item 0 table:get exemplar-means 1] of lefties
;269.09881666602894
;  print mean [item 0 table:get exemplar-means 1] of lefties
; ask turtle 1333 [print table:keys exemplar-means]
; [1 2 3 4 5 6 7 8]
;file-open "exemplar-means.csv"
;;file-print (word "+++++ " ticks " +++++") 
;let toPrint (word "tick,Vowel,mean_turtles_first,mean_turtles_second,var_turtles_first,var_turtles_second")
;set toPrint (word toPrint ",mean_lefties_first,mean_lefties_second,mean_righties_first,mean_righties_second,variance_lefties_first")
;set toPrint (word toPrint ",variance_lefties_second,variance_righties_first,variance_righties_second") 
;set toPrint (word toPrint ",mean_leaders_lefties_first,mean_leaders_lefties_second,mean_leaders_righties_first,mean_leaders_righties_second")
;set toPrint (word toPrint ",var_leaders_lefties_first,var_leaders_lefties_second,var_leaders_righties_first,var_leaders_righties_second")
;file-print toPrint
;file-close-all

let mean_leaders_lefties_first -1
let mean_leaders_lefties_second -1
let mean_leaders_righties_first -1
let mean_leaders_righties_second -1
let var_leaders_lefties_first -1
let var_leaders_lefties_second -1
let var_leaders_righties_first -1
let var_leaders_righties_second -1
  
file-open "exemplar-means.csv"
foreach [table:keys exemplar-means] of turtle 0 [
  set mean_leaders_lefties_first -1
  set mean_leaders_lefties_second -1
  set mean_leaders_righties_first -1
  set mean_leaders_righties_second -1
  set var_leaders_lefties_first -1
  set var_leaders_lefties_second -1
  set var_leaders_righties_first -1
  set var_leaders_righties_second -1
  if (numberofleftyleaders > 0)
  [set mean_leaders_lefties_first leader-exemplar-mean(lefties)(?)(1)
   set mean_leaders_lefties_second leader-exemplar-mean(lefties)(?)(2)
   set var_leaders_lefties_first leader-exemplar-variance(lefties)(?)(1)
   set var_leaders_lefties_second leader-exemplar-variance(lefties)(?)(2)
  ] 
  if (numberofrightyleaders > 0)
  [set mean_leaders_righties_first leader-exemplar-mean(righties)(?)(1)
   set mean_leaders_righties_second leader-exemplar-mean(righties)(?)(2)
   set var_leaders_righties_first leader-exemplar-variance(righties)(?)(1)
   set var_leaders_righties_second leader-exemplar-variance(righties)(?)(2)
  ] 
  
  file-print (word ticks "," ? 
    "," get-vowel-means(turtles)(?)(1) "," get-vowel-means(turtles)(?)(2)
    "," get-vowel-variance(turtles)(?)(1) "," get-vowel-variance(turtles)(?)(2)
    "," mean [item 0 table:get exemplar-means ?] of lefties "," mean [item 1 table:get exemplar-means ?] of lefties ","
    mean [item 0 table:get exemplar-means ?] of righties "," mean [item 1 table:get exemplar-means ?] of righties 
    "," get-vowel-variance(lefties)(?)(1)  "," get-vowel-variance(lefties)(?)(2)  ;LorR vNum fNum])
    "," get-vowel-variance(righties)(?)(1)  "," get-vowel-variance(righties)(?)(2)
    "," mean_leaders_lefties_first  "," mean_leaders_lefties_second
    "," mean_leaders_righties_first  ","  mean_leaders_righties_second
    "," var_leaders_lefties_first "," var_leaders_lefties_second
    "," var_leaders_righties_first "," var_leaders_righties_second 
  ) ; end of file-print
  ]
file-close-all
end

to record-variances
  
end

to record-uglies
  file-open "run-uglies.txt"
  ;; Get the size of the confusion matrices, which will be square
  let daSize item 0 matrix:dimensions lefty-confusions
  file-print (word "+++++ " ticks " +++++")
  foreach n-values daSize [?] [
   let i ?
   foreach n-values daSize [?] [
     let j ?
     let towrite matrix:get lefty-confusions i j
     file-type towrite
     if j < (daSize - 1)
     [file-type ","]
   ] 
   file-type "\n"
  ]
  foreach n-values daSize [?] [
   let i ?
   foreach n-values daSize [?] [
     let j ?
     let towrite matrix:get righty-confusions i j
     file-type towrite
     if j < (daSize - 1)
     [file-type ","]
   ] 
   file-type "\n"
  ]
  file-close-all
end

to-report get-pure-exemplar [which-vowel mymu] ; Only called in the pure exemplar context.
  let exemplar-list table:get exemplars which-vowel
  let locus random-exponential mymu
  ;print locus
  set locus floor locus
  ;print locus
  if locus >= length exemplar-list
  [set locus length exemplar-list - 1]
  set locus (length exemplar-list - 1 - locus)
  ;;print (list length exemplar-list " " locus " yo 235")
  if length exemplar-list = 0
  [print "I'm out of here. In get-pure-exemplar."
    let clark 0
    let robin 17 / clark]
  ;print exemplar-list
  report item locus exemplar-list
end

;to-report say-yo
;  print "Yo!"
;  report "Done!"
;end

to test
  ask lefty 1380 [set bob get-pure-exemplar 4 10]
  print bob
end

to testdumpconfusions
;file-open "run-pretties.txt"
;   file-print (word "+++++ " ticks " +++++")
;   file-print matrix:pretty-print-text lefty-confusions
;   file-print matrix:pretty-print-text righty-confusions
;   file-close-all
let dims matrix:dimensions lefty-confusions
let rows item 0 dims
let cols item 1 dims
print rows
print cols
foreach n-values rows [?]
[let row ?
 let darow matrix:get-row lefty-confusions row
 ;print darow
 let dastr (word item 0 darow)
 let dalength length darow
 foreach n-values (dalength - 1) [?]
 [let index ?
  set dastr (word dastr "," item (index + 1) darow)
 ]
 ;set dastr (word dastr  "\n")
 print dastr
  
]
end

to dumpconfusionscsv
  file-open "confusions.csv"
  ;file-print (word "+++++" ticks "+++++")
  ;file-print (word "******** lefty-confusions ***********")
  dumpmatrixcsv lefty-confusions "left"
  ;file-print (word "******** righty-confusions ***********")
  dumpmatrixcsv righty-confusions "right"
  file-close-all
end

to dumpmatrixcsv [amatrix rightorleft]
let dims matrix:dimensions amatrix
let rows item 0 dims
let cols item 1 dims

foreach n-values rows [?]
[let row ?
 let darow matrix:get-row amatrix row
 ;print darow
 let dastr (word ticks "," rightorleft "," item 0 darow)
 let dalength length darow
 foreach n-values (dalength - 1) [?]
 [let index ?
  set dastr (word dastr "," item (index + 1) darow)
 ]
 ;set dastr (word dastr  "\n")
 file-print dastr
  
]  
end


;;;;;;;;;;; sok 5/2/15 and with robin 5/4/15 testing utter-vowel

to examine-utter-vowel
  let thelefties []
  let therighties []
  repeat 100 [
    let result test-utter-vowel
    ;print result
    set thelefties lput item 0 result thelefties
    set therighties lput item 1 result therighties
  ]
  print (word mean thelefties " " mean therighties)
  
end

to-report test-utter-vowel
  let leftyratios []
  let rightyratios []
  ask lefties [let u utter-vowel
    let a item 0 u
    let b item 1 u
    let c item 2 u
    if a = 4 [set leftyratios lput (c / b) leftyratios]
  ]
  ask righties [let u utter-vowel
    let a item 0 u
    let b item 1 u
    let c item 2 u
    if a = 4 [set rightyratios lput (c / b) rightyratios]
  ]
  ;print (list variance leftyratios "  " variance rightyratios)
  report (list variance leftyratios variance rightyratios)
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
769
590
30
30
9.0
1
10
1
1
1
0
0
0
1
-30
30
-30
30
1
1
1
ticks
30.0

BUTTON
17
107
83
140
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
24
168
182
201
go
print date-and-time\n;repeat reps [go]\nif ticks < reps [go]\nprint date-and-time\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
14
265
208
298
numberofleftyleaders
numberofleftyleaders
0
1000
0
1
1
NIL
HORIZONTAL

SWITCH
14
333
189
366
leaderschange
leaderschange
0
1
-1000

SLIDER
12
423
184
456
leader-effect
leader-effect
1
count-exemplars
60
1
1
NIL
HORIZONTAL

SLIDER
210
591
382
624
lefty-radius-mean
lefty-radius-mean
2
20
5
1
1
NIL
HORIZONTAL

SLIDER
13
501
185
534
pleaderleader
pleaderleader
0
1.0
0
0.01
1
NIL
HORIZONTAL

SLIDER
12
384
184
417
dasd
dasd
0.01
100.0
4.5
0.01
1
NIL
HORIZONTAL

SWITCH
13
463
184
496
leader-leader-effect
leader-leader-effect
1
1
-1000

MONITOR
10
687
180
732
Lefty vowel mean
get-vowel-means(lefties)(VowelFollowed)(FormantFollowed)
3
1
11

PLOT
780
10
1206
188
Lefty First Formants
NIL
NIL
0.0
500.0
215.0
850.0
true
true
"" ""
PENS
"a" 1.0 0 -13840069 true "" "plot get-vowel-means-first(\"lefties\")(4)"
"u" 1.0 0 -16777216 true "" "plot get-vowel-means-first(\"lefties\")(8)"
"epsilon" 1.0 0 -1184463 true "" "plot get-vowel-means-first(\"lefties\")(3)"
"o" 1.0 0 -6459832 true "" "plot get-vowel-means-first(\"lefties\")(7)"
"schwa" 1.0 0 -955883 true "" "plot get-vowel-means-first(\"lefties\")(5)"
"e" 1.0 0 -2674135 true "" "plot get-vowel-means-first(\"lefties\")(2)"
"i" 1.0 0 -7500403 true "" "plot get-vowel-means-first(\"lefties\")(1)"
"open o" 1.0 0 -10899396 true "" "plot get-vowel-means-first(\"lefties\")(6)"

PLOT
780
188
1206
360
Righty First Formants
NIL
NIL
0.0
500.0
215.0
850.0
true
true
"" ""
PENS
"a" 1.0 0 -13840069 true "" "plot get-vowel-means-first(\"righties\")(4)"
"u" 1.0 0 -16777216 true "" "plot get-vowel-means-first(\"righties\")(8)"
"epsilon" 1.0 0 -1184463 true "" "plot get-vowel-means-first(\"righties\")(3)"
"o" 1.0 0 -6459832 true "" "plot get-vowel-means-first(\"righties\")(7)"
"schwa" 1.0 0 -955883 true "" "plot get-vowel-means-first(\"righties\")(5)"
"e" 1.0 0 -2674135 true "" "plot get-vowel-means-first(\"righties\")(2)"
"i" 1.0 0 -7500403 true "" "plot get-vowel-means-first(\"righties\")(1)"
"open o" 1.0 0 -10899396 true "" "plot get-vowel-means-first(\"righties\")(6)"

SLIDER
596
590
768
623
percentshift
percentshift
0
0.2
0.1
0.01
1
NIL
HORIZONTAL

PLOT
780
359
1206
532
Lefty Second Formants
NIL
NIL
0.0
500.0
215.0
400.0
true
true
"" ""
PENS
"a" 1.0 0 -13840069 true "" "plot get-vowel-means-second(\"lefties\")(4)"
"u" 1.0 0 -16777216 true "" "plot get-vowel-means-second(\"lefties\")(8)"
"epsilon" 1.0 0 -1184463 true "" "plot get-vowel-means-second(\"lefties\")(3)"
"o" 1.0 0 -6459832 true "" "plot get-vowel-means-second(\"lefties\")(7)"
"schwa" 1.0 0 -955883 true "" "plot get-vowel-means-second(\"lefties\")(5)"
"e" 1.0 0 -2674135 true "" "plot get-vowel-means-second(\"lefties\")(2)"
"i" 1.0 0 -7500403 true "" "plot get-vowel-means-second(\"lefties\")(1)"
"open o" 1.0 0 -10899396 true "" "plot get-vowel-means-second(\"lefties\")(6)"

MONITOR
568
690
735
735
Righty vowel mean
get-vowel-means(righties)(VowelFollowed)(FormantFollowed)
3
1
11

PLOT
779
531
1207
702
Righty Second Formants
NIL
NIL
0.0
500.0
215.0
2500.0
true
true
"" ""
PENS
"a" 1.0 0 -13840069 true "" "plot get-vowel-means-second(\"righties\")(4)"
"u" 1.0 0 -16777216 true "" "plot get-vowel-means-second(\"righties\")(8)"
"epsilon" 1.0 0 -1184463 true "" "plot get-vowel-means-second(\"righties\")(3)"
"o" 1.0 0 -6459832 true "" "plot get-vowel-means-second(\"righties\")(7)"
"schwa" 1.0 0 -955883 true "" "plot get-vowel-means-second(\"righties\")(5)"
"e" 1.0 0 -2674135 true "" "plot get-vowel-means-second(\"righties\")(2)"
"i" 1.0 0 -7500403 true "" "plot get-vowel-means-second(\"righties\")(1)"
"open o" 1.0 0 -10899396 true "" "plot get-vowel-means-second(\"righties\")(6)"

SLIDER
426
590
598
623
count-exemplars
count-exemplars
30
100
100
1
1
NIL
HORIZONTAL

CHOOSER
31
10
169
55
Scenarios
Scenarios
"Base Model" "Zero Model"
0

SLIDER
14
297
208
330
numberofrightyleaders
numberofrightyleaders
0
1000
0
1
1
NIL
HORIZONTAL

SLIDER
210
623
382
656
righty-radius-mean
righty-radius-mean
2
20
5
1
1
NIL
HORIZONTAL

SLIDER
209
656
415
689
lefty-leader-connectivity
lefty-leader-connectivity
1
10
5
1
1
NIL
HORIZONTAL

SLIDER
416
656
617
689
righty-leader-connectivity
righty-leader-connectivity
1
10
5
1
1
NIL
HORIZONTAL

MONITOR
10
731
180
776
Lefty leaders exemplar mean
precision leader-exemplar-mean(lefties)(VowelFollowed)(FormantFollowed) 3
17
1
11

MONITOR
566
736
734
781
Righty leader exemplar mean
precision leader-exemplar-mean(righties)(VowelFollowed)(FormantFollowed) 3
17
1
11

CHOOSER
17
558
155
603
VowelFollowed
VowelFollowed
1 2 3 4 5 6 7 8
3

CHOOSER
16
605
154
650
FormantFollowed
FormantFollowed
1 2
0

MONITOR
194
692
321
737
Lefty vowel variance
get-vowel-variance(lefties)(VowelFollowed)(FormantFollowed)
3
1
11

MONITOR
366
693
512
738
Righty vowel variance
get-vowel-variance(righties)(VowelFollowed)(FormantFollowed)
3
1
11

MONITOR
185
738
365
783
Lefty leaders exemplar variance
precision leader-exemplar-variance(lefties)(VowelFollowed)(FormantFollowed) 3
3
1
11

MONITOR
366
737
565
782
Righty leaders exemplary variance
leader-exemplar-variance(righties)(VowelFollowed)(FormantFollowed)
3
1
11

MONITOR
735
704
860
749
Turtle vowel mean
get-vowel-means(turtles)(VowelFollowed)(FormantFollowed)
3
1
11

MONITOR
862
704
1005
749
Turtle vowel variance
get-vowel-variance(turtles)(VowelFollowed)(FormantFollowed)
3
1
11

SLIDER
426
623
702
656
individual-percent
individual-percent
0
0.15
0.05
0.01
1
x 100 for percent
HORIZONTAL

SLIDER
617
657
777
690
bigotry
bigotry
0
1.0
0
0.01
1
NIL
HORIZONTAL

INPUTBOX
93
95
194
155
reps
40000
1
0
Number

MONITOR
14
212
201
257
NIL
dateandtime
17
1
11

BUTTON
40
58
157
91
NIL
set-scenarios
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1005
704
1094
749
NIL
count lefties
2
1
11

MONITOR
1093
704
1193
749
NIL
count righties
2
1
11

@#$#@#$#@



## DEVELOPMENT NOTES

### Created on 2015-08-13 by copying Exemplars and Simple Segregation Multiple Vowels 2.02.



## WHAT IS IT?

Two kinds of agents, reds and greens, with different langauge norms. They are segregated absolutely along a vertical axis. They talk with random neighbors and make speech adjustments. What happens? Blending under some circumstances. We hope, sustained diversity under some other circumstances.

**
Key issue: how can you maintain linguistic diversity, here with an exemplar model, which would seem to afford only blending.

1. Pure exemplar theory, no constraints blending with waves
2. Can exemplar theory explain how variation is preserved? This is the leaders-dont-change (or listen) option. This can stop blending, depending on parameter values.
3. Leaders can change, they mainly talk to each other and listen to each other. Here we get slow blending. Key variables: number of leaders, how often talk, how much weight.

**

Will use 2 formant vowel system of Nettle, page 41.

System works like the single vowel in many ways, but:

1. agent picks an interlocutor at random in neighborhood
2. interlocutor randomly picks a vowel to utter
3. interlocutor randomly generates first and second formant pair for vowel
4. agent gets the pair and does a similarly (distance) calculation for its formant means of various vowels, then adds the new pair to the best matching exemplar list, and updates list and statistics for list
 end of episode





## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="BaseModel" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = reps</exitCondition>
    <metric>get-vowel-means-first("lefties")(4)</metric>
    <metric>get-vowel-means-first("lefties")(8)</metric>
    <metric>get-vowel-means-first("lefties")(3)</metric>
    <metric>get-vowel-means-first("lefties")(7)</metric>
    <metric>get-vowel-means-first("lefties")(5)</metric>
    <metric>get-vowel-means-first("lefties")(2)</metric>
    <metric>get-vowel-means-first("lefties")(1)</metric>
    <metric>get-vowel-means-first("lefties")(6)</metric>
    <metric>get-vowel-means-first("righties")(4)</metric>
    <metric>get-vowel-means-first("righties")(8)</metric>
    <metric>get-vowel-means-first("righties")(3)</metric>
    <metric>get-vowel-means-first("righties")(7)</metric>
    <metric>get-vowel-means-first("righties")(5)</metric>
    <metric>get-vowel-means-first("righties")(2)</metric>
    <metric>get-vowel-means-first("righties")(1)</metric>
    <metric>get-vowel-means-first("righties")(6)</metric>
    <metric>get-vowel-means-second("lefties")(4)</metric>
    <metric>get-vowel-means-second("lefties")(8)</metric>
    <metric>get-vowel-means-second("lefties")(3)</metric>
    <metric>get-vowel-means-second("lefties")(7)</metric>
    <metric>get-vowel-means-second("lefties")(5)</metric>
    <metric>get-vowel-means-second("lefties")(2)</metric>
    <metric>get-vowel-means-second("lefties")(1)</metric>
    <metric>get-vowel-means-second("lefties")(6)</metric>
    <metric>get-vowel-means-second("righties")(4)</metric>
    <metric>get-vowel-means-second("righties")(8)</metric>
    <metric>get-vowel-means-second("righties")(3)</metric>
    <metric>get-vowel-means-second("righties")(7)</metric>
    <metric>get-vowel-means-second("righties")(5)</metric>
    <metric>get-vowel-means-second("righties")(2)</metric>
    <metric>get-vowel-means-second("righties")(1)</metric>
    <metric>get-vowel-means-second("righties")(6)</metric>
    <enumeratedValueSet variable="individual-percent">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bigotry">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lefty-leader-connectivity">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="numberofleftyleaders">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Scenarios">
      <value value="&quot;Base Model&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="lefty-radius-mean">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pleaderleader">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="reps">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="leaderschange">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="leader-effect">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="dasd">
      <value value="4.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FormantFollowed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="leader-leader-effect">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="VowelFollowed">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="righty-radius-mean">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="count-exemplars">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="numberofrightyleaders">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="righty-leader-connectivity">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="percentshift">
      <value value="0.1"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
