extensions [table nw]

globals [social-influence material-influence team-radius]

breed [people person]
breed [expectations expectation]
breed [technologies technology]


people-own [usage understanding influencer persistence teams node-clustering-coefficient eigen-cent]
expectations-own [feature strength]
technologies-own [affordances]


directed-link-breed [holds hold]

undirected-link-breed [connections connection]




to setup
  ca  
  
  ; If auto-total-people is on, make sure the number of people expecting a, b and both doesn't exceed
  ;; the total number set. 
  ;; Populate the rest of the world (up to the total number set) with people with no expectations.

  ifelse auto-total-people != "off" and #-expecting-a + #-expecting-b + #-expecting-both > auto-total-people
  [ user-message (word "There are too many people. Turn auto-total off, or reduce the number of people expecting a, b or both.")]
  [ if auto-total-people != "off"
    [ set #-no-expectations auto-total-people - (#-expecting-a + #-expecting-b + #-expecting-both)]
  

  ;; create the total number of people who will be in the model
  ;; distribute them randomly around the world
  ;; format their size and shape
  ;; set their usage blank
  ;; format their color to reflect their usage
  ;; give them no team affiliations
  ;; give each of them a blank table that will hold their expectations about the technology
  
  create-people total-# 
  [ setxy random-pxcor random-pycor
    set shape "person"
    set size 1.6
    set usage []
    color-code usage .8 
    set teams []
    set understanding table:make ]  
  
  
  ;; for the number of people expecting only a, pick people with blank tables, and set their understanding of feature a to 1 
  ask n-of #-expecting-a people with [ table:length understanding < 1 ]
  [ table:put understanding "a" 1 ]
  
  ;; for the number of people expecting only b, pick people with blank tables, and set their understanding of feature b to 1 
  ask n-of #-expecting-b people with [ table:length understanding < 1 ]
  [ table:put understanding "b" 1 ]
  
  ;; for the number of people expecting a and b, pick people with blank tables, and set their understanding of 
  ;; features a and b to 1   
  ask n-of #-expecting-both people with [ table:length understanding < 1 ]
  [ table:put understanding "a" 1
    table:put understanding "b" 1 ] 
  
  
  ;; put people into a social network
  build-network
  calculate-centralities
  do-layout


  if add-#-eigenvector != "off"
  [ add-expectations 
    adjust-counts  ]


  ask people
  
  ;;  build a network of expectations to provide a visual representation of each person's table of understanding
  [ update-expectations 
    
    ;; if limited persistence is on, give people a persistence of 3
    ;; if it's off, give people a persistence that is extremely high   
    ifelse limited-persistence
    [ set persistence 3 ]
    [ set persistence 1000 ] ]

  ;; format the network of expectations
  ask expectations
  [ format-expectations ]
  


  ;; set up the technology 
  adopt-technology  

  set social-influence []
  set material-influence []


  reset-ticks

]

end



to build-network
  ;; create a social network meant to mimic a workplace:
  ;; everyone is a member of one or more small teams (up to 4 teams comprised of 2-5 people each (??))
  ;; some people are also connected to others across the organization
  
  
  ;; set the network to be people and their connections
  nw:set-context people connections
  
  ;; decide how far away people should look for for team-members
  set-team-radius
  
  ;; ask people one at a time to each lead a small team (by asking a few neighbors to join)
  ;; until everyone is on at least one team
  while [any? people with [ teams = [] ] ]
  [ ask one-of people with [ teams = [] ]
    [ set teams (list who)
      ;show teams
      ask n-of (1 + random 4) other people in-radius team-radius with [length teams < 4]
      ;; [what happens if there isn't at least one other person already on less than 4 teams in your radius??]
      
      [set teams fput [who] of myself teams
        set teams (sentence teams)
        ;show teams 
      ] ] ]
  
  ;; create links between all the members of each team
  ;; by going through each potential team name (i.e. the who number of the leader) 
  ;; and asking everyone who has that name as one of their list of teams to link up
  let roll-call 0
  while [roll-call <= total-#]
  [ ;show roll-call
    ask people
    [ if member? roll-call teams
      [ if any? other people with [ member? roll-call teams and connection-neighbor? myself = false ]
        [create-connections-with other people with [member? roll-call teams and connection-neighbor? myself = false ] ] ] ]
    set roll-call roll-call + 1 ]
  
  ;; make sure everyone in the network is connected to one giant component
  fix-network
  
  ;; give the network small-world properties
  make-small
  
end


to set-team-radius
  
  ;; start with people looking only within 1 patch of themselves
  set team-radius 1
  
  ;; increment the distance up until everyone has at least 4 potential team members
  while [ any? people with [count other people in-radius team-radius < 4] ]
  [ set team-radius team-radius + 1]
  
end


to fix-network
  
  ;; if there is more than one component, ask one person at a time to make a connection 
  ;; with someone in another component until there is only one giant component
  
  ;show nw:weak-component-clusters
  if length nw:weak-component-clusters > 1
  [ while [ length nw:weak-component-clusters > 1 ]
    [ ask one-of people 
      [create-connection-with one-of people with [nw:distance-to myself = false] ] ] ]
  ;show nw:weak-component-clusters 
  
end



to make-small
  
  ;; ask apprx 10% of peopel to create extra cross-world connections
  ask n-of (ceiling (total-# / 10) ) people
  [ create-connection-with one-of other people ]
    ;show "new link"

end

to calculate-centralities
  ask people
  [ set eigen-cent nw:eigenvector-centrality ]
  
end

to do-layout
  
  layout-tutte (people with [count connection-neighbors = 1]) connections 14 

end


to update-expectations
  ;; ask people whose table of understandings doesn't match the visual representation of their expectations
  ask people with [table:length understanding != count out-hold-neighbors]
  
  ;; create a blank slate 
  [ ask out-hold-neighbors 
    [ die ]
  
  ;; build the network of expectations up from the understanding table  
  let instructions table:keys understanding
  while [instructions != []]
  [ hatch-expectations 1
    [ create-hold-from myself
      ask my-in-holds [hide-link] 
      set feature first instructions
      set strength [table:get understanding first instructions] of myself 
      set instructions but-first instructions  ] ] ]
  
  ask expectations
  [ format-expectations ] 
  
end



to format-expectations
  
  ;; make expectations color, shape and placement reflect what they indicate
  
if strength > 1
  [ set strength 1 ]
  if strength < -1
  [ set strength -1 ]  
  

  if strength > 0
  [ show-turtle 
    set shape "circle"
    set size .5 ]

  if strength < 0
  [ show-turtle
    set shape "x"
    set size .6 ]

  if strength = 0
  [hide-turtle]
  
  
  if feature = "a"
  [ setxy ( [ xcor ] of one-of in-hold-neighbors - .5) ( [ ycor ] of one-of in-hold-neighbors + .5)  ] 
  if feature = "b"
  [ setxy ( [ xcor ] of one-of in-hold-neighbors + .5) ( [ ycor ] of one-of in-hold-neighbors + .5) ]
  
  color-code feature 1.5
  
end




to adopt-technology
 
create-technologies 1
  [ set shape "box"
    set size 3
    
    ;; create a blank table that will hold affordances
    set affordances table:make
    
    ;; fill in the table from the features selected by the user
    let instructions (sentence technology-affordances)
    while [instructions != [] ]
  [ table:put affordances first instructions 1
    set instructions but-first instructions ] 
  
  color-code (table:keys affordances) -2.5]
    

end



to color-code [thing number]
  ;; color code expectations, usage and affordances to  provide a visual indication of whether they match.  
  
  if member? "a" thing and not member? "b" thing
    [ set color blue + number ] 
  if member? "b" thing and not member? "a" thing
    [ set color yellow + number ] 
  if member? "a" thing and member? "b" thing
    [ set color green + number] 
  if not member? "a" thing and not member? "b" thing
    [ set color gray + number ]
  
end


to add-expectations
  
  ;; translate the string into instructions
  let letter item 0 add-#-eigenvector
  let number read-from-string item 2 add-#-eigenvector
  let value item 4 add-#-eigenvector
  
  if value = "h"
  [ set value total-# - number ]
  if value = "l" 
  [ set value 0 ]
  if value = "a"
  [ set value floor (total-# - number) / 2 ]
  
  let all-eigen-cent sort-on [nw:eigenvector-centrality] people
  
  ;; identify the people with the lowest, median, or highest eigenvector centrality
  ;; clear their table of understandings and replace with either "a" or "b"
  repeat number 
  [ ask item value all-eigen-cent
    [ table:clear understanding
      table:put understanding letter 1 ] 
    set value value + 1 ]

  
end


to adjust-counts
  
  ;; make sliders reflect actual counts
  set #-expecting-a count people with [ member? "a" table:keys understanding and not member? "b" table:keys understanding]
  set #-expecting-b count people with [ not member? "a" table:keys understanding and member? "b" table:keys understanding]
  set #-expecting-both count people with [ member? "a" table:keys understanding and member? "b" table:keys understanding]
  set #-no-expectations count people with [ not member? "a" table:keys understanding and not member? "b" table:keys understanding]
      
end


to go
  
  ;; stop conditions
  if all? people [persistence = 0] 
    [ stop ]
  if ticks > 99 and (all? people [usage = []] 
    or all? people [usage = ["a"]] 
    or all? people [usage = ["b"]] 
    or all? people [usage = ["a" "b"] or usage = ["b" "a"]]) 
    [ stop ]
  
  ;; running the model
  ask people
  [ interact
    update-expectations ] 
  
  update-influences
  tick  
  
end  
  
  
 
  
to interact
  
  set influencer nobody
  
  if persistence > 0
  
  [ determine-influencer ]
    
    ;; if I am being influenced by a person (social interaction), then learn from their expectations
    if is-person? influencer 
    [ learn-from ([understanding] of influencer) 1 .2]
    
    ;; if I am being influenced by the technology (material interaction), then if I have some expectation of
    ;; what the technology is for I will try to use it accordingly.
    ;; otherwise I will learn from the technology
    if is-technology? influencer 
    [ ifelse table:length understanding > 0
      [ use-technology ] 
      [ learn-from ([affordances] of influencer) tech-transparency 1 ]  
    ;; increment my persistence down 
    set persistence persistence - 1 ] 
  
end 
   

to determine-influencer
  
  
  ;; figure out whether influencer will be the technology or a person
  if random-float 1 < .25
  [ ifelse random-float 1 < proportion-material-influence
    [ set influencer one-of technologies ]
    [ set influencer one-of connection-neighbors ] ] 

  ;; (THINK ABOUT CHANGING THIS TO A SYSTEM BASED ON LINK WEIGHTS
  ;; THEN THE WEIGHT FOR THE TECHNOLOGY CAN INCREASE BASED ON EXPECTATIONS)

end

 
to learn-from [source chance influence]
  
  ;; set probability of learning
  ;; if the influencer has any features to learn from,
  ;; pick one to be the insight I learn
  if table:length source > 0 and random-float 1 < chance
  [ let insight one-of table:keys source
    
    
    ;; if that feature is new to me, put the insight into my table of understandings
    
    if not table:has-key? understanding insight
    [ table:put understanding insight 0 ]
    
    ;; determine whether this is going to be learning something positive or negative
    let direction 1
    if table:get source insight > 0
      [ set direction 1 ]
    if table:get source insight < 0
      [ set direction -1 ]
    
    ;; change expectation to be more positive/more negative 
    ;; (amount of change is .2 for people influencers, 1 for technology influencer)
    table:put understanding insight table:get understanding insight + influence * direction 
    
    ;; cap expectation strength at >= -1 and <= 1 
    if abs table:get understanding insight > 1
      [table:put understanding insight direction] 
    
    ;; adjust visual version of expectations to match understanding
    ask out-hold-neighbors with [feature = insight ]
    [ set strength [table:get understanding insight] of myself ] ]
  
  
end 
 
 

to use-technology
  
  ;; if can-learn-unexpectly is on, give a 5% chance of trying to learn from the technology before trying to use it 
  if can-learn-unexpectedly
  [ if random-float 1 < .05
    [ learn-from ([affordances] of influencer) tech-transparency 1 ] ] 
  
  ;; if I have any positive expectations, pick one of those to be the way I try to use the technology 
  if any? out-hold-neighbors with [strength > 0] 
  [ let use [feature] of one-of out-hold-neighbors with [ strength > 0 ]
    
    ;; if use-can-fail is on, set the chance that use will be successful down to 95%
    let chance-use-works 1
    if use-can-fail 
    [ set chance-use-works .95 ]
 
    ;; if the feature I'm trying to use is one of the technology's affordances, then put that feature into my usage (if use is successful)
    ;; and set my expectation for that feature fully to 1
    ;; if not, set my expectation for that feature fully negative and remove it from my usage 
    ifelse table:has-key? [affordances] of influencer use and random-float 1 < chance-use-works
    [ table:put understanding use 1 
      set usage lput use usage
      set usage remove-duplicates usage 
      color-code usage .8 ]
    [ table:put understanding use -1
      set usage remove use usage
      color-code usage .8 ]
    
    ;;ask my visual expectation network to adjust accordingly
    ask out-hold-neighbors with [feature = use ]
    [ set strength [table:get understanding use] of myself ]  ]
  
end 



to update-influences
  
  ;; list the proportion of people experiencing social and material influences at each tick
  if any? people with [persistence > 0]
  [ set social-influence fput (count people with [is-person? influencer] / count people with [persistence > 0]) social-influence
    set material-influence fput (count people with [is-technology? influencer] / count people with [persistence > 0]) material-influence ]
  
end



to-report total-#
  ;; report the total number of people in the world
  report #-expecting-a + #-expecting-b + #-expecting-both + #-no-expectations
end



to-report %-open-to-influence
  ;; report the % of people who are still open to being influenced out of the total 
  report count people with [persistence > 0 ] / total-#
end


to-report %-being-influenced
  ;; report the % of people currently being influenced, out of the total (or out of the number open to being influenced)
  report count people with [influencer != nobody] / total-# ;count people with [persistence > 0]
end


to-report average-social-influence
  ;; report the average % of people influenced by other people out of the number open to being influenced at each tick
  report sum social-influence / (length social-influence)
end

to-report average-material-influence
  ;; report the average % of people influenced by the technology out of the number open to being influenced at each tick
  report sum material-influence / (length material-influence)
end


to-report in-neighborhood? [ hood ]
  report ( member? end1 hood and member? end2 hood )
end

to-report mean-clustering-coefficient
 
    let clustering-coefficient 0

    let total 0
    ask people with [ count connection-neighbors <= 1]
      [ set node-clustering-coefficient "undefined" ]
    ask people with [ count connection-neighbors > 1]
    [ let hood connection-neighbors
      set node-clustering-coefficient (2 * count connections with [ in-neighborhood? hood ] /
                                         ((count hood) * (count hood - 1)) )
      ;; find the sum for the value at turtles
      set total total + node-clustering-coefficient
    ]
    ;; take the average
    set clustering-coefficient total / count people with [count connection-neighbors > 1]
    report clustering-coefficient
end


to-report path-length
  
nw:set-context people connections
report nw:mean-path-length  
  
end

@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
68
26
134
59
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
19
196
191
229
#-expecting-a
#-expecting-a
0
100
1
1
1
NIL
HORIZONTAL

SLIDER
19
242
191
275
#-expecting-b
#-expecting-b
0
100
55
1
1
NIL
HORIZONTAL

SLIDER
19
287
191
320
#-expecting-both
#-expecting-both
0
100
0
1
1
NIL
HORIZONTAL

SLIDER
19
332
199
365
#-no-expectations
#-no-expectations
0
100
44
1
1
NIL
HORIZONTAL

CHOOSER
19
120
193
165
technology-affordances
technology-affordances
["a"] ["b"] ["a" "b"] []
0

BUTTON
26
73
107
106
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
122
72
185
105
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
893
342
1093
492
usages
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"using for a" 1.0 0 -13345367 true "" "plot count people with [ usage = [\"a\"] ]"
"using for b" 1.0 0 -4079321 true "" "plot count people with [ usage = [\"b\"] ]"
"not using" 1.0 0 -7500403 true "" "plot count people with [ usage = [] ]"

PLOT
670
340
870
490
expectations
NIL
NIL
0.0
10.0
0.0
110.0
true
false
"" ""
PENS
"a" 1.0 0 -13345367 true "" "plot count expectations with [feature = \"a\" and strength > 0]"
"not a" 1.0 0 -11053225 true "" "plot count expectations with [feature = \"a\" and strength < 0]"
"b" 1.0 0 -4079321 true "" "plot count expectations with [feature = \"b\" and strength > 0]"
"not b" 1.0 0 -4539718 true "" "plot count expectations with [feature = \"b\" and strength < 0]"

SWITCH
680
17
857
50
limited-persistence
limited-persistence
0
1
-1000

MONITOR
74
380
131
425
NIL
total-#
17
1
11

SLIDER
680
71
990
104
tech-transparency
tech-transparency
0
1
0.1
.01
1
NIL
HORIZONTAL

SWITCH
681
118
887
151
can-learn-unexpectedly
can-learn-unexpectedly
0
1
-1000

CHOOSER
33
449
171
494
auto-total-people
auto-total-people
"off" 100
1

SWITCH
681
169
814
202
use-can-fail
use-can-fail
0
1
-1000

SLIDER
681
219
922
252
proportion-material-influence
proportion-material-influence
.01
.99
0.05
.01
1
NIL
HORIZONTAL

CHOOSER
680
269
822
314
add-#-eigenvector
add-#-eigenvector
"off" "a 1 high" "a 1 average" "a 1 low" "a 5 high" "a 5 average" "a 5 low" "b 1 high" "b 1 average" "b 1 low"
0

@#$#@#$#@
## WHAT IS IT?

This model is based on Leonardi, P. M. (2012). Car Crashes Without Cars: Lessons About Simulation Technology and Organizational Change from Automotive Design. MIT Press (MA).

It is intended to represent what happens when a new technology is introduced into an organization and people are trying to figure out what it is good for. It models a cyclical set of dynamics between people's interactions with one another ("social interactions"), and interactions with the technology itself ("material interactions") working together to determine whether the tool is actually taken up and used. 

The cycle works as follows: Social interactions (i.e. people talking to one another about the new tool) shape people’s expectations for the technology. These expectations turn shape their material interactions with the tool, as people tend to find only the affordances they know to look for. Information from these material interactions (i.e. whether the technology worked as a person expected or not) is then incorporated into future social interactions.

In Leonardi's ethnographic work, he found that these dynamics led to two different patterns:

1. People begin with a widespread expectation that the technology affords something it does not. In the end, most people abandon the technology.
2. People begin without clear expectations of what the technology affords. In the end most people use the technology.

This model is the third in a series designed to model these dynamics. Changes from Model 2 are in **bold**.

## HOW IT WORKS

The agents in this model are people and technologies. People have three important attributes: 

1. A set of expectations about what the technology affords, including both a set of features (“a”, “b”, both, or neither) and a valence for each feature present (1 or -1). A valence of 1 represents a belief that the technology does afford that feature. A valence of -1 represents a belief that the technology does not afford that feature. People can also lack any expectation for a particular feature. 
2. A usage, which refers to the features of the technology that they use. Usage can also take values of “a,” “b,” both or neither, however it does not hold valence. 
3. A level of persistence. This refers to how many times people will interact with the technology before their expectations and usage become permanently fixed. 

The technology (there is never more than one technology in a given simulation) holds one attribute: its affordances. The technology’s set of affordances can include “a”, “b”, both, or neither. If a particular feature is present its valence is always positive. 

**In this model, people are linked together in a social network. The network is designed to mimic the organization of a workplace. It is created in three steps: first, each person is assigned to be a member of 1-3 different work teams, each of which have between 2-5 members. Second, 10% of people create random additional links to others in the network. Third, any disconnected components are connected to the larger network. The result is intended to roughly represent the aggregation of formal and informal relationships in a moderate-to-large organization, such as the one Leonardi studied. Because people are primarily connected to a relatively small number of others nearby to them but also have some cross-network ties, the network has small world properties.**


When the model begins, each person follows the following rules at each tick:

1. Turn a little and take a step forward 
2. If my persistence has not run out, **roll a 100-sided die; if it comes up 25 or under, interact with another agent. Roll another 100-sided die; if it comes up less that "proportion material influence" interact with the technology. Otherwise interact with one of my network neighbors**
    1. If I am interacting with another person adjust one of my expectations to be more like theirs
    2. If I am interacting with the technology: 
        1. If can-learn-unexpectedly is turned on, roll a die to see if I will learn one of the technology's affordances. Either way, next:
        2. Try to use the technology for one thing I expect it will be able to do
            1. If that feature is among the technology’s affordances, add it to my usage (if use-can-fail is on, roll a die first to see if I'll be successful). If not, change my expectations to not believing the technology affords that feature
        3. If I have no expectations, learn one feature from the technology itself (at a chance based on the technology's transparency)
        4. Increment my persistence down
        5. (If I have only negative expectations do nothing)


## HOW TO USE IT

Select a set of affordances for the technology. 

Select the number of people who will start with expectations of “a,” “b,” both, or neither. All starting expectations are positive. 

If auto-total people is set to a number, the model will automatically add people with no expectations to the model until the number of people equals auto-total number. If the user has already selected more than that number of people with starting expectations of "a", "b" or both, an error message will pop up.

Optionally “limited-persistence” can be turned on, which sets starting persistence at 3 for all people. When “limited persistence” is off, people start with a persistence of 1000—-an arbitrarily high number.

Select a level of transparency for the technology. This determines the chance a person will be able to learn one of its affordances when they interact with it with no expectations.

Optionally "can-learn-unexpectedly" can be turned on, which means that people have a 5% chance of trying to learn from the technology before trying to use it.

Optionally "use-can-fail" can be turned on, which means that people have only a 95% chance of being able to find a technology's affordance when if they interact with it even if their expectations are correct.

**Select the "proportion material influence" vs. social influence. In Model 2, with 100 people in the model, this comes out to about .05. That is, 5% of people's interactions are with the technology, and 95% are with other people.**

**Optionally change some people's starting expectations using add-#-eigenvector. This allows you to choose people who start expecting "a" or "b" based on their eigenvector-centrality.** 


## THINGS TO NOTICE

People's expectations are represented by little symbols next to their head. Blue represents "a", and yellow represents "b". Dots indicate a positive expectation. Xs indicate a negative expectation.

People's usage is represented by the color of their body. Gray represents no usage, blue represents "a," yellow represents "b" and green represents both "a" and "b."

## THINGS TO TRY

Try to match the reference patterns Leonardi found in his ethnographic work!

## EXTENDING THE MODEL

**What if people were in a different kind of social network? E.g. Preferential attachment?**

**What if people's chance of interacting with the technology were dependent on their expectations for what it does?**

## NETLOGO FEATURES

Note that both expectations and affordances are coded here using the Tables extension. **This model also uses the nw extension.**

## RELATED MODELS

Leonardi Model 1: http://modelingcommons.org/browse/one_model/4072

Leonardi Model 2: http://modelingcommons.org/browse/one_model/4074

## CREDITS AND REFERENCES

http://modelingcommons.org/browse/one_model/4075

Leonardi, P. M. (2012). Car Crashes Without Cars: Lessons About Simulation Technology and Organizational Change from Automotive Design. MIT Press (MA).
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           