;; Simplified Veteran Military-to-Civilian Transition + 3T Extremism Model
;; NetLogo 6.x – calibrated to Ukrainian veterans, May 2025
;; © 2025, open academic licence
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals [
  ;; transition thresholds
  tick-per-year          ;; number of ticks that represent one year
  partial-threshold      ;; minimum civilian identity for partial integration
  complete-threshold     ;; minimum civilian identity for complete integration
  centrality-cutoff      ;; maximum veteran identity centrality for complete integration

  ;; 3T extremism
  extremist-threshold    ;; threshold above which a veteran is considered at extremism risk

  ;; policy levers (hook to sliders)
;  pol-health-funding     ;; additional healthcare capacity
 ; pol-job-program        ;; probability of job placement
 ; pol-edu-support        ;; additional education/training capacity
 ; pol-housing-subsidy    ;; additional housing units
 ; pol-finance-benefit    ;; monthly financial benefit amount
 ; pol-spiritual-support  ;; spiritual support intensity
 ; pol-legal-aid          ;; probability of legal aid
 ; pol-anti-stigma        ;; reduction in public stigma
 ; pol-identity-coach     ;; identity coaching intensity
 ; pol-anti-prejudice     ;; reduction in prejudice transmission
 ; policy-locality        ;; How locally focused policies are (0-1, slider)

  ;; DDR Context
  ;conflict-intensity     ;; current intensity of conflict (0-1)
  remobilization-rate    ;; base rate of remobilization

  ;; Veteran Peoplehood variables
  veteran-group-count    ;; number of veteran peer groups formed

  ;; Societal dynamics
  ;media-event-chance    ;; probability of media event affecting veteran perception
  last-media-event-tick  ;; when the last media event occurred
  media-event-effect     ;; current effect of media events on public perception

  ;; Tracking variables for absolute counts
  integrated-count       ;; count of fully integrated veterans
  partial-count          ;; count of partially integrated veterans
  returned-count         ;; count of veterans returned to service
  extremist-count        ;; count of extremist veterans
  crisis-count           ;; count of veterans in identity crisis
  employed-count         ;; count of employed veterans
  unemployed-count       ;; count of unemployed veterans

  ;; Batch processing variables
  batch-size             ;; how many veterans to process per tick
  current-batch-start    ;; start index of current processing batch
  veteran-scale-factor   ;; scaling factor for visual representation

  ;; View mode
  ;view-mode              ;; Current view mode for patches (stigma, jobs, healthcare, etc.)

  ;; Visualization control
  show-links?            ;; Whether to show links
  ;show-labels?           ;; Whether to show labels
]

;; Define agent types
breed [veterans veteran]
breed [veteran-groups veteran-group]
breed [providers provider]
breed [employers employer]
breed [faith-centres faith-centre]
breed [community-groups community-group]
breed [hospitals hospital]
breed [universities university]

;; Veteran properties
veterans-own [
  ;; CORE: Military Transition Theory (MTT) - stage & trajectory
  stage              ;; current transition stage (approaching, managing, assessing)
  traj               ;; trajectory status (partial, complete, returned)

  ;; CORE: Dual-Salience Veteran Identity (Atuel & Castro 2018; Grimell 2024)
  identity-mil       ;; military identity salience (0-1)
  identity-civ       ;; civilian identity salience (0-1)
  identity-central   ;; centrality of veteran identity (0-1)
  moral-orient       ;; moral orientation (military vs civilian values)

  ;; CORE: 3T model of extremism risk factors
  prejudice-score    ;; Transmission-of-Prejudice score (0-1)
  trauma-score       ;; Trauma & Adversity score (0-1)
  transition-stress  ;; Transition Stress score (0-1)
  extremism-risk     ;; aggregate risk score (0-1)
  extremist?         ;; boolean flag for extremist status

  ;; UNIFIED: Well-being system (DiRamio's nine dimensions, 0-100)
  wb-physical        ;; physical well-being
  wb-mental          ;; mental well-being
  wb-social          ;; social well-being
  wb-family          ;; family well-being
  wb-work            ;; work well-being
  wb-housing         ;; housing well-being
  wb-finance         ;; financial well-being
  wb-spiritual       ;; spiritual well-being
  wb-legal           ;; legal well-being
  integration        ;; overall integration score (average of all dimensions)

  ;; Resource states (directly affect well-being)
  skills-civ         ;; civilian skills level (integer)
  education-status   ;; education status (none, enrolled, graduated)
  employment-status  ;; employment status (unemployed, employed)
  income             ;; monthly income (currency units)
  debt               ;; outstanding debt (currency units)
  housing-status     ;; housing status (at-risk, secure)
  legal-status       ;; legal issues status (none, pending, resolved)

  ;; Psychological resources and engagement
  resilience         ;; personal resilience level (0-1)
  coping             ;; coping ability level (0-1)
  spiritual-engagement ;; level of spiritual engagement (0-1)
  community-engagement ;; level of community engagement (0-1)
  looking-back       ;; tendency to idealize military service (0-1)

  ;; Transition dynamics
  transition-milestone-count ;; number of significant transition events experienced
  identity-crisis?          ;; boolean for whether experiencing identity crisis
  identity-crisis-duration  ;; how many ticks in current identity crisis

  ;; DDR Context Variables
  mobilization-risk        ;; risk of being remobilized (0-1)
  deployment-count         ;; number of times deployed/mobilized
  combat-exposure          ;; level of combat experience (0-1)

  my-group           ;; The veteran group this veteran belongs to (if any)
]

;; Social network properties
community-groups-own [
  community-resources   ;; resource level (0-9)
  community-acceptance  ;; acceptance of veterans (0-1)
  community-cohesion    ;; how connected community members are (0-1)
  community-influence   ;; radius of influence
  community-id          ;; unique identifier for this community
]

;; Patch (environment) properties
patches-own [
  jobs-available     ;; number of job openings
  clinic-capacity    ;; healthcare capacity
  training-seats     ;; education/training capacity
  affordable-units   ;; housing units available
  public-stigma      ;; level of public stigma (0-1)
]

veteran-groups-own [
  group-size           ;; number of veterans in the group
  group-cohesion       ;; internal cohesion of group (0-1)
  group-identity       ;; strength of collective identity (0-1)
  group-influence      ;; radius of influence
  group-militarization ;; how military-focused the group is (0-1)
  group-resources      ;; resources available to members (0-1)
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; SETUP PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Initialize the model
to setup
  clear-all
  reset-ticks

  ;; Set batch processing parameters
  set batch-size 10000  ;; Process 10,000 veterans per tick for performance
  set current-batch-start 0

  ;; Set scaling factor for visual representation
  set veteran-scale-factor 1000  ;; Each visual agent represents 1000 veterans

  ;; Initialize tracking counts
  set integrated-count 0
  set partial-count 0
  set returned-count 0
  set extremist-count 0
  set crisis-count 0
  set employed-count 0
  set unemployed-count 0

  ;; Set up globals
  setup-globals

  ;; Determine the right number of visual elements based on scale
  let visual-veterans 1000000 / veteran-scale-factor  ;; Creating 1000 visual agents for 1M veterans
  let visual-communities 50  ;; About 1 community per 10,000 veterans

  ;; Set up communities first
  setup-communities visual-communities

  ;; Set up veterans (visual representation)
  setup-veterans visual-veterans

  ;; Scale other resources based on veteran population
  let visual-providers round (1000000 / 500 / visual-communities / 2)
  let visual-employers round (1000000 / 300 / visual-communities / 2)
  let visual-faith-centres round (1000000 / 800 / visual-communities / 2)

  ;; Set up service providers with scaled numbers
  setup-service-providers-scaled visual-providers visual-employers visual-faith-centres

  let visual-hospitals round (max (list 15 (1000000 / 10000 / veteran-scale-factor)))
  let visual-universities round (max (list 12 (1000000 / 12000 / veteran-scale-factor)))

  ;; Set up specialized service providers
  setup-specialized-providers visual-hospitals visual-universities

  ;; Calculate initial statistics
  update-statistics

  ;; Visual setup
  setup-shapes
  update-visualization
end

;; Set global variables and thresholds
to setup-globals
  ;; Time settings
  set tick-per-year 12       ;; Each tick represents one month
  set view-mode "composite"  ;; Default view showing all resources

  ;; Integration thresholds
  set partial-threshold 0.40 ;; Minimum civilian identity for partial integration
  set complete-threshold 0.75 ;; Minimum civilian identity for complete integration
  set centrality-cutoff 0.30 ;; Maximum veteran centrality for complete integration

  ;; Extremism threshold
  set extremist-threshold 0.70 ;; Risk threshold for extremist designation

  ;; Initialize policy levels (baseline)
  ;set pol-health-funding 5
  ;set pol-job-program 0.5
  ;set pol-edu-support 5
  ;set pol-housing-subsidy 5
  ;set pol-finance-benefit 2000   ;; baseline benefit UAH/month equiv.
  ;set pol-spiritual-support 0.5
  ;set pol-legal-aid 0.5
  ;set pol-anti-stigma 0.02
  ;set pol-identity-coach 0.2
  ;set pol-anti-prejudice 0.02
  ;set policy-locality 0.5

  ;; Societal dynamics
  ;set media-event-chance 0.05
  set media-event-effect 0

  ;; Conflict context
  ;set conflict-intensity 0.3
  set remobilization-rate 0.005

  ;; Visualization controls
  ;set show-links? false
  ;set show-labels? false
end

;; Set up veteran agents
to setup-veterans [n]
  create-veterans n [
    ;; Position veterans more randomly around the world
    ;; Option 1: Random position with a tendency toward communities
    ifelse random-float 1 < 0.7 [
      ;; 70% of veterans start near communities but with wider spread
      let target-community one-of community-groups
      move-to target-community

      ;; Position with wider spread
      let rand-angle random-float 360
      let rand-dist 3 + random-float 6  ;; Larger distance range
      set heading rand-angle
      forward rand-dist
    ][
      ;; 30% of veterans start completely randomly
      setxy random-xcor random-ycor
    ]

    ;; MTT stage
    set stage "approaching"
    set traj "partial"

    ;; CORE: Identity initialization (Dual-Salience model)
    set identity-mil random-float 1
    set identity-civ random-float 0.2
    set identity-central 0.4 + random-float 0.4
    set moral-orient identity-mil - identity-civ

    ;; CORE: 3T risk factors
    set prejudice-score random-float 0.3
    set trauma-score random-float 0.2
    set transition-stress 0
    set extremism-risk 0
    set extremist? false

    ;; UNIFIED: Well-being initialization (0-100 scales)
    set wb-physical 50 + random 30
    set wb-mental 50 + random 30
    set wb-social 50 + random 30
    set wb-family 50 + random 30
    set wb-work 40 + random 40
    set wb-housing 50 + random 30
    set wb-finance 40 + random 40
    set wb-spiritual 40 + random 40
    set wb-legal 60 + random 30

    ;; Resource states
    set skills-civ random 5
    set education-status "none"
    set employment-status "unemployed"
    set income 0
    set debt 0
    set housing-status one-of ["secure" "at-risk"]
    set legal-status "none"

    ;; Psychological resources
    set spiritual-engagement random-float 1
    set community-engagement random-float 1
    set resilience random-float 1
    set coping random-float 1
    set looking-back random-float 0.3

    ;; Transition dynamics
    set transition-milestone-count 0
    set identity-crisis? false
    set identity-crisis-duration 0

    ;; DDR variables
    set mobilization-risk random-float 0.2
    set deployment-count 1 + random 2
    set combat-exposure random-float 1
    set my-group nobody

    ;; Calculate initial integration score
    calc-integration
  ]
end

;; CORRECTLY FIXED SETUP-COMMUNITIES PROCEDURE
to setup-communities [num-communities]
  ;; Create evenly spaced community centers using a grid approach
  let sqrt-num ceiling sqrt num-communities
  let spacing-x world-width / (sqrt-num + 0.5)  ;; More spacing between communities
  let spacing-y world-height / (sqrt-num + 0.5)  ;; More spacing between communities

  let current-id 0
  let y-pos (min-pycor + spacing-y / 2)  ;; Start closer to the edge

  ;; Create all communities from observer context
  repeat sqrt-num [
    let x-pos (min-pxcor + spacing-x / 2)  ;; Start closer to the edge
    repeat sqrt-num [
      if current-id < num-communities [
        create-community-groups 1 [
          setxy x-pos y-pos

          ;; Add larger random offset for more varied placement
          set xcor xcor + (random-float 4) - 2  ;; Larger spread
          set ycor ycor + (random-float 4) - 2  ;; Larger spread

          set shape "house"
          set size 1
          set color blue

          ;; Each community has characteristics
          set community-resources random 10
          set community-acceptance random-float 1
          set community-cohesion random-float 1
          set community-influence 2 + random 2  ;; Slightly smaller influence

          set community-id current-id
        ]
        set current-id current-id + 1
      ]
      set x-pos x-pos + spacing-x
    ]
    set y-pos y-pos + spacing-y
  ]

  ;; Set patch characteristics with sharper falloff
  ask patches [
    ;; Find closest community center
    let nearest-community min-one-of community-groups [distance myself]

    ;; Only color and set resources if relatively close to a community
    let dist distance nearest-community
    let influence-factor 0

    if dist <= [community-influence] of nearest-community * 1.2 [
      set influence-factor max (list 0 (1 - (dist / [community-influence] of nearest-community))) ^ 2  ;; Squared for sharper falloff

      ;; Set patch stigma based on community acceptance (inverted)
      set public-stigma (1 - [community-acceptance] of nearest-community) * influence-factor

      ;; Set resources based on community resources and distance
      let base-resources [community-resources] of nearest-community / 10

      ;; Resources are concentrated near community centers with sharper falloff
      ifelse influence-factor > 0.7 [  ;; Only very near community centers
        set jobs-available round (2 * base-resources * (1 + random-float 0.5))
        set clinic-capacity round (1 * base-resources * (1 + random-float 0.5))
        set training-seats round (1 * base-resources * (1 + random-float 0.5))
        set affordable-units round (1 * base-resources * (1 + random-float 0.5))
      ] [ ifelse influence-factor > 0.3 [  ;; Medium distance
        set jobs-available round (base-resources * random-float 1)
        set clinic-capacity round (base-resources * random-float 0.5)
        set training-seats round (base-resources * random-float 0.3)
        set affordable-units round (base-resources * random-float 0.5)
      ]  [  ;; Edge of influence
        set jobs-available 0
        set clinic-capacity 0
        set training-seats 0
        set affordable-units 0
      ]
      ]

      ;; Only color patches with significant influence
      if influence-factor > 0.4 [  ;; Higher threshold
        set pcolor scale-color ([color] of nearest-community + 2) (influence-factor / 3) 0 1  ;; Lighter coloring
      ]
    ]
  ]
end

;; Setup service providers with scaling
to setup-service-providers-scaled [num-providers num-employers num-faith-centres]
  ;; Create providers
  create-providers num-providers [
    ;; 70% near communities, 30% random
    ifelse random-float 1 < 0.7 [
      ;; Target a community
      let target-community one-of community-groups
      move-to target-community

      ;; Move to a position near the community
      set heading random 360
      forward 3 + random 3
    ][
      ;; Random position
      setxy random-xcor random-ycor
    ]

    ;set shape "circle"
    set shape "square"
    set color blue - 1
    set size 1.5

    ;; Increase local healthcare capacity
    ask patches in-radius 2 [
      set clinic-capacity clinic-capacity + 1
    ]
  ]

  ;; Create employers
  create-employers num-employers [
    ;; Similar positioning as providers
    ifelse random-float 1 < 0.7 [
      let target-community one-of community-groups
      move-to target-community
      set heading random 360
      forward 3 + random 3
    ][
      setxy random-xcor random-ycor
    ]

    set shape "box"
    set color cyan - 1
    set size 1.5

    ;; Increase local job opportunities
    ask patches in-radius 2 [
      set jobs-available jobs-available + 1 + random 1
    ]
  ]

  ;; Create faith centers
  create-faith-centres num-faith-centres [
    ;; Similar positioning as providers
    ifelse random-float 1 < 0.7 [
      let target-community one-of community-groups
      move-to target-community
      set heading random 360
      forward 3 + random 3
    ][
      setxy random-xcor random-ycor
    ]

    set shape "triangle"
    set color magenta
    set size 1.5
  ]
end

;; Setup specialized healthcare and education providers
to setup-specialized-providers [num-hospitals num-universities]
  ;; Create hospitals
  create-hospitals num-hospitals [
    ;; 80% near communities, 20% random
    ifelse random-float 1 < 0.8 [
      ;; Target a community
      let target-community one-of community-groups
      move-to target-community

      ;; Move to a position near the community
      set heading random 360
      forward 3 + random 2
    ][
      ;; Random position
      setxy random-xcor random-ycor
    ]

    set shape "triangle"
    set color white
    set size 1.5

    ;; Increase local healthcare capacity significantly
    ask patches in-radius 3 [
      set clinic-capacity clinic-capacity + 2
    ]
  ]

  ;; Create universities/training centers
  create-universities num-universities [
    ;; Similar positioning as hospitals
    ifelse random-float 1 < 0.8 [
      let target-community one-of community-groups
      move-to target-community
      set heading random 360
      forward 2.5 + random 3
    ][
      setxy random-xcor random-ycor
    ]

    set shape "square"
    set color violet
    set size 1.5

    ;; Increase local training capacity
    ask patches in-radius 3 [
      set training-seats training-seats + 2
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; MAIN LOOP
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  ;; Process the veterans in batches for better performance
  process-veteran-batch

  ;; Apply policy effects to the environment
  apply-policies

  ;; Update conflict and societal dynamics
  update-war-dynamics
  update-societal-dynamics

  ;; Conditionally update veteran groups
  if enable-veteran-groups? [
    manage-simplified-veteran-groups
  ]

  ;; Update social interactions for the current batch
  update-social-effects

  ;; Update statistics tracking
  update-statistics

  ;; Update visualization
  update-visualization

  ;; Move to next batch
  set current-batch-start current-batch-start + batch-size
  if current-batch-start >= count veterans [
    set current-batch-start 0  ;; Start over with first batch next tick
  ]

  tick
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; VETERAN TRANSITION PROCESSES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Process a batch of veterans
to process-veteran-batch
  ;; Calculate range for current batch
  let batch-end min list (current-batch-start + batch-size) (count veterans)
  let current-batch veterans with [who >= current-batch-start and who < batch-end]

  ;; Process only veterans in current batch
  ask current-batch [
    ;; Core processes
    update-identity           ;; Update identity variables
    assess-and-seek-services  ;; Service-seeking based on well-being scores
    update-3T-risk            ;; Update 3T extremism risk factors
    evaluate-trajectories     ;; Update transition trajectory
    update-wellbeing          ;; Update well-being scores
    calc-integration          ;; Calculate overall integration score

    ;; Identity dynamics
    process-transition-dynamics
  ]
end

;; Update veteran identity salience and centrality (Dual-Salience model)
to update-identity
  ;; Calculate factors that boost civilian identity
  let employment-factor ifelse-value (employment-status = "employed") [1] [0]
  let education-factor ifelse-value (education-status = "graduated") [1] [0]

  ;; Calculate civilian identity boost - 2% per factor
  let civ-boost (employment-factor + education-factor +
                community-engagement + pol-identity-coach) * 0.02

  ;; Add boost from psychological resources
  let psych-boost (resilience + coping) / 100

  ;; Add boost from milestone count
  let milestone-boost transition-milestone-count * 0.005

  ;; Adjust for looking-back effect
  let nostalgia-penalty looking-back * 0.01

  ;; Calculate final civilian identity change
  let final-civ-change civ-boost + psych-boost + milestone-boost - nostalgia-penalty

  ;; Update civilian identity (max 1.0)
  set identity-civ min (list 1 (identity-civ + final-civ-change))

  ;; Military identity decreases as civilian identity increases (max 0.0)
  ;; But looking-back slows this decline
  let mil-decay final-civ-change * 0.8 * (1 - looking-back / 2)
  set identity-mil max (list 0 (identity-mil - mil-decay))

  ;; Update veteran identity centrality based on gap between identities
  let gap abs(identity-mil - identity-civ)

  ;; Veteran groups increase centrality
  let group-influence1 0
  let nearby-groups veteran-groups in-radius 5
  if any? nearby-groups [
    set group-influence1 mean [group-identity] of nearby-groups * 0.01
  ]

   ;; Calculate how much centrality should decrease
  let centrality-decrease 0.01  ;; base decrease
  if gap > 0.5 [set centrality-decrease centrality-decrease * 0.5]  ;; slow decrease when gap is large

  ;; Update centrality
  set identity-central max (list 0 (identity-central - centrality-decrease + group-influence1))

  ;; Update moral orientation (military vs civilian values)
  set moral-orient identity-mil - identity-civ
end

;; Service seeking based on well-being scores (replaces need flags system)
to assess-and-seek-services
  ;; Health services - seek if physical or mental well-being is low
  if wb-physical < 60 or wb-mental < 60 [
    try-clinic
  ]

  ;; Job seeking - seek if work well-being is low
  if wb-work < 50 and random-float 1 < pol-job-program [
    try-job
  ]

  ;; Education seeking - seek if work well-being is low and no job
  if wb-work < 40 and [training-seats] of patch-here > 0 [
    start-training
  ]

  ;; Housing seeking - seek if housing well-being is low
  if wb-housing < 60 [
    try-housing
  ]

  ;; Legal assistance - seek if legal well-being is low
  if wb-legal < 70 and random-float 1 < pol-legal-aid [
    resolve-legal
  ]

  ;; Spiritual engagement - seek if spiritual well-being is low
  if wb-spiritual < 50 [
    ;; Increase spiritual engagement based on support level
    set spiritual-engagement min (list 1 (spiritual-engagement + pol-spiritual-support))
  ]
end

;; Try to access healthcare services
to try-clinic
  ;; Check if clinic capacity available
  if [clinic-capacity] of patch-here > 0 [
    ;; Consume one clinic capacity unit
    ask patch-here [ set clinic-capacity clinic-capacity - 1 ]

    ;; Improve physical and mental well-being based on needs
    if wb-physical < 60 [
      set wb-physical min (list 100 (wb-physical + 5))
    ]

    if wb-mental < 60 [
      set wb-mental min (list 100 (wb-mental + 5))
      ;; Also reduce trauma score when mental health services are received
      set trauma-score max (list 0 (trauma-score - 0.05))
    ]
  ]
end

;; Try to find employment
to try-job
  ;; Check if jobs available
  if [jobs-available] of patch-here > 0 [
    ;; Consume one job vacancy
    ask patch-here [ set jobs-available jobs-available - 1 ]

    ;; Update employment status
    set employment-status "employed"

    ;; Improve work well-being
    set wb-work wb-work + 10

    ;; Add income (base + benefit)
    set income income + 1000 + pol-finance-benefit

    ;; Add milestone if first job
    if transition-milestone-count = 0 [
      set transition-milestone-count transition-milestone-count + 1
    ]
  ]
end

;; Try to access education/training
to start-training
  ;; Check if near a university first
  let nearby-university min-one-of universities [distance myself]

  ;; If near a university, higher chance of success
  ifelse nearby-university != nobody and distance nearby-university < 3 [
    ;; Update education status
    set education-status "graduated"

    ;; Increase civilian skills more at university
    set skills-civ skills-civ + 2

    ;; Improve work well-being
    set wb-work wb-work + 8

    ;; Add milestone
    set transition-milestone-count transition-milestone-count + 1
  ][
    ;; Otherwise check regular training seats
    if [training-seats] of patch-here > 0 [
      ;; Consume one training seat
      ask patch-here [ set training-seats training-seats - 1 ]

      ;; Update education status
      set education-status "graduated"

      ;; Increase civilian skills
      set skills-civ skills-civ + 1

      ;; Improve work well-being
      set wb-work wb-work + 5

      ;; Add milestone
      set transition-milestone-count transition-milestone-count + 1
    ]
  ]
end

;; Try to secure housing
to try-housing
  ;; Check if affordable housing available
  if [affordable-units] of patch-here > 0 [
    ;; Consume one housing unit
    ask patch-here [ set affordable-units affordable-units - 1 ]

    ;; Update housing status
    set housing-status "secure"

    ;; Improve housing well-being
    set wb-housing wb-housing + 10

    ;; Add milestone
    if transition-milestone-count < 3 [
      set transition-milestone-count transition-milestone-count + 1
    ]
  ]
end

;; Try to resolve legal issues
to resolve-legal
  ;; Check if veteran has legal issues
  if legal-status != "none" [
    ;; Update legal status
    set legal-status "none"

    ;; Improve legal well-being
    set wb-legal wb-legal + 10
  ]
end

;; Update 3T risk factors for extremism
to update-3T-risk
  ;; Calculate prejudice score
  ;; Base prejudice affected by local stigma but reduced by policy
  let base-prejudice-change [public-stigma] of patch-here * 0.01 - pol-anti-prejudice * 0.02

  ;; Identity gap increases prejudice vulnerability
  let identity-gap abs(identity-mil - identity-civ) * 0.01

  ;; Veteran group influence can increase prejudice
  let group-factor 0
  let extremist-groups veteran-groups in-radius 5 with [group-militarization > 0.7]
  if any? extremist-groups [
    set group-factor mean [group-militarization] of extremist-groups * 0.02
  ]

  ;; Update prejudice score
  set prejudice-score max (list 0 (min (list 1 (prejudice-score + base-prejudice-change + identity-gap + group-factor))))

  ;; Calculate trauma score
  ;; Trauma increases with low mental health, decreases naturally or with treatment
  let mental-health-factor max (list 0 (0.02 - (wb-mental / 500)))

  ;; Identity crises can trigger trauma
  let crisis-factor 0
  if identity-crisis? [set crisis-factor 0.02]

  ;; Combat exposure increases trauma vulnerability
  let combat-factor combat-exposure * 0.01

  ;; Update trauma score (natural recovery of 0.5% per tick)
  set trauma-score max (list 0 (min (list 1 (trauma-score + mental-health-factor + crisis-factor + combat-factor - 0.005))))

  ;; Calculate transition stress
  ;; Integration factor: lower integration increases stress
  let integration-factor max (list 0 (0.7 - integration / 100)) * 0.02

  ;; Resource factors: lacking resources increases stress
  let employment-factor ifelse-value (employment-status = "unemployed") [0.02] [0]
  let housing-factor ifelse-value (housing-status = "at-risk") [0.02] [0]

  ;; Identity factors: large gaps and looking back increase stress
  let identity-stress identity-gap * 0.02
  let nostalgia-stress looking-back * 0.01

  ;; Update transition stress
  set transition-stress max (list 0 (min (list 1 (transition-stress + integration-factor + employment-factor + housing-factor + identity-stress + nostalgia-stress))))

  ;; Transition stress decreases if integration is high
  if integration > 70 [
    set transition-stress max (list 0 (transition-stress - 0.05))
  ]

  ;; Add spatial clustering effects
  calculate-extremism-influence

  ;; Calculate overall extremism risk (average of 3 factors)
  set extremism-risk (prejudice-score + trauma-score + transition-stress) / 3

  ;; Set extremist flag if risk exceeds threshold
  ifelse extremism-risk >= extremist-threshold
    [ set extremist? true ]
    [
      ;; Can still become extremist through proximity, even if below threshold
      if not extremist? and random-float 1 < 0.05 [  ;; 5% chance check
        ;; Calculate extremist influence in surroundings
        let nearby-extremists veterans in-radius 3 with [extremist?]
        let extremist-count1 count nearby-extremists

        ;; Higher chance of becoming extremist if surrounded by others
        if extremist-count1 > 0 [
          let conversion-chance 0.02 * extremist-count1 * extremism-risk
          if random-float 1 < conversion-chance [
            set extremist? true
          ]
        ]
      ]
    ]
end

;; Calculate extremism influence within communities
to calculate-extremism-influence
  ;; Extremists cluster together due to similar views
  if extremist? and random-float 1 < 0.2 [  ;; 20% chance each tick
    ;; Look for other extremists
    let other-extremists other veterans with [extremist?]

    ;; Move toward other extremists if found
    if any? other-extremists [
      face min-one-of other-extremists [distance myself]
      forward 0.5
    ]
  ]

  ;; Extremists increase local prejudice
  if extremist? [
    ask patches in-radius 2 [
      set public-stigma min (list 1 (public-stigma + 0.01))
    ]
  ]

  ;; Calculate spatial influence on prejudice (contagion effect)
  let local-extremist-count count veterans in-radius 3 with [extremist?]

  ;; Local extremist presence increases prejudice
  if local-extremist-count > 0 [
    set prejudice-score min (list 1 (prejudice-score + (0.01 * local-extremist-count)))
  ]

  ;; Communities can help reduce extremism risk
  let supporting-communities community-groups in-radius 5 with [
    community-acceptance > 0.6
  ]

  if any? supporting-communities [
    ;; Calculate mitigation effect based on community acceptance
    let mitigation-factor count supporting-communities * 0.02

    ;; Reduce prejudice and transition stress through community support
    set prejudice-score max (list 0 (prejudice-score - mitigation-factor))
    set transition-stress max (list 0 (transition-stress - mitigation-factor))
  ]
end

;; Evaluate and update transition trajectory
to evaluate-trajectories
  ;; Only evaluate if not already returned to service
  if traj != "returned" [
    ;; Check for complete transition criteria
    if identity-civ >= complete-threshold and identity-central < centrality-cutoff [
      set traj "complete"
    ]

    ;; Check for partial transition criteria
    if identity-civ >= partial-threshold and traj = "partial" [
      set traj "partial"  ;; Maintain partial status
    ]
  ]

  ;; Chance of remobilization based on conflict intensity and mobilization risk
  if random-float 1 < (remobilization-rate * mobilization-risk) and traj != "returned" [
    set traj "returned"

    ;; Reset identity values for remobilized veterans
    set identity-mil 1                   ;; Full military identity
    set identity-civ identity-civ * 0.3  ;; Reduced civilian identity
    set identity-central 0.6             ;; Increased veteran centrality

    ;; Remobilization affects identity
    set looking-back min (list 1 (looking-back + 0.2)) ;; Increased nostalgia/connection

    ;; Increment deployment count
    set deployment-count deployment-count + 1

    ;; Update combat exposure
    set combat-exposure min (list 1 (combat-exposure + random-float 0.3))
  ]
end

;; Track and process transition milestones and identity crises
to process-transition-dynamics
  ;; Check for identity crisis conditions
  ;; Random chance of crisis for veterans in partial transition with high mismatch
  if not identity-crisis? and traj = "partial" and abs(identity-mil - identity-civ) > 0.5 and random-float 1 < 0.02 [
    set identity-crisis? true
    set identity-crisis-duration 0
  ]

  ;; Update existing crises
  if identity-crisis? [
    set identity-crisis-duration identity-crisis-duration + 1

    ;; During crisis, identity is more volatile
    ifelse random-float 1 < 0.5 [
      ;; 50% chance of moving toward military identity
      set identity-mil min (list 1 (identity-mil + 0.02))
      set identity-civ max (list 0 (identity-civ - 0.02))
    ] [
      ;; 50% chance of moving toward civilian identity
      set identity-mil max (list 0 (identity-mil - 0.02))
      set identity-civ min (list 1 (identity-civ + 0.02))
    ]

    ;; Crisis increases transition stress
    set transition-stress min (list 1 (transition-stress + 0.05))

    ;; Crisis ends after 5-10 ticks or with strong community support
    if identity-crisis-duration > 5 + random 5 or community-engagement > 0.8 [
      set identity-crisis? false

      ;; After crisis, identity tends to stabilize in a new direction
      ifelse identity-mil > identity-civ [
        ;; Resolved toward military identity
        set identity-mil min (list 1 (identity-mil + 0.1))
        set identity-civ max (list 0 (identity-civ - 0.05))
      ] [
        ;; Resolved toward civilian identity
        set identity-mil max (list 0 (identity-mil - 0.05))
        set identity-civ min (list 1 (identity-civ + 0.1))
      ]
    ]
  ]
end

;; Update well-being dimensions
to update-wellbeing
  ;; Physical well-being: natural decay offset by resilience
  set wb-physical max (list 0 (wb-physical - 0.05 + resilience * 0.02))

  ;; Mental well-being: natural decay offset by coping
  set wb-mental max (list 0 (wb-mental - 0.05 + coping * 0.02))

  ;; Financial well-being: based on income/debt ratio
  set wb-finance max (list 0 (wb-finance + (income - debt) / 2000))

  ;; Social well-being: based on community engagement
  set wb-social max (list 0 (wb-social - 0.03 + community-engagement * 0.1))

  ;; Family well-being: gradual improvement if social well-being is high
  if wb-social > 60 [
    set wb-family max (list 0 (wb-family + 0.02))
  ]

  ;; Spiritual well-being: based on spiritual engagement
  set wb-spiritual max (list 0 (wb-spiritual - 0.02 + spiritual-engagement * 0.1))

  ;; Work well-being: employment status affects it
  ifelse employment-status = "employed" [
    set wb-work max (list 0 (wb-work + 0.1))
  ][
    set wb-work max (list 0 (wb-work - 0.1))
  ]

  ;; Housing well-being: housing status affects it
  ifelse housing-status = "secure" [
    set wb-housing max (list 0 (wb-housing + 0.1))
  ]  [
    set wb-housing max (list 0 (wb-housing - 0.1))
  ]

  ;; Legal well-being: legal status affects it
  ifelse legal-status = "none" [
    set wb-legal max (list 0 (wb-legal + 0.05))
  ]  [
    set wb-legal max (list 0 (wb-legal - 0.1))
  ]

  ;; Cap all well-being scores at 100
  set wb-physical min (list 100 wb-physical)
  set wb-mental min (list 100 wb-mental)
  set wb-social min (list 100 wb-social)
  set wb-family min (list 100 wb-family)
  set wb-work min (list 100 wb-work)
  set wb-housing min (list 100 wb-housing)
  set wb-finance min (list 100 wb-finance)
  set wb-spiritual min (list 100 wb-spiritual)
  set wb-legal min (list 100 wb-legal)

  ;; Crises affect well-being
  if identity-crisis? [
    set wb-mental max (list 0 (wb-mental - 0.2))
    set wb-social max (list 0 (wb-social - 0.1))
  ]

  ;; Extremism affects community well-being
  if extremist? [
    set wb-social max (list 0 (wb-social - 0.1))
  ]

  ;; Update psychological resources based on well-being
  update-psychological-resources
end

;; Calculate overall integration score
to calc-integration
  ;; Average of all nine well-being dimensions
  set integration (wb-physical + wb-mental + wb-social + wb-family +
                  wb-work + wb-housing + wb-finance + wb-spiritual +
                  wb-legal) / 9
end

;; Update psychological resources based on well-being
to update-psychological-resources
  ;; Resilience correlates with physical and mental well-being
  set resilience max (list 0 (min (list 1 (resilience + (wb-physical + wb-mental - 100) / 2000))))

  ;; Coping correlates with mental and social well-being
  set coping max (list 0 (min (list 1 (coping + (wb-mental + wb-social - 100) / 2000))))

  ;; Looking back decreases with higher civilian identity and social well-being
  if identity-civ > 0.5 and wb-social > 60 [
    set looking-back max (list 0 (looking-back - 0.01))
  ]

  ;; Community engagement increases with social well-being
  set community-engagement max (list 0 (min (list 1 (community-engagement + (wb-social - 50) / 1000))))

  ;; Spiritual engagement increases with spiritual well-being
  set spiritual-engagement max (list 0 (min (list 1 (spiritual-engagement + (wb-spiritual - 50) / 1000))))
end

;; Update social effects for veterans
to update-social-effects
  ask veterans [
    ;; Find nearby veterans (implicit network)
    let nearby-vets other veterans in-radius 5

    ;; Calculate social influence based on proximity
    if any? nearby-vets [
      ;; Identity influence from similar veterans
      let similar-vets nearby-vets with [
        abs (identity-mil - [identity-mil] of myself) < 0.2 and
        abs (identity-civ - [identity-civ] of myself) < 0.2
      ]

    if any? similar-vets [
      ;; Similar veterans with similar identities reinforce each other, but effect depends on identities
      ifelse identity-mil > identity-civ [
        ;; If military identity is stronger, boost centrality
        set identity-central min (list 1 (identity-central + 0.01))
      ] [
        ;; If civilian identity is stronger, reduce centrality
        set identity-central max (list 0 (identity-central - 0.01))
      ]

      set wb-social wb-social + 1
    ]

      ;; Extremism influence
      let nearby-extremists nearby-vets with [extremist?]
      if any? nearby-extremists and random-float 1 < 0.05 [
        set prejudice-score prejudice-score + 0.02
      ]
    ]

    ;; Community influence (implicit)
    let nearby-community min-one-of community-groups [distance myself]
    if nearby-community != nobody and distance nearby-community < [community-influence] of nearby-community [
      ;; Community influence on transition stress and engagement
      set transition-stress max (list 0 (transition-stress - 0.02))
      set community-engagement min (list 1 (community-engagement + 0.01))
      set wb-social min (list 100 (wb-social + 0.5))
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; VISUALIZATION FUNCTIONS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Set up shape for veterans and other agents
to setup-shapes
  set-default-shape veterans "person"
  set-default-shape veteran-groups "flag"
  set-default-shape community-groups "house"
  ;set-default-shape providers "circle"
  set-default-shape providers "square"
  set-default-shape employers "box"
  set-default-shape faith-centres "triangle"
  ;set-default-shape hospitals "triangle"
  set-default-shape hospitals "square"
  set-default-shape universities "square"
end

;; Comprehensive visualization update
to update-visualization
  ;; Update veteran movement patterns
  update-movement

  ;; Update appearances
  update-veteran-appearance

  ;; Update patch colors
  update-patch-colors

  ;; Update labels
  update-label-visibility
end

;; Update veteran appearance with visual encodings
to update-veteran-appearance
  ask veterans [
    ;; Base color indicates trajectory
    ifelse traj = "complete" [
      set color green  ;; Green for complete transition
    ][
      ifelse traj = "partial" [
        set color yellow  ;; Yellow for partial integration
      ][
        set color orange  ;; Red-orange for returned to service
      ]
    ]

    ;; Size indicates integration level but keep it smaller for less cluttering                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               