globals [ cascade-count cascade-list ]

;;
;; Setup Procedures
;;
to setup
  ca
  set-default-shape turtles "circle"
  setxyz (max-pxcor * 2) (max-pycor * 2) (max-pzcor * 1.5)
  facexyz 0 0 (min-pzcor)
  set cascade-list []
  set cascade-count 0
end

;; fill up the world with sand piles
;; each column should have between 0 and 3 grains of sand.
to fill-world
  ask patches with [ pzcor = min-pzcor ] [
    let number-of-grains random 4
    sprout number-of-grains [
        set pitch -90
        set heading 0
        while [ any? other turtles-here ]
        [
          set zcor zcor + 1
        ]
        set color brown + zcor
    ]
  ]
end

;;
;; Runtime Procedures
;;
to go
  ask one-of patches with [pzcor = max-pzcor] [
    sprout 1 [
      set pitch -90
      set heading 0
      set color brown
      fall
    ]
  ]
  tick
end

;; have any grains of sand that can fall, fall
to fall ;; turtle procedure
  while [ (not any? turtles-on patch-ahead 1) and (zcor > min-pzcor) ]
  [ fd 1 display ]

  trigger-cascade
  set color brown + zcor
end

to trigger-cascade
  ;; have any sand which should cascade
  set cascade-count 0
  while [any? turtles with [zcor >= (min-pzcor + 3)]]
  [ cascade display ]
  update-plot
end

;; distribute the current sand pile to the neighboring sand piles
to cascade
  ;; ask one of the grains of sands that needs to cascade to cascade
  ask one-of turtles with [ zcor >= (min-pzcor + 3) and not any? turtles-at 0 0 1 ] [
    ;; update the cascade-count
    set cascade-count cascade-count + 1

    ;;  we really don't want all the turtles in the column just the top four.
    let turtles-in-column turtles with
      [xcor = [xcor] of myself and ycor = [ycor] of myself and zcor >= (([zcor] of myself) - 3)]
    ask turtles-in-column [
      set pitch 0
      ;; make sure each of the grains of sand
      ;; go in four different directions.
      set heading 90 * (zcor mod 4)

      ;; if it falls off the edge remove it
      ifelse out-of-bounds?
      [ die ]
      [
        set zcor min-pzcor
        fd 1
        let my-column turtles with [ xcor = [xcor] of myself and ycor = [ycor] of myself ]
        let top-turtle max-one-of my-column [ zcor ]

        ;; if we are not the top turtle then move to the top of the pile
        if top-turtle != self [ set zcor [zcor] of top-turtle + 1 ]
        set color brown + zcor
      ]
    ]
  ]
end

to-report out-of-bounds? ;; turtle reporter
  let p patch-ahead 1
  report (abs( [pxcor] of p - xcor ) > 1 or
          abs( [pycor] of p - ycor ) > 1)
end

;;
;; Plotting Procedures
;;
to update-plot

  ;; Check to see if we have anything to plot this turn
  if cascade-count > 0 [

    ;; clear the old line
    plot-pen-reset

    ;; put the current count on the list of all counts
    set cascade-list fput cascade-count cascade-list

    ;; create a list of -1's as long as size of the greatest cascade
    let log-histogram n-values (max cascade-list + 1) [-1]

    ;; foreach element of the list calculate the frequency with which it occurred
    foreach cascade-list [
      let old-value item ?1 log-histogram
      if old-value = -1 [
        set old-value 0
      ]
      set log-histogram replace-item ?1 log-histogram (old-value + 1)
    ]

    ;; take the log of that frequency
    set log-histogram map [safe-log ?] log-histogram

    ;; now determine the log of that value and plot the relationship
    let counter 1
    foreach log-histogram [
      if (?1 >= 0) [
        plotxy (log (counter - 1) 10) ?1
      ]
      set counter counter + 1
    ]
  ]
end

;; a special reporter so we do not ask for the log of -1
to-report safe-log [input]
  ifelse input = -1
  [ report -1 ]
  [ report log input 10 ]
end


; Copyright 2006 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
0
0
314
335
9
9
16.0
1
10
1
1
1
0
1
1
1
-9
9
-9
9
0
12
1
1
1
ticks

BUTTON
37
11
139
44
setup empty
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
38
94
141
127
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
39
129
141
162
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

PLOT
161
11
460
310
Log Size of Avalanche vs. Log Frequency
Log Size of Avalanche
Log Frequency
0.0
0.5
0.0
1.0
true
false
PENS
"default" 1.0 0 -16777216 true

BUTTON
38
46
139
79
setup full
setup\\nfill-world
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This model illustrates a phenomenon known as self-organized criticality. The world is filled with sand organized in columns. Falling sand stacks on top of the sand that is already there. Eventually a column will fall over because it gets too high, and the sand will spill into the surrounding area. This is called a cascade. When a falling column causes other columns to fall, the series of cascades is called an avalanche.  The size of an avalanche is the number of cascades that occur from one grain of sand falling.


HOW IT WORKS
------------
Initially the world is either empty or full, at each step of the model a grain of sand drops from the top of the world. The sand continues to fall until it hits another grain of sand; at that point it rests where it is and checks to see if it is the fourth grain of sand in the current pile. If it is it causes a cascade, which distributes the four grains of sand in the current pile in the four cardinal directions. These grains of sand then in turn check to see if they make their new sandpile taller than three grains, if they do then they cascade as well. If a particle of sand goes over the edge of the world then it disappears forever.


HOW TO USE IT
-------------
SETUP EMPTY initializes the model with no sand in it.
SETUP FULL initializes the model with all the piles at random heights but below the threshold of four.
GO ONCE will add one particle of sand to the world.
GO will continue adding particles of sand at random locations.


THINGS TO NOTICE
----------------
The graph illustrates the relationship between the logarithm of the size of cascades and the logarithm of the frequency of their occurrence. Per Bak and others pointed out that in this model this graph would eventually become a straight line. A straight line on a log-log graph is indicative of a power law, which means that the relationship between the x and y axes is of the form y = A^Bx. The fact that this power law occurs regardless of the starting circumstances and despite the fact that the process is random is what Per Bak called self-organized criticality.

THINGS TO TRY
-------------
Slow down the model using the speed slider to watch the avalanches occur.
Try using SETUP EMPTY and SETUP FULL; is the resultant pattern any different?
Can you explain why the graph is not a perfectly straight line?

EXTENDING THE MODEL
-------------------
The particles are currently colored by their depth in the sandpile. What if you colored them based on whether or not they were involved in an avalanche recently?
What if you make the limit on high the sandpiles can go larger than four? How about eight or twelve?

NETLOGO FEATURES
----------------
This model makes use of the fact that you can set the origin of a 3D model anywhere you want. It also creates its own logarithmic graph.


RELATED MODELS
--------------
There is a 2D version of this model in the Community Models called SANDPILE.


CREDITS AND REFERENCES
----------------------
Self organized criticality was originally investigated by Per Bak and collaborators.  The basis for this model is in Per Bak's book "How Nature Works".


HOW TO CITE
-----------
If you mention this model in an academic publication, we ask that you include these citations for the model itself and for the NetLogo software:
- Wilensky, U. (2006).  NetLogo Sandpile 3D model.  http://ccl.northwestern.edu/netlogo/models/Sandpile3D.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
- Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use:
- Copyright 2006 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/Sandpile3D for terms of use.


COPYRIGHT NOTICE
----------------
Copyright 2006 Uri Wilensky. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 30 30 240

circle 2
false
0
Circle -7500403 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7500403 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 3D 4.1pre7
@#$#@#$#@
setup
fill-world
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
