globals [
  vend-lines  ;; list of vending machines containing a list students for each vending machine
  vend-positions ;; list of xcors of each vending machine

  prize-streaks  ;; list of prize streaks for each line
  wealth-distribution-list  ;; list of (num-prizes / (money-spent)
  vending-machine-colors  ;; list of colors for each line

  max-line-length  ;; the longest a line can be (determined to be 14 for the current screen size)

  ;; lists used to create the various turtles
  shape-names       ;; list that holds the names of the non-sick shapes a student's turtle can have
  colors            ;; list that holds the colors used for students' turtles
  color-names       ;; list that holds the names of the colors used for students' turtles
  used-shape-colors ;; list that holds the shape-color pairs that are already being used

  ;; misc
  clock        ;; keeps track of the number of times through the go procedure (if there is at least one turtle infected)
  max-possible-codes ;; total number of possible unique shape/color combinations

  ;; quick start instructions variables
  quick-start  ;; current quickstart instruction displayed in the quickstart monitor
  qs-item      ;; index of the current quickstart instruction
  qs-items     ;; list of quickstart instructions
]


breed [ students student ]
breed [ vending-machines vending-machine ]

students-own [
  line-number   ;; the vending machine number you are standing in line for
  money-spent   ;; the number of times student has 'vended'
  line-position ;; the place they are in the line, 1 being the front
  num-prizes    ;; the number of prizes won
  user-id  ;; unique id, input by the client when they log in, to identify each student turtle
]

vending-machines-own [goods-dispensed prizes-dispensed last-vended-prize]


to startup
  setup
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
end

to setup
  ca
  setup-vars
  create-vending-machines 4
  [
    set shape "vending machine"
    set size 2
    set color (item who vending-machine-colors)
    set goods-dispensed 0
    set prizes-dispensed 0
    set last-vended-prize 0
    setxy ((floor (max-pxcor / 2) * (who - 2) + 3 )) (max-pycor - 1)
    set vend-positions (lput xcor vend-positions)
  ]
  setup-quick-start
end

to setup-vars
  set clock 0
  set max-line-length 14
  set vend-lines [[] [] [] []]
  set vend-positions []
  set prize-streaks [[] [] []  []]
  set wealth-distribution-list []
  set vending-machine-colors [red green blue]
  set vending-machine-colors lput (orange + 1) vending-machine-colors

  set shape-names ["wide wedge"  "thin wedge" "big boat" "pickup truck"
                   "nu" "uu"  "butterfly" "wolf" "sheep" "lobster" "monster"
                   "moose" "bear" "teddy bear"]
  set colors      [ white   gray   brown   yellow   green   lime   turquoise
                    cyan   sky   blue   violet ]
  ;; adjust a few colors so they don't blend in with the red infection dot too much
  set colors lput (orange + 1) colors
  set colors lput (magenta + 0.5) colors
  set colors lput (pink + 2.5) colors
  set color-names ["white" "gray" "brown" "yellow" "green" "lime" "turquoise"
                   "cyan" "sky" "blue" "violet" "orange" "magenta" "pink"]
  set max-possible-codes (length colors * length shape-names)
  set used-shape-colors []
end


to re-run
  set vend-lines [[] [] [] []]
  set vend-positions []
  set prize-streaks [[] [] []  []]
  set wealth-distribution-list []
  ask vending-machines
  [
    set label ""
    set goods-dispensed 0
    set prizes-dispensed 0
    set last-vended-prize 0
  ]
  ask students
  [
    set num-prizes 0
    set money-spent 0
    send-info-to-clients
  ]
  clear-all-plots
end

;; give the user some information about what the setup button does so they can
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? (word "The SETUP button should only be used when starting "
              "over with a new group (such as a new set of students) since "
              "all data is lost.  Use the RE-RUN button for continuing with "
              "an existing group."
              "\\n\\nDo you really want to setup the model?")
  [
    user-message (word "Before closing this dialog, please do the following:"
                 "\\n  -Have everyone that is currently logged in, log off and "
                 "then kick all remaining clients with the HubNet Console.")
    setup
  ]
end


to go
  every 0.1
  [
    ;; get commands and data from the clients
    listen-clients

   ask students with [line-position = 1]
   [
      hubnet-send user-id "Instructions:" "Click Vend!"
   ]
   ask students
   [
     send-info-to-clients
   ]
  do-plot
  set clock (clock + 1)
  ]
end

;; observer procedure
to add-student-to-line [student num]
  set vend-lines (replace-item num vend-lines (lput student (item num vend-lines)))
  ask student 
  [ 
    set line-position num 
    set line-number num
    setxy ((item line-number vend-positions)) ( max-pycor - 2.5 - 2 * line-length line-number)
  ]
end

to leave-line
  hide-turtle
  foreach (item line-number vend-lines)
  [
    ask ?
    [
      if (line-position > [line-position] of myself)
        [fd 2 set line-position (line-position - 1)]
    ]
  ]
  set vend-lines (replace-item line-number vend-lines (remove self (item line-number vend-lines)))
end

;; student procedure
;; moves student to the back of the chosen line
;; if the line chosen is too long, the student will remain in the line he is currently in
to move-to-line [num]
  if line-length num >= max-line-length  ;; if the line is too long, go to the back of the current line
    [set num line-number]
  foreach (item line-number vend-lines)
  [
    ask ? [
      if (line-position > [line-position] of myself)
        [fd 2 set line-position (line-position - 1)]
    ]
  ]
  set vend-lines (replace-item line-number vend-lines (remove self (item line-number vend-lines)))
  add-student-to-line self num
end

;; observer procedure
;; returns the number of students at a particular vending machine
to-report line-length [num]
  report length (item num vend-lines)
end


;;vending-machine procedure
to vend [student]
  if ((who = 0 and random 10 < vend-1) or
      (who = 1 and random 10 < vend-2) or
      (who = 2 and random 10 < vend-3) or
      (who = 3 and random 10 < vend-4))
  [
    ask student [ set num-prizes num-prizes + 1 ]
    set prizes-dispensed (prizes-dispensed + 1)
    set prize-streaks (replace-item who prize-streaks (lput last-vended-prize (item who prize-streaks)))
    set last-vended-prize 0
  ]
  ask student [ set money-spent money-spent + 1 ]
  set goods-dispensed (goods-dispensed + 1)
  set last-vended-prize (last-vended-prize + 1)
  set label last-vended-prize
end

to-report average-prizes-in-line [num]
let num-prizes-list 0

  set num-prizes-list [num-prizes] of students with [line-number = num]
  ifelse (sum num-prizes-list = 0)
    [report 0]
    [report mean num-prizes-list]
end


to do-plot
  set-current-plot "Prizes Won vs. Money Spent"
  histogram ([num-prizes / money-spent] of students with [money-spent != 0])
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to run the HubNet activity."
    "Optional: Zoom In (see Tools in the Menu Bar)"
    "Press the GO button."
    "Everyone: Open up a HubNet Client on your machine and..."
      "type your user name, select this activity and press ENTER."
    "Teacher: Have the students change their move-to slider to..."
      "acquaint themselves with the interface."
    "When the students receive the 'Vend Now' instruction ..."
      "they should click VEND"
    "Teacher: To start the simulation over ..."
      "stop the model by pressing the NetLogo GO button, if it is on..."
        "click RE-RUN to reset the data and the plots, and press GO"
    "Teacher: To start the simulation over with a new group,..."
      "stop the model by pressing the NetLogo GO button, if it is on..."
        "click SETUP, and follow these instructions again from the beginning."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-student ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end

to execute-command [command]
  if member? "Move to" command
  [
    ask students with [user-id = hubnet-message-source]
    [
      move-to-line ((read-from-string last command) - 1)
      hubnet-send user-id "Instructions:" "Wait in Line"
    ]
    stop
  ]

  if command = "vend"
  [
    ask students with [user-id = hubnet-message-source]
    [
      ifelse line-position = 1
      [
        ;; since vending machines are the first four turtles, their who is the line-number
        ask turtle line-number [vend myself]
        move-to-line line-number
        hubnet-send user-id "Instructions:" "Wait in Line"
      ]
      [
        hubnet-send user-id "Instructions:" "Wait for your turn"
      ]
    ]
  ]
end

;; Create a student, set its shape and color, put it in a line
;; and tell the node about itself
to create-new-student
  create-students 1
  [
    setup-student-vars
    add-student-to-line self (who mod 4)
    hubnet-send user-id "Instructions:" "Wait in Line"
    send-info-to-clients
  ]

end

;; Remove the student from her line and kill her
to remove-student
  ask students with [user-id = hubnet-message-source]
  [
    leave-line
    set used-shape-colors remove my-code used-shape-colors
    die
  ]
end


;; sets the turtle variables to appropriate initial values
to setup-student-vars  ;; turtle procedure
    set user-id hubnet-message-source
    set-unique-shape-and-color
    set heading 0
    set money-spent 0
    set num-prizes 0
    set size 1.75
    set line-number (who mod 4)
    setxy (item line-number vend-positions) ( max-pycor - 2.5 - 2.2 * line-length line-number)
    set line-position line-length line-number
end

;; sends the appropriate monitor information back to the client
to send-info-to-clients
  hubnet-send user-id "You are a:" (word (color-string color) " " shape)
  hubnet-send user-id "Money Spent:" money-spent
  hubnet-send user-id "Position In Line:" line-position
  hubnet-send user-id "Line Number:" line-number + 1
  hubnet-send user-id "Prizes Won:" num-prizes
  if money-spent != 0
    [hubnet-send user-id "Prizes Won vs. Money Spent:" (num-prizes / money-spent)]
end

to set-unique-shape-and-color
  let code 0

  set code random max-possible-codes
  while [member? code used-shape-colors and count students < max-possible-codes]
  [
    set code random max-possible-codes
  ]
  set used-shape-colors (lput code used-shape-colors)
  set shape item (code mod length shape-names) shape-names
  set color item (code / length shape-names) colors
end

;; translates a student turtle's shape and color into a code
to-report my-code
  report (position shape shape-names) + (length shape-names) * (position color colors)
end


;; report the string version of the turtle's color
to-report color-string [color-value]
  report item (position color-value colors) color-names
end
@#$#@#$#@
GRAPHICS-WINDOW
111
46
517
473
16
16
12.0
1
16
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks

CC-WINDOW
5
534
922
629
Command Center
0

SLIDER
119
13
211
46
vend-1
vend-1
0
10
5
1
1
NIL
HORIZONTAL

SLIDER
215
13
308
46
vend-2
vend-2
0
10
6
1
1
NIL
HORIZONTAL

SLIDER
314
13
406
46
vend-3
vend-3
0
10
6
1
1
NIL
HORIZONTAL

SLIDER
415
14
507
47
vend-4
vend-4
0
10
4
1
1
NIL
HORIZONTAL

BUTTON
12
109
98
148
setup
setup-prompt
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
12
182
98
222
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
122
474
203
519
Wealth of 1
average-prizes-in-line 0
2
1
11

MONITOR
215
475
296
520
Wealth of 2
average-prizes-in-line 1
2
1
11

MONITOR
310
475
391
520
Wealth of 3
average-prizes-in-line 2
2
1
11

MONITOR
409
475
490
520
Wealth of 4
average-prizes-in-line 3
2
1
11

PLOT
527
132
913
338
Prizes Won vs. Money Spent
Prizes Won vs. Money Spent
Frequency
0.0
1.0
0.0
10.0
true
false
PENS
"default" 0.05 1 -16777216 true

MONITOR
526
10
911
55
Quick Start Instructions
quick-start
0
1
11

BUTTON
526
58
680
91
Restart Instructions
setup-quick-start
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
741
59
826
92
Previous
view-prev
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
826
59
911
92
Next
view-next
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
12
147
98
180
NIL
re-run
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
VERSION
-------
$Id: vending.nlogo 38634 2008-03-11 13:35:54Z everreau $


Each student stands in line to get something from the vending machine.  For each vending machine, there is a certain probability that it will give out a prize.  After each student gets something from the vending machine, he picks a random line to stand at the end of. If the line he picks is too long, he just moves to the back of his own line, which is known not to be too long.

NOTE: "wealth" in the UI refers to prizes won, not prizes won vs. money spent

WHAT IS IT?
-----------
The students stand in lines to get an item from a vending machine.  Each vending machine has a certain probability of giving away a prize with the item.  After they receive the item, they move to the back of their line.  Students can change the line they are in at any time, taking place at the end of the line.

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------
QUICKSTART INSTRUCTIONS:
------------------------
Teacher: Follow these directions to run the HubNet activity.
Optional: Zoom In (see Tools in the Menu Bar)
Optional: Change any of the settings.  Press the GO button to start the model.
Everyone: Open up a HubNet Client on your machine and input the IP Address of this computer, type your user name in the user name box and press ENTER.

Teacher: Have the students move their turtles around to acquaint themselves with the interface.  Then press the INFECT NetLogo button to start the simulation.  Watch the plot of the number infected.

Teacher: To rerun the activity with the same group, stop the model by pressing the NetLogo GO button, if it is on.  Change any of the settings that you would like.  To overlay the new run's plot data change the value of the slider DATA-SET. If you set DATA-SET to a value that already had data in it, the new data will over-write the old data.  Press the NetLogo RE-RUN button.  Restart the simulation by pressing the NetLogo GO button again.  Infect some turtles and continue.

Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on and follow these instructions again from the beginning.

BUTTONS:
--------
SETUP - clears all turtles and patches and the plot.  This should only be pressed when starting out with a new group of users since all data is lost.
GO - runs the simulation
RE-RUN - cures all turtles, and clears the current plot pen selected with DATA-SET.  This should be used to setup the model again for collecting more data or running the model again with the same users connected.
CREATE ANDROIDS - adds randomly moving turtles to the simulation
INFECT - infects some of the turtles in the simulation
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

SLIDERS:
--------
NUMBER - determines how many androids are created by the CREATE ANDROIDS button
ANDROID-DELAY - the delay time, in seconds, for android movement - the higher the number, the slower the androids move
INITIAL-NUMBER-SICK - the number of turtles that become infected spontaneously when the INFECT button is pressed
PERCENTAGE-INFECTION - sets the percentage chance that every tenth of a second a healthy turtle will become sick if it is on the same patch as an infected turtle
DATA-SET - sets which plot pen the data from this run will be plotted with.  This is useful for comparing several runs of the model to one another.  All pens are cleared by SETUP.

SWITCHES:
---------
WANDER? - when on, the androids wander randomly.  When off, they sit still
SHOW-ILL? - when on, sick turtles add to their original shape a red circle.  When off, they can move through the populace unnoticed
SHOW-ILL-ON-CLIENTS? - when on, the clients will be told if their turtle is sick or not.

MONITORS:
---------
TURTLES - the number of turtles in the simulation
NUM-INFECTED - the number of turtles that are infected

PLOTS:
------
NUMBER SICK - shows the number of sick turtles versus time


CLIENT INFORMATION
------------------
After logging in, the client interface will appear for the students, and if GO is pressed in NetLogo they will be assigned a turtle which will be described in the YOU ARE A: monitor.  And their current location will be shown in the LOCATED AT: monitor.  If the student doesn't like their assigned shape and/or color they can hit the CHANGE APPEARANCE button at any time to change to another random appearance.

The SICK? monitor will show one of three values: "true" "false" or "N/A".  "N/A" will be shown if the NetLogo SHOW-ILL-ON-CLIENTS? switch is off, otherwise "true" will be shown if your turtle is infected, or "false" will be shown if your turtle is not infected.

The student controls the movement of their turtle with the UP, DOWN, LEFT, and RIGHT buttons and the STEP-SIZE slider.  Clicking any of the directional buttons will cause their turtle to move in the respective direction a distance of STEP-SIZE.

The students can watch the progress of the disease in the NUMBER SICK plot which is identical to the plot of the same name in NetLogo.


THINGS TO NOTICE
----------------
No matter how you change the various parameters, the same basic plot shape emerges.  After using the model once with the students, ask them how they think the plot will change if you alter a parameter.  Altering the initial percentage sick and the percentage infection will have different effects on the plot.


THINGS TO TRY
-------------
Use the model with the entire class to serve as an introduction to the topic.  Then have students use the NetLogo model individually, in a computer lab, to explore the effects of the various parameters.  Discuss what they find, observe, and can conclude from this model.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

android
false
0
Rectangle -7500403 true true 105 74 210 239
Polygon -7500403 true true 104 78 34 129 47 148 114 89
Polygon -7500403 true true 198 81 274 108 258 142 192 104
Polygon -7500403 true true 115 239 115 289 133 289 133 237
Polygon -7500403 true true 176 235 176 287 192 287 192 234
Rectangle -7500403 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bear
false
0
Rectangle -7500403 true true 106 35 181 92
Circle -7500403 true true 110 17 18
Circle -7500403 true true 128 17 19
Rectangle -7500403 true true 80 58 111 89
Rectangle -7500403 true true 98 95 101 95
Rectangle -7500403 true true 92 93 192 104
Rectangle -7500403 true true 83 101 199 114
Rectangle -7500403 true true 78 107 205 130
Rectangle -7500403 true true 74 122 209 240
Rectangle -7500403 true true 77 236 204 249
Rectangle -7500403 true true 106 89 180 104
Rectangle -7500403 true true 80 247 198 258
Rectangle -7500403 true true 85 255 108 281
Rectangle -7500403 true true 169 252 195 283
Rectangle -7500403 true true 158 276 171 283
Rectangle -7500403 true true 74 276 90 281
Circle -7500403 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7500403 true true 51 163 87 184
Rectangle -7500403 true true 45 166 53 169
Rectangle -7500403 true true 42 172 52 173
Rectangle -7500403 true true 38 178 53 183
Circle -7500403 true true 72 62 20

big boat
false
0
Polygon -6459832 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7500403 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7500403 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7500403 true true 30 206 234 220
Rectangle -7500403 true true 52 224 234 236
Rectangle -7500403 true true 78 240 234 250

box
true
0
Polygon -7500403 true true 45 255 255 255 255 45 45 45

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7500403 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7500403 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7500403 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7500403 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7500403 true 167 47 159 82
Line -7500403 true 136 47 145 81
Circle -7500403 true true 165 45 8
Circle -7500403 true true 134 45 6
Circle -7500403 true true 133 44 7
Circle -7500403 true true 133 43 8

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

horse
false
0
Polygon -7500403 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7500403 true true 61 180 75 268
Rectangle -7500403 true true 89 187 104 275
Rectangle -7500403 true true 210 186 225 269
Rectangle -7500403 true true 241 178 254 275
Polygon -7500403 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7500403 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7500403 true true 65 65 70 50
Polygon -7500403 true true 65 67 74 46 79 65 93 49 88 73 65 67

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

lobster
false
0
Polygon -7500403 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7500403 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7500403 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7500403 true true 71 125 117 133
Rectangle -7500403 true true 73 149 119 157
Rectangle -7500403 true true 78 178 123 187
Rectangle -7500403 true true 183 131 219 138
Rectangle -7500403 true true 180 152 216 160
Rectangle -7500403 true true 173 176 217 184
Rectangle -7500403 true true 127 56 136 82
Rectangle -7500403 true true 160 55 169 81

monster
false
0
Rectangle -7500403 true true 77 100 213 245
Rectangle -7500403 true true 189 243 213 273
Rectangle -7500403 true true 78 242 102 271
Rectangle -7500403 true true 59 119 83 239
Rectangle -7500403 true true 203 118 225 239
Rectangle -7500403 true true 225 123 264 158
Rectangle -7500403 true true 262 124 272 131
Rectangle -7500403 true true 263 133 273 140
Rectangle -7500403 true true 263 145 271 156
Rectangle -7500403 true true 19 122 61 156
Rectangle -7500403 true true 9 125 19 130
Rectangle -7500403 true true 11 135 20 140
Rectangle -7500403 true true 9 146 20 151
Rectangle -7500403 true true 89 79 198 101
Rectangle -7500403 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7500403 true true 123 17 115 34 131 34
Polygon -7500403 true true 165 15 149 33 172 34

moose
false
0
Polygon -7500403 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7500403 true true 61 180 75 268
Rectangle -7500403 true true 89 187 104 275
Rectangle -7500403 true true 210 186 225 269
Rectangle -7500403 true true 241 178 254 275
Polygon -7500403 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7500403 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7500403 true true 65 65 70 50
Polygon -7500403 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7500403 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66

nu
false
0
Rectangle -7500403 true true 16 31 60 285
Rectangle -7500403 true true 120 31 164 285
Rectangle -7500403 true true 182 31 226 285
Rectangle -7500403 true true 256 31 298 285
Rectangle -7500403 true true 214 241 274 285
Polygon -7500403 true true 60 31 118 209 120 281 60 105
Rectangle -7500403 true true 110 211 140 249

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 124 22 63
Rectangle -7500403 true true 128 79 187 164
Polygon -7500403 true true 128 78 80 126 101 140 128 106 135 107
Polygon -7500403 true true 185 79 236 119 217 139 184 103
Polygon -7500403 true true 129 162 107 233 145 233 145 162
Polygon -7500403 true true 175 162 175 232 207 232 186 158

pickup truck
false
0
Polygon -7500403 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7500403 true true 195 194 59
Circle -7500403 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7500403 true true 21 142 139 142 139 136 13 136

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

teddy bear
false
0
Circle -7500403 true true 110 21 81
Circle -7500403 true true 97 7 39
Circle -7500403 true true 171 5 39
Polygon -7500403 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7500403 true true 134 238 34
Polygon -7500403 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7500403 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7500403 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7500403 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 127 75 150 85
Line -16777216 false 151 85 177 72
Polygon -1 true false 152 204 115 167 131 150 150 168 168 152 184 167

thin wedge
false
0
Polygon -7500403 true true 133 20 70 252 127 148 177 245

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

uu
false
0
Rectangle -7500403 true true 58 44 104 226
Rectangle -7500403 true true 88 178 194 224
Rectangle -7500403 true true 180 44 222 224
Rectangle -7500403 true true 122 74 168 284
Rectangle -7500403 true true 124 240 250 286
Rectangle -7500403 true true 240 76 284 286

vending machine
false
0
Rectangle -7500403 true true 60 15 225 285
Rectangle -1 true false 75 105 90 120
Rectangle -1 true false 75 135 90 150
Rectangle -1 true false 75 165 90 180
Rectangle -1 true false 75 195 90 210
Rectangle -1 true false 75 75 90 90
Rectangle -16777216 true false 90 225 195 255
Rectangle -7500403 true true 180 120 210 135

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wide wedge
false
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

wolf
false
0
Rectangle -7500403 true true 15 105 105 165
Rectangle -7500403 true true 45 90 105 105
Polygon -7500403 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7500403 true true 105 120 263 195
Rectangle -7500403 true true 108 195 259 201
Rectangle -7500403 true true 114 201 252 210
Rectangle -7500403 true true 120 210 243 214
Rectangle -7500403 true true 115 114 255 120
Rectangle -7500403 true true 128 108 248 114
Rectangle -7500403 true true 150 105 225 108
Rectangle -7500403 true true 132 214 155 270
Rectangle -7500403 true true 110 260 132 270
Rectangle -7500403 true true 210 214 232 270
Rectangle -7500403 true true 189 260 210 270
Line -7500403 true 263 127 281 155
Line -7500403 true 281 155 281 192

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1pre1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
238
10
634
406
16
16
12.0
1
16
1
1
1
0
1
1
1
-16
16
-16
16

MONITOR
26
237
121
286
Money Spent:
NIL
0
1

MONITOR
126
237
221
286
Prizes Won:
NIL
0
1

MONITOR
264
482
357
531
Line Number:
NIL
3
1

BUTTON
27
119
209
167
vend
NIL
NIL
1
T
OBSERVER
NIL
NIL

MONITOR
26
65
209
114
Instructions:
NIL
3
1

MONITOR
358
482
468
531
Position In Line:
NIL
3
1

MONITOR
26
289
221
338
Prizes Won vs. Money Spent:
NIL
2
1

MONITOR
27
10
146
59
You are a:
NIL
0
1

BUTTON
263
444
355
477
Move to 1
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
355
444
447
477
Move to 2
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
447
444
539
477
Move to 3
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
539
444
631
477
Move to 4
NIL
NIL
1
T
OBSERVER
NIL
NIL

@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
