;A simplified Lake Michigan Foodweb with Obstacles
globals [

  reproduce-energy-mid
  distinction
  
  energy-from-phytoplankton ;; keep track of how much phytoplankton there is
  phytoplankton-regrowth-time ;; how quickly phytoplankton grows back
  start-percent-phytoplankton
  phytoplankton-growth-rate ;; odds of phytoplankton spreading
   
  zooplankton-food-filter 
  zooplankton-start-amount 
  zooplankton-gain-from-food 
  zooplankton-reproduce 
  zooplankton-reproduce-energy
  
  oshrimp-food-filter 
  oshrimp-start-amount 
  a-oshrimp-gain-from-food 
  a-oshrimp-reproduce 
  a-oshrimp-reproduce-energy
  
  mollusks-food-filter 
  mollusks-start-amount 
  mollusk-gain-from-food 
  mollusk-reproduce 
  mollusk-reproduce-energy

  burbot-food-filter 
  burbot-start-amount 
  a-burbot-gain-from-food 
  a-burbot-reproduce 
  a-burbot-reproduce-energy
  
  round_gobys-food-filter
  round_gobys-start-amount 
  round_goby-gain-from-food 
  round_goby-reproduce
  round_goby-reproduce-energy
  
  top_preds-food-filter 
  top_preds-start-amount 
  top_pred-gain-from-food 
  top_pred-reproduce
  top_pred-reproduce-energy

  ignore-zooplankton
  ignore-oshrimp
  ignore-mollusks
  ignore-burbot 
  ignore-round_gobys
  ignore-top_preds
  invalid-food-web invaded?
]

breed [ phytoplankton a-phytoplankton ]
breed [ zooplankton a-zooplankton ]
breed [ oshrimp a-oshrimp ] 
breed [ mollusks mollusk ]
breed [ round_gobys round_goby ]
breed [ top_preds top_pred ]
breed [ burbot a-burbot ]

turtles-own [ energy ] 
patches-own [ countdown has-roots ]

to initialize
  clear-turtles
  clear-all-plots
  reset-ticks
  tick
    
  set zooplankton-start-amount 100
  set oshrimp-start-amount 50
  set mollusks-start-amount 100
  set burbot-start-amount 5
  set round_gobys-start-amount 15
  set top_preds-start-amount 0

  set invalid-food-web false
  
  ifelse zooplankton-start-amount = 0
   [ set ignore-zooplankton true]
   [ set ignore-zooplankton false]
  ifelse oshrimp-start-amount = 0
   [ set ignore-oshrimp true]
   [ set ignore-oshrimp false]
  ifelse mollusks-start-amount = 0
   [ set ignore-mollusks true]
   [ set ignore-mollusks false]
  ifelse burbot-start-amount = 0
   [ set ignore-burbot true]
   [ set ignore-burbot false]
  ifelse round_gobys-start-amount = 0
   [ set ignore-round_gobys true]
   [ set ignore-round_gobys false]
   ifelse top_preds-start-amount = 0
   [ set ignore-top_preds true]
   [ set ignore-top_preds false]
  set zooplankton-food-filter ""
  set oshrimp-food-filter ""
  set mollusks-food-filter ""
  set burbot-food-filter ""
  set round_gobys-food-filter ""
  set distinction 0
  
  parse-food-web
  setup-constants

  ifelse not invalid-food-web [
    ;; set globals
    set start-percent-phytoplankton 10
    set energy-from-phytoplankton 3
    set phytoplankton-regrowth-time 37
    set phytoplankton-growth-rate 3
    
    set zooplankton-gain-from-food 40 ;; zooplankton
    set zooplankton-reproduce 55
    set zooplankton-reproduce-energy 45
    
    set a-oshrimp-gain-from-food 10
    set a-oshrimp-reproduce 38.9
    set a-oshrimp-reproduce-energy 65
    
    set mollusk-gain-from-food 35
    set mollusk-reproduce 39
    set mollusk-reproduce-energy 10
  
    set a-burbot-gain-from-food 35
    set a-burbot-reproduce 38.2
    set a-burbot-reproduce-energy 48
    
    set round_goby-gain-from-food 35
    set round_goby-reproduce 38.2
    set round_goby-reproduce-energy 65
    
    set invaded? false
    set top_pred-gain-from-food 75
    set top_preds-start-amount 15
    set top_pred-reproduce 5
    set top_pred-reproduce-energy 55
    
  
  ;; create the phytoplankton, then initialize their variables
    set-default-shape phytoplankton "phytoplankton"
    ask patches [
      if pcolor != black [set pcolor blue + 2]
      if random 100 < start-percent-phytoplankton 
      [ sprout-phytoplankton 1 [ init-phytoplankton rt random 90 fd 0.5] ]
      ]
  
  ;; create the zooplankton, then initialize their variables
    set-default-shape zooplankton "zooplankton"
    populate-zooplankton zooplankton-start-amount true
    
  ;; create the oshrimp, then initialize their variables
    set-default-shape oshrimp "oshrimp"
    populate-oshrimp oshrimp-start-amount true
    
  ;; create the mollusks, then initialize their variables
    set-default-shape mollusks "mollusk"
    populate-mollusks mollusks-start-amount true
  
  ;; create the burbot, then initialize their variables
    set-default-shape burbot "black fish"
    populate-burbot burbot-start-amount true
  
    set-default-shape round_gobys "roundgoby"
    populate-round_gobys round_gobys-start-amount true
    
    set-default-shape top_preds "big fish"
    
    update-plot
  ]
  [ ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
    set invalid-food-web true]
end

to setup-constants
   set reproduce-energy-mid 75
end

to go
  if 0 = ticks [ user-message "Please press the 'Setup' button" stop ]
  ifelse invalid-food-web [ user-message "You need to update your food web and press set up" stop ]
  [
  if not any? zooplankton and not ignore-zooplankton [
    ifelse user-yes-or-no? "No more zooplankton in this part of the harbor.\nHave a new zooplankton scuttle in?"
    [ populate-zooplankton 1 false ]
    [ set ignore-zooplankton true ]
  ]
  if not any? oshrimp and not ignore-oshrimp [
    ifelse user-yes-or-no? "No more oshrimp in this part of the harbor.\nHave a new oshrimp swim in?"
    [ populate-oshrimp 1 false ]
    [ set ignore-oshrimp true ]
  ]
  if not any? mollusks and not ignore-mollusks [
    ifelse user-yes-or-no? "No more mollusks in this part of the harbor.\nHave a new mollusk swim in?"
    [ populate-mollusks 1 false ]
    [ set ignore-mollusks true ]
  ]
  if not any? round_gobys and not ignore-round_gobys [
    ifelse user-yes-or-no? "No more round gobys in this part of the harbor.\nHave a new round goby swim in?"
    [ populate-round_gobys 1 false ]
    [ set ignore-round_gobys true ]
  ]
  if not any? top_preds and not ignore-top_preds [
    ifelse user-yes-or-no? "No more top preds in this part of the harbor.\nHave a new top pred swim in?"
    [ populate-top_preds 1 false ]
    [ set ignore-top_preds true ]
  ]
  if not any? burbot and not ignore-burbot [
    ifelse user-yes-or-no? "No more burbots in this part of the harbor.\nHave a new burbot swim in?"
    [ populate-burbot 1 false ]
    [ set ignore-burbot true ]
  ]
  ask zooplankton [
    moveslow
    set energy energy - 0.1
    eat zooplankton-food-filter
    death
    reproduce-zooplankton
  ]
  ask oshrimp [
    avoid-obstacles
    move
    set energy energy - 0.1
    eat oshrimp-food-filter
    death
    reproduce-oshrimp
    
  ]
  ask mollusks [
    moveslow
    set energy energy - 0.07
    eat mollusks-food-filter
    death
    reproduce-mollusks
  ]
  ask round_gobys [
    avoid-obstacles
    movefast
    set energy energy - 0.5
    eat round_gobys-food-filter
    death
    reproduce-round_gobys
     ]
  
  ask top_preds [
    avoid-obstacles
    movefaster
    set energy energy - 2
    eat top_preds-food-filter
    death
    reproduce-top_preds
      ]
  
  ask burbot [
    avoid-obstacles
    movefaster
    set energy energy - 0.5
    eat burbot-food-filter
    death
    reproduce-burbot
  ]

  grow-phytoplankton
  tick
  update-plot
  ]
end

to populate-zooplankton [ census random? ]
  create-zooplankton census
  [
    set color blue
    set size 2  ;; easier to see
    set label-color blue - 2
    ifelse random?
    [ set energy random (2 * zooplankton-gain-from-food) ]
    [ set energy zooplankton-gain-from-food ]
    
     setxy random-xcor random-ycor
   
  ]
end

to populate-mollusks [ census random? ]
  create-mollusks census
  [
    set color brown
    set size 1  ;; easier to see
    ifelse random?
    [ set energy random (2 * mollusk-gain-from-food) ]
    [ set energy mollusk-gain-from-food ]
   
    setxy random-xcor random-ycor
  ]
end

to populate-oshrimp [ census random? ]
  create-oshrimp census
  [
    set color black
    set size 2  ;; easier to see
    ifelse random?
    [ set energy random (2 * a-oshrimp-gain-from-food) ]
    [ set energy a-oshrimp-gain-from-food ]
   
    setxy random-xcor random-ycor
    ask oshrimp with [ pcolor = black ] [ fd 15 ]
  ]
end



to populate-burbot [ census random? ]
  create-burbot census
  [
    set color black
    set size 2  ;; easier to see
    ifelse random?
    [ set energy random (2 * a-burbot-gain-from-food) ]
    [ set energy a-burbot-gain-from-food ]
   
    setxy random-xcor random-ycor
    ask burbot with [ pcolor = black ] [ fd 15 ]
  ]
end

to populate-round_gobys [ census random?]
  create-round_gobys census
  [
    set color red
    set size 2  ;; easier to see
    ifelse random?
    [ set energy random (2 * round_goby-gain-from-food) ]
    [ set energy round_goby-gain-from-food ]
    
    setxy random-xcor random-ycor 
    ask round_gobys with [ pcolor = black ] [ fd 15 ]
    
  ]
end
to populate-top_preds [ census random?]
  create-top_preds census
  [
    set invaded? true
    set color yellow - 1
    set size 3  ;; easier to see
    ifelse random?
    [ set energy random (2 * top_pred-gain-from-food) ]
    [ set energy top_pred-gain-from-food ]
    
    setxy random-xcor random-ycor  
    ask top_preds with [ pcolor = black ] [ fd 15 ] 
  ]
end

to avoid-obstacles
    if (any? patches in-cone 3 120 with [pcolor = black])
    [ rt 90 ] 
end

to make-obstacle
  if mouse-down?
  [ ask patches
    [ if ((abs (pxcor - mouse-xcor)) < 4) and ((abs (pycor - mouse-ycor)) < 4)
      [ set pcolor black ]]]
  display
end

to eraser
  if mouse-down?
  [ ask patches
    [ if ((abs (pxcor - mouse-xcor)) < 2) and ((abs (pycor - mouse-ycor)) < 2)
      [ set pcolor blue + 2 ]]]
  display
end

to revert
  clear-all
  reset-ticks
  ask patches [ set pcolor blue + 2 ]
  end

to move  ;; turtle procedure
  if pcolor = black [fd 10]
  if pcolor != black
  [
  rt random 90
  lt random 90
  fd 0.5
  ]
end

to moveslow  ;; turtle procedure
  rt random 45
  lt random 45
  fd 0.1
end

to movesuperslow  ;; turtle procedure
  rt random 55
  lt random 55
  fd 0.05
end

to movefast 
  if pcolor = black [fd 10]
  if pcolor != black
  [ 
  rt random 35
  lt random 35
  fd 1
  ]
end

to movefaster
  if pcolor = black [fd 10]
  if pcolor != black
  [
  rt random 30
  lt random 30
  fd 2
  ]
end

to eat [ is-food ] ; turtle routine
  if "" != is-food [
    let prey one-of turtles-here with [ run-result is-food ]
    if prey != nobody  and random-normal 10 5 > 2[
        eat-eat prey
    ]
  ]
end

to eat-eat [prey] ; turtle routine
  set energy ( energy + [energy] of prey )
  ask prey [ die ]
end


to reproduce-zooplankton  ;; zooplankton procedure
  if energy > zooplankton-reproduce-energy and random-normal 50 5 < zooplankton-reproduce [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)                ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 fd 1 ]   ;; hatch an offspring and move it forward 1 step
  ]
end

to reproduce-oshrimp  ;; a-oshrimp procedure
  if energy > a-oshrimp-reproduce-energy and random-normal 50 5 < a-oshrimp-reproduce [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)               ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 ]  ;; hatch an offspring and move it forward 1 step
  ]
end

to reproduce-mollusks  ;; mollusk procedure
  if energy > mollusk-reproduce-energy and random-normal 50 5 < mollusk-reproduce [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)               ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 fd 1 ]  ;; hatch an offspring and move it forward 1 step
  ]
end

to reproduce-burbot  ;; a-burbot procedure
  if energy > a-burbot-reproduce-energy and random-normal 50 5 <  a-burbot-reproduce   [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)               ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 fd 2 ]  ;; hatch an offspring and move it forward 1 step
  ]
end

to reproduce-round_gobys  ;; round_goby procedure
  if energy > round_goby-reproduce-energy and random-normal 50 5 <  round_goby-reproduce  [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)               ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 fd 2]  ;; hatch an offspring and move it forward 1 step
  ]
end

to reproduce-top_preds  ;; round_goby procedure
  if energy > top_pred-reproduce-energy and random-normal 50 5 < top_pred-reproduce [  ;; throw "dice" to see if you will reproduce
    set energy (energy / 2)               ;; divide energy between parent and offspring
    
    hatch 1 [ rt random-float 360 fd 1]  ;; hatch an offspring and move it forward 1 step
  ]
end


to death  ;; turtle procedure
  ;; when energy dips below zero, die
  if energy < 0 [ die ]
end

to setup-phytoplankton
  ask patches with [ has-roots ] [
    sprout-phytoplankton 1 [
      init-phytoplankton
    ]
  ]
end

to init-phytoplankton
  set energy energy-from-phytoplankton
  set size 0.6
  set color green
  
end

to grow-phytoplankton
  ask phytoplankton
  [
    ask patch-here [ ask neighbors [
      if ( 0 = count phytoplankton-here ) and  random-normal 50 5 < phytoplankton-regrowth-time [ 
        sprout-phytoplankton 1 [ init-phytoplankton fd 0.5 ]
      ]
    ] ]
  ]
end



to update-plot
  set-current-plot "Population Size"
  set-current-plot-pen "phytoplankton / 4"
  plot count phytoplankton / 4  ;; divide by four to keep it within similar
                       ;; range as a-burbot and zooplankton populations
  set-current-plot-pen "Zooplankton"
  plot count zooplankton 
  set-current-plot-pen "Oshrimp"
  plot count oshrimp
  set-current-plot-pen "Mollusks"
  plot count mollusks
  set-current-plot-pen "Burbots"
  plot count burbot
  set-current-plot-pen "Round Gobys"
  plot count round_gobys
  set-current-plot-pen "Top Preds"
  plot count top_preds
end


; ===============================================
; ====== food web parser functions ==============
; ===============================================

to parse-food-web
  let web_description to-lower-string Food-Web
  let temp ""
  while [ not empty? web_description ]
  [
    ifelse member? "\n" web_description
    [
      set temp substring web_description 0 ( position "\n" web_description )
      update_filter temp
      set web_description remove word temp "\n" web_description
    ]
    [
      update_filter web_description
      set web_description ""
    ]
  ]
  if not invalid-food-web [
    if not empty? burbot-food-filter [
      set burbot-food-filter word "(" word substring burbot-food-filter 4 length burbot-food-filter ")"
    ]
    if not empty? top_preds-food-filter [
      set top_preds-food-filter word "(" word substring top_preds-food-filter 4 length top_preds-food-filter ")"
    ]
    if not empty? round_gobys-food-filter [
      set round_gobys-food-filter word "(" word substring round_gobys-food-filter 4 length round_gobys-food-filter ")"
    ]
    if not empty? zooplankton-food-filter [
      set zooplankton-food-filter word "(" word substring zooplankton-food-filter 4 length zooplankton-food-filter ")"
    ]
    if not empty? oshrimp-food-filter [
      set oshrimp-food-filter word "(" word substring oshrimp-food-filter 4 length oshrimp-food-filter ")"
    ]
    if not empty? mollusks-food-filter [
      set mollusks-food-filter word "(" word substring mollusks-food-filter 4 length mollusks-food-filter ")"
    ]
  ]
end

to update_filter [ update ]
  ;; this function could be improved by testing
  ;; update is of the form member? valid-species "eat/eats" valid-species
  ;; and then throw out impossibilites of cannibalism or phytoplankton predators

  let bb [ "burbot eat " "burbots eat " "burbot eats " ]
  let cf  [ "zooplankton eat " "zooplankton eats " ]
  let os  [ "oshrimp eat " "oshrimp eats " ]
  let mf  [ "mollusks eat " "mollusk eat " "mollusk eats " ]
  let rgf [ "round gobys eat " "round goby eat " "round goby eats " ]
  let tpf [ "top preds eat " "top pred eat " "top pred eats " ]
  let pf  [ "phytoplankton eat " "phytoplankton eats " ]
  
  let valid-species [ "phytoplankton" "zooplankton" "round gobys" "top preds" "mollusks"  "oshrimp" "burbot" ]

  let menu_item ""
  let match ""
  
  ifelse not predator-starts-string update valid-species [
    user-message word "The predator at line '" (word update "'. is misspelled") set invalid-food-web true 
  ]
  [
    set match multimatch bb update
    if match != ""
    [
      set burbot-food-filter check_web_item match update "burbot" valid-species burbot-food-filter
    ]
    set match multimatch rgf update
    if match != ""
    [
      set round_gobys-food-filter check_web_item match update "round gobys" valid-species round_gobys-food-filter
    ]
    set match multimatch tpf update
    if match != ""
    [
      set top_preds-food-filter check_web_item match update "top preds" valid-species top_preds-food-filter
    ]
    set match multimatch cf update
    if match != ""
    [
      set zooplankton-food-filter check_web_item match update "zooplankton" valid-species zooplankton-food-filter
    ]
    set match multimatch os update
    if match != ""
    [
      set oshrimp-food-filter check_web_item match update "oshrimp" valid-species oshrimp-food-filter
    ]
    set match multimatch mf update
    if match != ""
    [
      set mollusks-food-filter check_web_item match update "mollusks" valid-species mollusks-food-filter
    ]
    
    if "" != multimatch pf update
    [
      user-message "Biologically impossible. Please edit your food web."
      set invalid-food-web true
    ]
  ]
end

to-report predator-starts-string [ s valid-species ]
  let length-of-longest-name length last sort-by [ length ?1 < length ?2 ] valid-species
  let test ""
  foreach n-values length-of-longest-name [item ? s] [ 
    set test word test ?
    if member? test valid-species [ report true ]
  ]
  report false
end

to-report check_web_item [ match update predator valid-targets food-filter ]
    let menu_item remove match update
    if menu_item = predator
      [ user-message "No cannibalism. Please edit your food web." set invalid-food-web true ]
    if not member? menu_item valid-targets
      [ user-message word "The prey at line '" (word update "' is misspelled.") set invalid-food-web true ]
    ifelse not invalid-food-web
      [ set menu_item remove_spaces menu_item 
        report word food-filter word " or breed = " menu_item ]
      [ report food-filter ]
end

; ======================================
; ====== String functions ==============
; ======================================
 
to-report multimatch [ testlist target ]
  let match ""
  set match filter [member? ? target] testlist
  ifelse not empty? match 
  [ report first match ]
  [ report "" ]
end

to-report to-lower-string [s]
let lower ""
foreach n-values length s [item ? s] [ set lower word lower to-lower ? ]
report lower
end

to-report to-lower [c] ; c is single character string
let i position c "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
report ifelse-value (is-number? i) [item i
"abcdefghijklmnopqrstuvwxyz"][c]
end

to-report remove_spaces [string]
  while [ member? " " string ]
  [
    set string replace-item (position " " string) string "_"
  ]
  report string
end

; Copyright 2010 Northwestern University's Office of STEM Eeducation Partnerships
; based on code by Uri Wilenksy (c) 1997
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
520
10
1094
605
30
30
9.25
1
14
1
1
1
0
1
1
1
-30
30
-30
30
1
1
0
ticks
25.0

BUTTON
10
90
205
123
Insert/ Reset Species
initialize
NIL
1
T
OBSERVER
NIL
3
NIL
NIL
1

BUTTON
10
135
90
168
Go/Stop
go
T
1
T
OBSERVER
NIL
4
NIL
NIL
1

PLOT
10
450
505
740
Population Size
Time
Population
0.0
100.0
0.0
100.0
true
true
"" ""
PENS
"Phytoplankton / 4" 1.0 0 -10899396 true "" ""
"Zooplankton" 1.0 0 -14070903 true "" ""
"Round Gobys" 1.0 0 -6759204 true "" ""
"Burbots" 1.0 0 -16777216 true "" ""
"Top Preds" 1.0 0 -987046 true "" ""
"Oshrimp" 1.0 0 -612749 true "" ""
"Mollusks" 1.0 0 -2570826 true "" ""

INPUTBOX
10
230
505
440
Food-Web
zooplankton eat phytoplankton\noshrimp eat zooplankton\noshrimp eat phytoplankton\nmollusks eat phytoplankton\nround gobys eat mollusks\nround gobys eat oshrimp\nburbot eat mollusks\nburbot eat round gobys\ntop preds eat burbot\ntop preds eat round gobys\n
1
1
String

BUTTON
230
135
440
170
Start Invasion
populate-top_preds top_preds-start-amount true\nset ignore-top_preds false
NIL
1
T
OBSERVER
NIL
5
NIL
NIL
1

TEXTBOX
10
205
405
223
You may edit the food web below. See \"Information\" tab for details.
12
103.0
1

BUTTON
105
135
205
168
Go One Step
go
NIL
1
T
OBSERVER
NIL
O
NIL
NIL
1

MONITOR
1110
10
1240
55
Top Predators
count top_preds
17
1
11

MONITOR
1190
60
1240
105
Burbot
count burbot
17
1
11

MONITOR
1110
60
1185
105
Round Gobys
count round_gobys
17
1
11

MONITOR
1110
210
1240
255
Phytoplankton
count phytoplankton
17
1
11

MONITOR
1110
160
1240
205
Zooplankton
count zooplankton
17
1
11

MONITOR
1110
110
1185
155
Opossum Shrimp
count oshrimp
17
1
11

MONITOR
1190
110
1240
155
Mollusks
count mollusks
17
1
11

BUTTON
10
50
102
83
Draw Rocks
make-obstacle
T
1
T
OBSERVER
NIL
2
NIL
NIL
1

BUTTON
110
50
207
83
Erase Rocks
eraser
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
10
10
205
43
Initialize Environment
revert
NIL
1
T
OBSERVER
NIL
1
NIL
NIL
1

TEXTBOX
230
10
450
106
Workflow:\n1. Initialize the environment\n2. Draw some rocks\n3. Inserts/ Reset Species\n4. Run the model (Go/Stop)\n5. Insert New Top Predator (Start Invasion)
11
0.0
1

@#$#@#$#@
## LAKE MICHIGAN FOOD WEB

This model decribes a part of the Lake Michigan Food web as found on
 http://www.glerl.noaa.gov/pubs/brochures/foodweb/LMfoodweb.pdf
The original model as found in the COPYRIGHT NOTICE at the bottom of this page was extended and new species were implemented to rebuild more complex relationships between different trophic levels. Another editional feature is the possibility to draw rocks where smaller species can hide an therefore cannot be eaten by predators

## HOW IT WORKS

Species (except phytoplankton) wander randomly around the bay, and what they eat is determined by the Food-Web box. Moving costs energy, and they must eat in order to replenish their energy. When a creature runs out of energy it dies. To allow the population to continue, each species has a fixed probability of reproducing at each time step. Phytoplankton is growing autonomously regardless of other environmental conditions. 


## HOW TO USE IT

1. Edit the food web by pressing the CHANGE button.  
2. Enter the new food chain you want added to the food web.  
3. Press the APPLY button.  
4. Press the OK button.  
5. Press the SETUP button.  
6. Press the GO button to begin the simulation.  
7. Look at the POPULATIONS plot to watch the populations fluctuate over time  

## BUTTONS

1. INITIALIZE ENVIRONMENT - Initialize the Lake Patches
2. DRAW ROCKS - holding the left mouse button one can draw rocks where smaller species can hide
3. ERASE ROCKS - holding the left mouse button one can partially delete rocks
4. INSERT/ RESET SPECIES - all populations are set to their initial values
5. GO/ STOP - starts/ stops the simulation
6. START INVASION - inserts new top predator species

## EDITING THE FOOD WEB

You may edit the food web to create new predator-prey relationships.

Valid species are:  
Phytoplankton
Zooplankton 
Opossum Shrimp
Mollusks 
Round Gobys
Burbots
Top Predators


Example:  
"zooplankton eat phytoplankton"



## CREDITS AND REFERENCES

Wilensky, U. (1997).  NetLogo Wolf Sheep Predation model.  http://ccl.northwestern.edu/netlogo/models/WolfSheepPredation.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.


## COPYRIGHT NOTICE

Copyright 2010 Office of STEM Education Partnerships, Northwestern University. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:  
a) this copyright notice is included.  
b) this model will not be redistributed for profit without permission from the Office of STEM Education Partnerships [OSEP]. Contact OSEP for appropriate licenses for redistribution for profit.

This model was created as part of the project: ECOLOGICAL FORECASTING: FRAMEWORK TO EVALUATE THE EFFECTS OF MULTIPLE STRESSES IN LAKE MICHIGAN FOOD WEBS AND GUIDE REMEDIATION.  The project gratefully acknowledges the support of the National Oceanic and Atmospheric Administration -- grant number NA09NMF4630406.

This model extends and modifies code from the model "Wolf Sheep Predation" - Copyright 1997 Uri Wilensky.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

big fish
true
15
Rectangle -1 true true 120 15 135 60
Rectangle -1 true true 150 30 165 75
Polygon -1 true true 131 256 87 279 86 285 120 300 150 285 180 300 214 287 212 280 166 255
Polygon -1 true true 30 165 62 157 88 169 99 154 90 75 60 105
Polygon -1184463 true false 136 270 77 149 90 30 105 15 120 15 150 75 165 30 180 75 165 135 165 165 166 270
Circle -16777216 true false 106 55 30
Polygon -1 true true 150 165 165 180 180 195 195 240 150 225

black fish
true
11
Polygon -1 true false 131 256 87 279 86 285 120 300 150 285 180 300 214 287 212 280 166 255
Polygon -1 true false 195 165 235 181 218 205 210 224 204 254 165 240
Polygon -1 true false 45 225 77 217 103 229 114 214 78 134 60 165
Polygon -8630108 true true 136 270 77 149 81 74 119 20 146 8 160 8 170 13 195 30 210 105 212 149 166 270
Circle -7500403 true false 106 55 30

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

crayfish
true
0
Polygon -7500403 true true 120 105 180 105 195 135 195 195 180 270 195 285 105 285 120 270 105 195 105 135
Polygon -7500403 true true 195 150 240 105 240 60 210 30 225 60 195 45 210 75 225 75 225 105 180 135 195 150
Polygon -7500403 true true 105 150 60 105 60 60 90 30 75 60 105 45 90 75 75 75 75 105 120 135 105 150
Polygon -7500403 true true 195 180 255 180 270 195 195 195
Polygon -7500403 true true 180 210 240 210 255 225 180 225
Polygon -7500403 true true 165 240 225 240 240 255 165 255
Polygon -7500403 true true 135 240 75 240 60 255 135 255
Polygon -7500403 true true 120 210 60 210 45 225 120 225
Polygon -7500403 true true 105 180 45 180 30 195 105 195
Circle -1 true false 105 90 30
Circle -1 true false 165 90 30
Polygon -16777216 true false 120 90 105 105 120 105
Polygon -16777216 true false 180 90 180 105 195 105

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

mollusk
true
0
Polygon -6459832 true false 150 0 105 15 60 75 45 150 45 180 60 225 90 270 135 300 165 300 210 285 240 240 270 180 270 135 255 105 210 45
Polygon -16777216 true false 135 15 90 75 75 150 90 240 135 285 105 180 105 105 120 60

oshrimp
true
0
Polygon -955883 true false 45 45 90 15 135 30 180 45 240 105 255 180 240 210 270 225 255 240 195 210 180 165 195 135 180 120 150 135 135 150 105 180 150 120 105 165 135 105 105 90 60 120 90 90 60 105 90 75 45 75 30 105 45 60 45 45
Circle -16777216 true false 75 30 30

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

phytoplankton
true
0
Circle -10899396 true false 54 54 192
Polygon -10899396 true false 60 180 15 165 60 150 30 135 60 135 30 90 75 105 60 60 90 75 90 30 120 60 135 30 150 60 180 15 180 60 225 45 210 90 270 75 225 120 285 120 240 150 285 165 225 180 255 240 210 210 225 255 180 225 180 285 150 240 120 285 120 225 75 255 90 225 45 225 75 195 15 195 60 180
Circle -6459832 true false 99 159 42
Circle -6459832 true false 71 87 67
Circle -6459832 true false 165 165 30
Circle -6459832 true false 144 114 42
Circle -6459832 true false 189 129 42
Circle -6459832 true false 135 210 30
Circle -6459832 true false 135 75 30

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

roundgoby
true
15
Polygon -1 true true 75 60 45 75 30 90 45 105 60 120 75 120 75 60
Polygon -1 true true 135 150 165 165 165 240 135 195 135 150
Polygon -1 true true 90 135 60 135 60 180 105 255 135 240 90 135
Polygon -1 true true 120 90 150 135 180 120 180 90 135 75 120 90
Polygon -11221820 true false 120 270 75 165 75 45 135 0 165 30 165 90 150 150 165 270 195 285 120 300 120 270
Polygon -1 true true 135 75 180 45 210 75 135 90 135 75
Circle -16777216 true false 90 45 30

sheep
false
15
Rectangle -1 true true 166 225 195 285
Rectangle -1 true true 62 225 90 285
Rectangle -1 true true 30 75 210 225
Circle -1 true true 135 75 150
Circle -7500403 true false 180 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

stone
true
14
Polygon -7500403 true false 165 45 135 30 105 15 75 45 60 45 45 90 15 120 30 180 15 195 15 225 45 240 90 255 90 285 150 285 180 270 225 270 255 225 255 165 285 165 300 120 285 30 210 0 105 15

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Rectangle -7500403 true true 195 106 285 150
Rectangle -7500403 true true 195 90 255 105
Polygon -7500403 true true 240 90 217 44 196 90
Polygon -16777216 true false 234 89 218 59 203 89
Rectangle -1 true false 240 93 252 105
Rectangle -16777216 true false 242 96 249 104
Rectangle -16777216 true false 241 125 285 139
Polygon -1 true false 285 125 277 138 269 125
Polygon -1 true false 269 140 262 125 256 140
Rectangle -7500403 true true 45 120 195 195
Rectangle -7500403 true true 45 114 185 120
Rectangle -7500403 true true 165 195 180 270
Rectangle -7500403 true true 60 195 75 270
Polygon -7500403 true true 45 105 15 30 15 75 45 150 60 120

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

zooplankton
true
0
Polygon -13345367 true false 30 60 75 75 150 90 225 75 255 45 225 90 165 105 180 135 180 195 165 240 150 255 180 285 120 285 150 255 135 240 120 195 120 135 135 105 75 90 30 60
Circle -2674135 true false 135 135 30

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
setup
set grass? true
repeat 75 [ go ]
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="100" runMetricsEveryStep="false">
    <setup>initialize</setup>
    <go>go</go>
    <exitCondition>distinction = 1</exitCondition>
    <metric>count phytoplankton</metric>
    <metric>count zooplankton</metric>
    <metric>count oshrimp</metric>
    <metric>count mollusks</metric>
    <metric>count round_gobys</metric>
    <metric>count burbot</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
