breed [particles particle] ; turtles that represent planets, celestial bodies
breed [plotpoints plotpoint] ; turtles used to draw orbital paths
particles-own [ x-velocity y-velocity z-velocity ; used to store the 3-dimensional velocity of the particle
   x-pos y-pos z-pos ; used to store the 3d position (separate from actual xcor/ycor/zcor to allow the world to zoom and change perspective
   mass ; represents physical mass. Higher mass means greater gravitational pull
   density ; turtle size = mass / density (I wanted particles with greater mass to be larger, but didn't want something extremely massive to completely dominate the screen)
   identifier ; gives a simple way for the user to select specific planets
   copy? ; if true, the particle is a hidden copy used only for calculating
   ]
plotpoints-own [ number ; when an orbital path is plotted, the points are numbered for easy link creation
   x-pos y-pos z-pos ; same as particles - these are separate from actual x/y/zcor for zooming purposes
   ]
globals [ zoom-factor ; used to effectively zoom the world in and out; just scales positions and sizes accordingly
   max-plotpoint ; stores the next plotpoint number
   ]

to setup
  clear-all
  set-default-shape turtles "circle"
  set zoom-factor 1
  create-particles 1 [
    setxyz 0 0 0
    set x-velocity 0
    set y-velocity 0
    set z-velocity 0
    set mass 1000
    set density 1
    set size mass ^ (1 / 3) / density
    set x-pos xcor
    set y-pos ycor
    set z-pos zcor
    set color yellow
    set identifier "sun"
    set copy? false
    ]
  create-particles 1 [
    setxyz -20 0 0
    set x-velocity 0
    set y-velocity 1
    set z-velocity 0
    set mass 10
    set density 1
    set size mass ^ (1 / 3) / density
    set x-pos -40
    set y-pos ycor
    set z-pos zcor
    set identifier 1
    set copy? false
    set color red
    ]
  create-particles 1 [
    ;setxyz 45 0 0
    set x-velocity 0
    set y-velocity -.7
    set z-velocity 0
    set mass 40
    set density 1
    set size mass ^ (1 / 3) / density
    set x-pos 90
    set y-pos 0
    set z-pos 0
    set identifier 2
    set copy? false
    set color green
  ]
  create-particles 1 [
    ;setxyz 0 40 0
    set x-velocity .75
    set y-velocity 0
    set z-velocity 0
    set mass 10
    set density 1
    set size mass ^ (1 / 3) / density
    set x-pos 0
    set y-pos 80
    set z-pos 0
    set identifier 3
    set copy? false
    set color blue
  ]
  set-zoom 3
  set-visible-positions particles
  reset-ticks
end

to go
  compute-acceleration-changes particles
  check-boundaries
  move particles
  check-slide
  tick
end

to tick-once
  compute-acceleration-changes particles
  check-boundaries
  move particles
  check-slide
  tick
end

; checks center of mass of entire system, shifts positions accordingly to keep view centered
to check-slide
  let CoM center-of-mass particles
  slide-view (item 0 CoM) (item 1 CoM) (item 2 CoM)
end

; calculates center of mass of a turtleset
to-report center-of-mass [turtleset]
  ; R = 1/M sigma (mi * ri)
  let M 0
  let totalx 0
  let totaly 0
  let totalz 0
  ask turtleset [
    let mi [mass] of self
    set M M + mi
    set totalx totalx + [x-pos] of self * mi
    set totaly totaly + [y-pos] of self * mi
    set totalz totalz + [z-pos] of self * mi
  ]

  report (list (totalx / M) (totaly / M) (totalz / M))

end

; checks position of turtles relative to boundaries of world, zooms in/out if needed
to check-boundaries
  let flag false
  ask particles [
    if abs (x-pos + x-velocity) > max-pxcor * zoom-factor * .8 or
    abs (y-pos + y-velocity) > max-pycor * zoom-factor * .8 or
    abs (z-pos + z-velocity) > max-pzcor * zoom-factor * .8
    [
      set flag true
    ]
  ]

  if flag [
    print "zooming out"
    set-zoom 1.02
  ]

  set flag false
  ask particles [
    if abs (x-pos + x-velocity) > max-pxcor * zoom-factor * .5 or
    abs (y-pos + y-velocity) > max-pycor * zoom-factor * .5 or
    abs (z-pos + z-velocity) > max-pzcor * zoom-factor * .5
    [
      set flag true
    ]
  ]
  if (not flag) [
    ;print "zooming in"
    ;set-zoom .98
  ]

end

; top-level function to compute changes in acceleration due to gravitational force
to compute-acceleration-changes [turtleset]
  ask turtleset [
    let turtle1 self
    ask other turtleset [
      gravitationally-affect turtle1 self
    ]
  ]
end

; move all turtles based on their velocity
to move [turtleset]
  ask turtleset [
    set x-pos x-pos + x-velocity
    set y-pos y-pos + y-velocity
    set z-pos z-pos + z-velocity

    if not hidden? [
      set xcor x-pos / zoom-factor
      set ycor y-pos / zoom-factor
      set zcor z-pos / zoom-factor
    ]
  ]
end

to set-visible-positions [turtleset]
  ask turtleset [
    if not hidden? [
      set xcor x-pos / zoom-factor
      set ycor y-pos / zoom-factor
      set zcor z-pos / zoom-factor
    ]
  ]
end

; given two particles, calculate the gravitational pull of one upon the other
to gravitationally-affect [ affected affecting ]
  ; g = m1 * m2 / r^2
  ; r = sqrt(xdist^2 + ydist^2) -> r^2 = xdist^2 + ydist^2
  let xdist [x-pos] of affecting - [x-pos] of affected
  ;print xdist
  let ydist [y-pos] of affecting - [y-pos] of affected
  ;print ydist
  let zdist [z-pos] of affecting - [z-pos] of affected
  let total-acc 0
  let xyangle 90
  let zangle 90
  let xydist (xdist ^ 2 + ydist ^ 2) ^ .5
  if abs(xdist) + abs(ydist) + abs(zdist) > 0 [
    set total-acc [mass] of affecting / max (list (xdist ^ 2 + ydist ^ 2 + zdist ^ 2) ((([size] of affected + [size] of affecting) / 2) ^ 2))
    let angle-heading atan xdist ydist
    let z-angle-heading atan xydist zdist
    set xyangle (90 - angle-heading) mod 360
    set zangle (90 - z-angle-heading) mod 360
  ]

  if total-acc > 50 [ set total-acc 50 ]

  set total-acc total-acc / 20
  let xyacc total-acc * (cos zangle)

  ask affected [
    set x-velocity (x-velocity + xyacc * (cos xyangle))
    set y-velocity (y-velocity + xyacc * (sin xyangle))
    set z-velocity (z-velocity + total-acc * (sin zangle))
  ]

end

; shifts positions of particles and plotpoints (effectively sliding "camera")
to slide-view [deltax deltay deltaz]
  ask turtles [
    set x-pos x-pos - deltax
    set y-pos y-pos - deltay
    set z-pos z-pos - deltaz
  ]
  set-visible-positions plotpoints
end

; changes zoom factor, effectively zooming "camera" in or out
to set-zoom [new-zoom]
  set zoom-factor zoom-factor * new-zoom
  print zoom-factor
  ask particles [set size mass ^ (1 / 3) / density / zoom-factor]
  set-visible-positions plotpoints
end

;; reports whichever turtle is selected in the "selected-body" dropdown menu
to-report selected-turtle
  let selected-set particles with [identifier = selected-body and not copy?]
  ;ifelse length selected-set = 1
  ;[
    report one-of selected-set
  ;]
  ;[ ; something is wrong
  ;  print "error selecting set"
  ;  report 0
  ;]
end

;; Plots the projected path of the currently selected body
;; It does this by spawning hidden copies of every turtle, running physics on only the copies,
;; and hatching plotpoints every "tick"
to plot-orbital-path
  clear-orbital-path
  let selected selected-turtle
  if selected != 0 [
    ; spawn copies of all particles
    ask particles [
      hatch 1 [ set copy? true
        set hidden? true
      ]
    ]
    set max-plotpoint 0
    let temp path-plot-length
    while [temp > 0] [
      tick-copies
      set temp temp - 1
    ]

    ;delete copies (this is important)
    ask particles with [copy?] [die]
  ]
end

;; Clears currently plotted path
to clear-orbital-path
  ask plotpoints [die]
end

;; runs through physics only with copies
to tick-copies
  let copies particles with [copy?]
  compute-acceleration-changes copies
  move copies
  set-orbital-path-point
end

;; puts down a new point on the projected orbital path
to set-orbital-path-point
  ask one-of particles with [copy? and identifier = selected-body]
  [
    hatch-plotpoints 1
    [
      set hidden? false
      set x-pos [x-pos] of myself
      set xcor x-pos / zoom-factor
      set y-pos [y-pos] of myself
      set ycor y-pos / zoom-factor
      set z-pos [z-pos] of myself
      set zcor z-pos / zoom-factor
      set size .1 / zoom-factor
      set number max-plotpoint
      if number > 0 [
        let prev-plotpoint one-of plotpoints with [number = max-plotpoint - 1]
        create-link-with prev-plotpoint
      ]
    ]
  ]
  set max-plotpoint max-plotpoint + 1
end

;; A function to modify the velocity of the selected particle based on the corresponding slider
;; values in the interface.
to modify-velocity
  ask selected-turtle [
    set x-velocity x-velocity + x-velocity-mod
    set y-velocity y-velocity + y-velocity-mod
    set z-velocity z-velocity + z-velocity-mod
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
0
0
1323
1344
50
50
13.0
1
10
1
1
1
0
1
1
1
-50
50
-50
50
-50
50
1
0
1
ticks
30.0

BUTTON
120
20
183
53
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
190
20
271
53
NIL
tick-once
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
415
25
565
70
NIL
zoom-factor
3
1
11

PLOT
380
255
640
425
Kinetic Energy
time
kinetic energy
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"total" 1.0 0 -16777216 true "" "plot sum [mass * (x-velocity ^ 2 + y-velocity ^ 2 + z-velocity ^ 2)] of particles with [not copy?]"
"1" 1.0 0 -2674135 true "" "plot sum [mass * (x-velocity ^ 2 + y-velocity ^ 2 + z-velocity ^ 2)] of particles with [color = red and not copy?]"
"2" 1.0 0 -13840069 true "" "plot sum [mass * (x-velocity ^ 2 + y-velocity ^ 2 + z-velocity ^ 2)] of particles with [color = green and not copy?]"
"3" 1.0 0 -13345367 true "" "plot sum [mass * (x-velocity ^ 2 + y-velocity ^ 2 + z-velocity ^ 2)] of particles with [color = blue and not copy?]"

BUTTON
45
20
114
53
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
15
75
155
120
selected-body
selected-body
1 2 3
0

BUTTON
25
130
145
163
NIL
plot-orbital-path
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
25
170
145
203
NIL
clear-orbital-path\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
170
130
350
163
path-plot-length
path-plot-length
0
5000
667
1
1
NIL
HORIZONTAL

MONITOR
20
215
155
260
x-velocity of selected
[x-velocity] of selected-turtle
3
1
11

MONITOR
20
260
155
305
y velocity of selected
[y-velocity] of selected-turtle
3
1
11

MONITOR
20
305
155
350
z velocity of selected
[z-velocity] of selected-turtle
3
1
11

BUTTON
195
185
315
218
NIL
modify-velocity
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
175
220
340
253
x-velocity-mod
x-velocity-mod
-.2
.2
-0.2
.001
1
NIL
HORIZONTAL

SLIDER
175
265
340
298
y-velocity-mod
y-velocity-mod
-.2
.2
-0.024
.001
1
NIL
HORIZONTAL

SLIDER
175
310
340
343
z-velocity-mod
z-velocity-mod
-.2
.2
0
.001
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model is part of the Gravitational Model Suite. It is a modification of the other models in the suite, and allows the user to play around with a simple solar system model. It is meant for users to be able to play around and see how changing the velocity of an object at different points along the orbit changes its orbital path.

## HOW IT WORKS

Every particle has mass, position, and velocity. At every time step, every particle pulls on every other particle with a gravitational force proportional to its mass and inversely proportional to the distance between the two particles. Each particle sums up these forces to calculate its total acceleration at every tick, and updates its velocity accordingly. These rules are simply an implementation of well known Newtonian laws of physics - but some interesting behaviors can result.

## HOW TO USE IT

First click the SETUP button, and then GO starts or stops the model. The SELECTED-BODY dropdown menu allows you to select one of the three bodies orbiting around the "sun". The PLOT-ORBITAL-PATH button plots the calculated path of the body (up to whichever amount of ticks is specified by the PATH-PLOT-LENGTH slider). The MODIFY-VELOCITY button changes the x, y, and z-velocity of the selected body by the amounts specified by the sliders right below it.

## THINGS TO NOTICE

Notice that along an orbit, kinetic energy and potential energy are in opposition. When a particle reaches the highest point in its orbit (i.e., has the most potential energy), it is moving the slowest. When the particle reaches its lowest point, it is moving fastest. Conservation of energy states that a particle's total energy (kinetic + potential) must be constant unless it gains or loses energy by interaction with something else. Therefore, as it is moving through its orbit, it converts kinetic energy to potential energy and vice versa.

This also means that if (for example) a particle gains kinetic energy at the lowest point of its orbit, its highest point will correspondingly have more potential energy, which requires it to be farther away. In simpler terms, speeding up widens the opposite end of the orbit, and slowing down brings the opposite end closer. These are all emergent properties that result naturally from Newtonian laws of gravitation.

Also, a cool side effect of using small colored balls and grey links to plot orbital paths is the fact that when a particle is moving more slowly, the plot points (colored balls) are closer together, and thus the color of the path is stronger. This is very obvious with strongly elliptical paths (which, as discussed above, would result in large changes in velocity along the orbit), where the plotted path's color clearly changes along the orbit in correspondence with the velocity.

## THINGS TO TRY

Try modifying the orbits of bodies to pass through each other - or, for a real challenge, to orbit each other. Make a particle's orbit wider or narrower by modifying its velocity, and then re-circularize.

## EXTENDING THE MODEL

The model currently can only plot one orbital path at a time. Try extending it to support more. Also, some glitches or inaccuracies arise because the model uses discrete time steps - time steps could be made smaller to mitigate this. There also may be more intuitive ways of changing velocity - for example, rather than modifying velocity in terms of x, y, and z (which is significantly easier to program), modifying velocity in terms of prograde (in the direction of motion), radial (outward), and normal (up and down), which would require some more trigonometry to program but would make it very simple to speed up or slow down without changing direction.

## NETLOGO FEATURES

One workaround that was used was "zooming" in and out to compensate for the fact that Netlogo doesn't support resizing the world. So when a particle tries to go outside the bounds of the world, rather than making the world bigger, I just scaled down all the positions. This is why every turtle has a separate x-pos, y-pos and z-pos defined in addition to the default xcor, ycor and zcor; the new -pos variables are not scaled down and are used in all physics calculations, while the default cor variables are modified by the global zoom-factor variable and are used for displaying turtles. As a result, the world can effectively become bigger and smaller as needed.

## RELATED MODELS

This model is part of a suite of models, along with the Basic Gravitation model and the Gravity Wells model.

## CREDITS AND REFERENCES

http://modelingcommons.org/browse/one_model/4699

Wilensky, U. (1999). NetLogo [computer software]. Evanston, IL: Center for Connected Learning and Computer-Based Modeling, Northwestern University.  http://ccl.northwestern.edu/netlogo .
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 3D 6.0-M5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
