;; Model the # of your Ancestors according to the random selection of parents of your Ancestors out of the Parents, by random selection out of:
;;    (a: unPooled) - the totality of Parents, and
;;    (b: Pooled)   - individual parishes selected by their proximity to the parish of the child

patches-own [
  Hits                       ;; #distinct picked Ancestors out of Parents in this parish
  Picks                      ;; #picks out of Parents in this parish (resulting from the scattering of 2*Hits picks from across all Parishes in the children's Gen). Made integral when applied in SetPicks.
]

globals [
  ;; Interface Inputs:
  ;#Gens                     ;; # Gens to trace back
  ;%MarriagesInfertile       ;; % of marriages that are childless. Default = 10% (just a guess)
  ;Trace                     ;; Switch to trace monitors for each Gen

  GenYears                   ;; Generation interval of 30 years.  The average age at childbirth.
  Gen                        ;; generation (counting backwards, starting with 0 for the current Gen if you were born after 1966, else starting at 1)
  GenStartYearList           ;; List by Gen of start year of each birth generation. This is made negative so that it increases from left-right on the plot x-axis (to get around a limitation of NetLogo)
  Marriages                  ;; # marriages in a Gen in E&W
  MarriagesList              ;; List by Gen of #marriages
  Parents                    ;; # Parents in a Gen  (note: calculated from Marriages, assuming all parents are married, which was generally true in more-religious times)
  ParentsList                ;; List by Gen of # Parents
  NonExtincts                ;; # non-extinct Parents in a Gen, i.e. Ancestors in E&W of everyone in E&W today
  NonExtinctsList            ;; List by Gen of NonExtincts
  Immigrant%                 ;; average across all Gens of Immigrant% into E&W (mostly from Ireland and Scotland). Used in SetNonExtinctsList, SetAncestors, SetPicks.
  Ancestors                  ;; # Ancestors in E&W, unPooled
  ImmigratedAncestors        ;; Total Ancestors that immigrated across the Gens (note: this is cumulated across all Gens, and does not include Foreign ancestors of 1st generation immigrants)

  ;; For Pooled (by Parishes and their proximity) only:
  secs                       ;; for timing Pooled runs (mostly for WeightedParishPick)
  Run#                       ;; # of pooled runs
  ImmigratedAncestorsPooled  ;; Total Pooled Ancestors that immigrated across the Gens (similar to unPooled ImmigratedAncestors)
  Parishes                   ;; black and white patches representing Parishes
  #Parishes                  ;; This decreases in each successive earlier Gen by a factor = (decrease in Marriages in that Gen relative to the later Gen)^MarriagesDecreaseExponent
  ParentsPerParish           ;; # Parents averaged across all parishes. Used as the pool size for Picks within each parish
  ProximityParameter         ;; Proximity parameter to fit %ParentHere, #Parishes and their distribution
  %ParentHere                ;; % chance a Parent married in same Parish as the child.  Dependent on World size, #Parishes and ProximityParameter.  Note that it increases in earlier Gens due to Parishes becoming sparser
  TargetOf1                  ;; converges to 1 when iteratively set to (sumParishProximity*%ParentHere/100) in SetProximityParameterBy%ParentHere, because %ParentHere = 100/sumParishProximity
  sumWeights                 ;; sum of Weights within the largest radius (note: not all patches)
  Win-radiusList             ;; sum of Weights within each digital radius
  AncestorsPooled            ;; # Ancestors with pooling in E&W (output only)
  %ParishesWithAncestors     ;; %Parishes with any of Ancestors (output only)
]

to Setup   ;; Called by user and SetupPooled
  print  "***"               ;; to delineate runs
  clear-all
  set GenYears 30            ;; based on maternity/paternity ages

  ;; Date the Gens. Note: we use data for parents' marriage generations in our calculations but we always present birth generations to the user.
  let Year0 1967             ;; Start year of Gen0, the first birth generation. Their corresponding parents' marriage generation is 1961-1990.
  set GenStartYearList n-values #Gens [? -> (GenYears * ?) - Year0]  ;; note: they are negative in order to increase from left to right on the plot x-axis (to overcome a restriction of NetLogo)

  ;; Initialise lists by Gen
  SetMarriagesList           ;; Set list by Gen of Marriages
  set ParentsList            (map [[m r] -> 2 * m *(1 - r / 100) * (1 - %MarriagesInfertile / 100)] MarriagesList Remarriage%List)  ;; 2 * (marriages - remarriages - InfertileMarriages)    Note: ignoring unmarried parents
  set Immigrant%  3          ;; based on average of 4.1% for 1851-1911 and the expectation that it will be less prior to this
  SetNonExtinctsList         ;; Set list by Gen of NonExtincts

  ;; set outputs for your parents' generation, Gen 0 or Gen 1, depending on whether you were BornAfter1966
  set Gen                 ifelse-value BornAfter1966? [0] [1] ;; your birth generation
  set Ancestors           1                                   ;; you
  set ImmigratedAncestors 0
  set Parents             item Gen ParentsList                ;; the # Parents in your parents' marriage Gen.  Only required to plot an initial non-zero point, equal to the next point
  set NonExtincts         item Gen NonExtinctsList            ;; = Parents

  ;; setup plot
  set-plot-x-range (first GenStartYearList)  (last GenStartYearList)
  set-plot-y-range 0 ifelse-value (#Gens > 17) [2000000][80000]
  update-plots                                                ;; plot the initial unPooled points
end

to go  ;; Called by user
  if Gen + 1 >= #Gens [stop] ;; stop after the last Gen
  set Gen Gen + 1

  set Parents             item Gen ParentsList
  set NonExtincts         item Gen NonExtinctsList
  SetAncestors               ;; Set Ancestors, excl. immigrants, and cumulate ImmigratedAncestors
  update-plots

  if Trace [                 ;; print monitors in each Gen Trace
    let Ancestors%NonExtincts 100 * Ancestors / Parents
    print (word
      "Gen = "                     substring (word Gen                               "      ") 0 2
      "  Parents = "               substring (word round Parents                     "      ") 0 8
      "  NonExtincts = "           substring (word round NonExtincts                 "      ") 0 8
      "  Ancestors = "             substring (word round Ancestors                   "      ") 0 7
      "  Ancestors%NonExtincts = " substring (word precision Ancestors%NonExtincts 2 "      ") 0 5
  )]
end

to SetAncestors ;; Set Ancestors, excl. immigrants, and cumulate ImmigratedAncestors.  Called by go and goPooled
  let lPicks              2 * Ancestors
  let ImmigrantAncs       lPicks * Immigrant% / 100
  set ImmigratedAncestors ImmigrantAncs + ImmigratedAncestors ;; Cumulate ImmigratedAncestors for this Gen.  Note: we do not cumulate ancestors of immigrated ancestors (as they are too difficult to estimate and not of much interest)
  let NonImmigrantPicks   lPicks - ImmigrantAncs
  set Parents             item Gen ParentsList
  set Ancestors           ifelse-value (Gen = 1) [2] [Parents * (1 - (1 - 1 / Parents) ^ NonImmigrantPicks)] ;; # distinct Ancestors when making NonImmigrantPicks random picks from a pool of size Parents
end

to SetMarriagesList          ;; Initialise MarriagesList. Used to estimate #Parents in each Gen (ignoring unmarried parents). Called by Setup
  set MarriagesList [        ;; Data copied from "Marriages by Gen.xlsx" for E&W
    11045925  ;; 1961-1990   Gen 0  The generation in which your parents were married, if you were     BornAfter1966
    10765634  ;; 1931-1960   Gen 1  The generation in which your parents were married, if you were not BornAfter1966
    8741190   ;; 1901-       Gen 2
    6402076   ;; 1871-       Gen 3  (for 1881 census)
    4766844   ;; 1841-       Gen 4
    3112089   ;; 1811-       Gen 5
    2211846   ;; 1781-       Gen 6
    1796616   ;; 1751-       Gen 7
    1557547   ;; 1721-       Gen 8
    1292386   ;; 1691-       Gen 9
    1203037   ;; 1661-       Gen 10
    1339099   ;; 1631-       Gen 11
    1235352   ;; 1601-       Gen 12
    1158028   ;; 1571-       Gen 13
    1101507   ;; 1541-       Gen 14
    842659    ;; 1511-       Gen 15
    719918    ;; 1481-       Gen 16
    657588    ;; 1451-       Gen 17
    649670    ;; 1421-       Gen 18 (Black Death pinch point)
    706478    ;; 1391-       Gen 19
    792849    ;; 1361-       Gen 20
    1044416   ;; 1331-       Gen 21
    1372791   ;; 1301-       Gen 22
    1493844   ;; 1271-       Gen 23
    1427854   ;; 1241-       Gen 24
    1296865   ;; 1211-       Gen 25
    1103467   ;; 1181-       Gen 26
    917004    ;; 1151-       Gen 27
    784708    ;; 1121-       Gen 28
    652411    ;; 1091-       Gen 29
    575238    ;; 1061-       Gen 30 (Norman Conquest)
  ]
  ;; pad out list with 500,000 to #Gens
  if #Gens > length MarriagesList [set MarriagesList sentence MarriagesList n-values (#Gens - length MarriagesList) [500000]]
end

to SetNonExtinctsList        ;; Set List by Gen of Ancestors in E&W of everyone in E&W today (or in Gen 1). Called by Setup
  ;; start the list with Gen0 (and Gen1, if BornAfter1966?)
  set NonExtincts            item 0 ParentsList                                    ;; the current Parental population i.e. the assumed current population of midlife generation (age 30-59). Not all ages, as only counting non-removed cousins
  set NonExtinctsList        (list NonExtincts)                                    ;; start the list
  if not BornAfter1966? [                                                          ;; if born before 1966, set first 2 Gens
    set NonExtincts          item 1 ParentsList                                    ;; the current Parental population scaled to Gen1 by the Marriages ratio
    set NonExtinctsList      lput NonExtincts NonExtinctsList                      ;; start the list with Gen0 and Gen1
  ]

  let lGen length            NonExtinctsList                                       ;; 1 or 2, depending on whether BornAfter1966?
  let lImmigrant%            0                                                     ;; Gen1 (or Gen2) Ancestors and NonExtincts are not immigrants, as we assume Brit is not a child of immigrants
   repeat #Gens - lGen
  [ let NonImmigrantPicks    2 * NonExtincts * (1 - lImmigrant% / 100)             ;; parents of nonImmigrant NonExtincts
    set Parents              item lGen ParentsList   ;; # Parents  for this Gen
    set NonExtincts          Parents * (1 - (1 - 1 / Parents) ^ NonImmigrantPicks) ;; # distict picks from Parents
    set NonExtinctsList      lput NonExtincts NonExtinctsList                      ;; append the list
    set lGen lGen + 1
    set lImmigrant%          Immigrant%
  ]
end

to-report Remarriage%List    ;; Report Remarriage% list of %Marriages that are Remarriages. Called by Setup
                             ;; The first 4 Gens are derived from https://www.ons.gov.uk/peoplepopulationandcommunity/birthsdeathsandmarriages/marriagecohabitationandcivilpartnerships/datasets/ageandpreviousmaritalstatusatmarriage

  let %Gen1  8.715339567     ;; Remarriage% of Marriages for 1931-1961
  let %Gen2  7.37811024      ;; Remarriage% of Marriages for 1901-1931
  let %Gen3  8.679078133     ;; Remarriage% of Marriages for 1871-1901
  let %Gen4  9.110858924     ;; Remarriage% of Marriages for 1841-1871

  let %Gen14 30              ;; Remarriage% of Marriages for 1541-1571  (Ref: Wrigley & Schofield, The Population History of England 1541-1871, page 259)
  let increment (%Gen14 - %Gen4)/ 10 ;; linear increment
  report (sentence %Gen1 %Gen1 %Gen2 %Gen3 n-values (length MarriagesList  - 4) [? -> min list 30 (%Gen4 + increment * ?)]) ;; assume it increases to 30%  in Gen14 then stays constant
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Following is for Pooled (and including unPooled data in the monitors and Trace)

to SetupPooled    ;; setup Pooled simulation.  Called by user
  print "BEWARE before you run goPooled! It is a slow simulation which could take up to 1 hour or more to run through all Gens if run online."
  print "You can stop it at any point by clicking the blackened button (though there will be some delay until the Gen completes)."
  if GenYears = 0 [Setup]                            ;; In case Setup has not already been run

  set Gen                 ifelse-value BornAfter1966? [0] [1] ;; your birth generation

  ;; Restart unPooled to run in parallel
  set Ancestors           1                                   ;; you
  set ImmigratedAncestors 0
  set Parents             item Gen ParentsList                ;; the # Parents in your parents' marriage Gen.  Only required to plot an initial non-zero point, equal to the next point
  set NonExtincts         item Gen NonExtinctsList            ;; = Parents

  ;; Restart Pooled, ready to reTrace and draw the next Pooled curve
  set AncestorsPooled           1                             ;; you
  set ImmigratedAncestorsPooled 0

  ;; Setup the world to hold just enough Parishes
  set #Parishes    14051                             ;; assuming that this has hardly changed since 1881. Source: VisionOfBritain data on Parishes in 1881 (ref: #Parishes from VoB.xlsx)
  let xmax         ceiling ((sqrt #Parishes - 1)/ 2)
  resize-world     (- xmax) xmax (- xmax) xmax
  clear-patches                                      ;; for re-runs of Pooled simulation

  ;; Seed the central patch and colour non-parishes green
  ask patch 0 0
  [ set Hits 1                                       ;; you in the Parish you were born
    set pcolor white                                 ;; white is just a visual indicator of the Parishes with Ancestors, it coincides with Hits > 0
  ]
  set Parishes patches                               ;; initialise to all patches before it is reduced to #Parishes
  GreenNonParishes                                   ;; Colour patches green in line with #Parishes and reset Parishes

  ;; Now we have setup the World, we can derive ProximityParameter from %ParentHere
  set %ParentHere 50 ;; 50% in 19th-20th centuries. ref: Migration and Mobility in Britain from the Eighteenth to the Twentieth Centuries, Pooley and Turnbull (1996). http://www.localpopulationstudies.org.uk/pdf/lps57/lps57_1996_50-71.pdf
                     ;; Additional evidence: 50% implies 76% from the same county (see %ParentsInSameCounty procedure) which correlates well with 75% derived from the 1881 census.
  SetupWeighting     ;; set ProximityParameter (dependent on World size), Win-radiusList and sumWeights

  ;; Setup Pooled plot
  create-temporary-plot-pen (word "Pooled" Run#)
  set-plot-pen-color 15 + Run# * 30                  ;; cycle through the base colors, starting at red
  plotxy item Gen GenStartYearList  0                ;; plot the initial point

  set   Run#   Run# + 1                              ;; count the Pooled Runs
  print word "*** Pooled Run " Run#                  ;; delineate Pooled Runs
  set secs 0                                         ;; to time this Pooled run
end

to goPooled   ;; Called by user
  if Gen = 0 [reset-timer]                           ;; zero time at start of the first goPooled
  if Gen + 1 >= #Gens [stop]                         ;; stop after the last Gen
  set Gen Gen + 1
  SetAncestors                                       ;; For unPooled comparison: Set Ancestors, excl immigrants, and cumulate ImmigratedAncestors

  ;; Set #Parishes and GreenNonParishes
  set Marriages                 item Gen MarriagesList
  let ParishesFactor            4498 / 7864          ;; #Parishes in  1560 and for 1820. See #Records retrieved using advanced searches saved in C:\Users\ianjh\NetLogo\Ancestors\theclergydatabase Ref: http://db.theclergydatabase.org.uk/jsp/search/index.jsp
  let MarriagesFactor           1101507 / 3112089    ;; Marriages for 1560 and for 1820
  let MarriagesDecreaseExponent (ln ParishesFactor) / (ln MarriagesFactor) ;; Used to calculate #Parishes decrease
  if Gen > 3 [set #Parishes round((Marriages / item (Gen - 1) MarriagesList)^ MarriagesDecreaseExponent * #Parishes)] ;; Prior to 1871, vary both #Parishes and PotSize. Note: As the exponent > .5, #Parishes decreases a bit faster than PotSize
  let #Whites    count patches with [pcolor = white]
  set #Parishes  max list #Whites #Parishes          ;; since Whites cannot be Greened
  GreenNonParishes                                   ;; Colour black patches green in line with #Parishes

  SetPicks                                           ;; Gravitationally scatter 2 Picks for each Hit in the children's Gen

  ;; Set #distinct Hits from the Picks in all Parishes in this Gen
  set ParentsPerParish    (item Gen ParentsList) / #Parishes
  ask patches with [Picks > 0]
  [ set Hits  (#Hits ParentsPerParish)               ;; Set the Hits from this patch's Picks within pool size ParentsPerParish
    set Picks 0                                      ;; Zero these used Picks, ready for setPicks in the next Gen
  ]

  set %ParishesWithAncestors  100 * count patches with [Hits > 0] / #Parishes
  ask Parishes [set pcolor ifelse-value (Hits = 0)  [black] [white]] ;; recolour patches in line with Hits

  ;; Plot Pooled
  set AncestorsPooled               sum [Hits] of patches
  plotxy item Gen GenStartYearList  AncestorsPooled

  ;; Trace pooled monitors
  Set%ParentHere                                     ;; set for use in Trace and monitor
  if Trace [                                         ;; print monitors in each Gen
    let AncestorsPooled%Ancestors     100 * AncestorsPooled / Ancestors
    print (word
      "Gen = "                         substring (word Gen                                   "     ") 0 2
      "  Ancestors = "                 substring (word round Ancestors                       "     ") 0 6
      "  AncestorsPooled = "           substring (word round AncestorsPooled                 "     ") 0 6
      "  AncestorsPooled%Ancestors = " substring (word precision AncestorsPooled%Ancestors 1 "     ") 0 5
      "  #Parishes = "                 substring (word #Parishes                             "     ") 0 5
      "  %ParentHere = "               substring (word precision %ParentHere  1              "     ") 0 4
      "  ParentsPerParish = "          substring (word round ParentsPerParish                "     ") 0 4
      "  %ParishesWithAncestors = "    substring (word precision %ParishesWithAncestors 2    "     ") 0 5
  )]

  set secs timer                                     ;; update total time for Pooled run
end

to GreenNonParishes      ;; Green patches to match #Parishes, by turning them green. Called by SetupPooled, goPooled and %ParentInSameCounty
  let #ToGreen         count Parishes - #Parishes

  ifelse #ToGreen >= 0
  [ ask n-of    #ToGreen  patches with [pcolor = black] [set pcolor green]] ;; Green  patches when Marriages decrease
  [ ask n-of (- #ToGreen) patches with [pcolor = green] [set pcolor black]] ;; revive patches when Marriages increase

  set Parishes  patches with [pcolor != green]                              ;; patch set of Parishes for speed in WeightedParishPick
  set #Parishes count Parishes
end

to-report #Hits [lPoolSize]  ;; # distinct items picked in this patch's random Picks from lPoolSize. Called by goPooled
  report   lPoolSize * (1 - (1 - 1 / lPoolSize) ^ Picks)                    ;; using a similar formula to the one in SetAncestors
end

to SetPicks ;; set Picks from the Hits of the children's generation (2 gravitationally scattered picks per hit, one for each parent). Called by Go
  let lImmigratedAncestorsPooled  ImmigratedAncestorsPooled                 ;; use a local variable to avoid updating the monitor until the end of this procedure
  ask patches with [Hits  > 0] [
    let lPicks                    2 * Hits                                  ;; Pick 2 parents for each Ancestor in the children's generation
    let ImmigrantAncs             lPicks * Immigrant% / 100                 ;; # your Immigrant Ancestors in this Gen
    set lImmigratedAncestorsPooled ImmigrantAncs + lImmigratedAncestorsPooled ;; Cumulate into lImmigratedAncestorsPooled
    set lPicks                    lPicks - ImmigrantAncs                    ;; # non-immigrant Picks

    ;; Scatter lPicks
    let floorlPicks floor lPicks                                            ;; to be scattered one-by-one
    set picks picks + lPicks - floorlPicks                                  ;; add the fractional part of lPicks into local Picks, i.e. the fractional part is not scattered
    repeat floorlPicks [ask self[Pick]]                                     ;; weighted Picks in Parishes. Must "ask self" in order to establish "myself" in Pick and its sub-procedures

    set Hits 0                                                              ;; Reset used Hits
 ]
  set ImmigratedAncestorsPooled  lImmigratedAncestorsPooled                 ;; now set the monitor
end

to Pick   ;; Pick in a Weighted random Parish.  Called by SetPicks
  ask WeightedRandomParish   [set Picks   Picks + 1]
end

to-report WeightedRandomParish  ;; report a random parish from within a weighted (by proximity) random radius of myself.  Called by Pick
  ;; scan Win-radiusList for a weighted radius containing a random sum of Weights
  let RandomSumOfWeights  random-float sumWeights             ;; a random sum of Weights from the spectrum of 0 to sumWeights
  if RandomSumOfWeights < 1 [report myself]                   ;; fast shortcut for 50% (approx. 100/sumWeights) of Picks, as any radius < 1 contains only "myself"
  let radius 0 while [RandomSumOfWeights > item radius Win-radiusList] [set radius  radius + 1] ;; get the first weighted digital radius containing the random sum of Weights

  report one-of Parishes with [distance myself <= radius]     ;; report a random Parish from within the weighted random radius of myself
end

to SetupWeighting   ;; set ProximityParameter, Win-radiusList and sumWeights,  Can use patch 0 0 as the origin, as these are largely insensitive to choice of origin (i.e. myself).
  SetProximityParameterBy%ParentHere                          ;; set ProximityParameter. Dependent on World size
  set Win-radiusList map [radius -> [sum [Proximity] of patches in-radius radius] of patch 0 0] range (max-pxcor + 1)  ;; sum of Weights within each digital radius
  set sumWeights last Win-radiusList                          ;; it is the total with the largest radius (note: not for all patches)
end

to SetProximityParameterBy%ParentHere  ;; Called by SetupPooled and %ParentInSameCounty
  ;; fit ProximityParameter so that TargetOf1 (dependent on %ParentHere, ProximityParameter, World size and Parish distribution) converges to 1
  set ProximityParameter  5                                   ;; initial estimate for %ParentHere=50%
  setTargetOf1                                                ;; set initial value of TargetOf1
  while [abs(TargetOf1 - 1) > 1E-14] [
    set ProximityParameter  ProximityParameter * TargetOf1    ;; TargetOf1 converges to 1 since TargetOf1 decreases as ProximityParameter increases (and by a smaller ratio)
    if ProximityParameter < .1 [error ProximityParameter]     ;; experience shows TargetOf1 is probably diverging, if ProximityParameter < .1
    setTargetOf1                                              ;; iterate using the updated ProximityParameter
  ]
end

to SetTargetOf1  ;; set TargetOf1 so that %ParentHere=100/sumParishProximity.   Called by SetProximityParameterBy%ParentHere
  let sumParishProximity  [sum [Proximity] of patches in-radius max-pxcor] of patch 0 0
  set TargetOf1           sumParishProximity * %ParentHere / 100  ;; as the Proximity of patch 0 0 = 1
end

to Set%ParentHere ;; % chance a Parent is married in same Parish.  Dependent on World size, #Parishes and ProximityParameter. Called by goPooled
  set %ParentHere 100 / [sum [Proximity] of Parishes in-radius max-pxcor] of patch 0 0  ;; maximum parent distance = max-pxcor (as assumed in SetTargetOf1 when setting ProximityParameter). patch 0 0 is a good origin as it is largely insensitive to this
end

to-report Proximity  ;;  This is the Weight applied to each Pick. It is a measure of the Proximity of the Picked Parish to myself. Called by PooledTrace, report%ParentHere, SetCumProxList and %ParentInSameCounty
  report 1 / (1 + ProximityParameter *(distance myself))^ 2
end

to-report %ParentInSameCounty ;; This user-called utility validates the default 50% for %ParentHere as it reports 76% which tallies well with 75% estimated in "1881 BornInCounty%.xlsx" which is based on data downloaded from "The UK Data Service".  Called by user
  set %ParentHere 50

  ;; Setup World
  set #Parishes    14051                                      ;; assuming that this has hardly changed since 1881. Source: VisionOfBritain data on Parishes in 1881 (ref: #Parishes from VoB.xlsx)
  clear-patches

  ;; Seed the central patch and Green non-parishes
  ask patch 0 0
  [ set Hits 2                                                ;; your Parents
    set pcolor white
  ]
  set Parishes patches
  GreenNonParishes

  SetProximityParameterBy%ParentHere

  let countyRadius               sqrt(count patches / 55 / pi) ;; radius of circular county of average area, out of 55 counties
  let sumParishProximityInCounty [sum [Proximity] of Parishes in-radius countyRadius] of patch 0 0
  let sumParishProximity         [sum [Proximity] of Parishes] of patch 0 0
  report 100 * sumParishProximityInCounty / sumParishProximity
end
@#$#@#$#@
GRAPHICS-WINDOW
270
895
516
1142
-1
-1
2.0
1
20
1
1
1
0
0
0
1
-59
59
-59
59
0
0
0
ticks
30.0

BUTTON
5
210
75
243
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
80
210
155
243
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
60
305
110
350
NIL
Gen
17
1
11

MONITOR
325
260
427
305
AncestorsPooled
round AncestorsPooled
0
1
11

INPUTBOX
5
290
55
350
#Gens
48.0
1
0
Number

MONITOR
325
210
430
255
Ancestors
round Ancestors
0
1
11

SLIDER
255
315
420
348
%MarriagesInfertile
%MarriagesInfertile
0
20
10.0
1
1
NIL
HORIZONTAL

MONITOR
615
260
822
305
Average Parents per parish
round ParentsPerParish
0
1
11

MONITOR
615
210
682
255
#Parishes
precision #Parishes 2
2
1
11

PLOT
10
360
890
890
How many Ancestors by year
Year born from
NIL
0.0
2000000.0
0.0
-761.0
false
true
"" ""
PENS
"Parents" 1.0 0 -2674135 true "" "plotxy item Gen GenStartYearList Parents"
"NonExtincts" 1.0 0 -10899396 true "" "plotxy item Gen GenStartYearList NonExtincts"
"Ancestors" 1.0 0 -16777216 true "" "plotxy item Gen GenStartYearList Ancestors"

MONITOR
435
210
595
255
ImmigratedAncestors
round ImmigratedAncestors
0
1
11

BUTTON
5
250
75
283
NIL
SetupPooled\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
80
250
155
283
NIL
goPooled
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
160
250
235
283
NIL
goPooled
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
160
210
235
243
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
435
260
595
305
ImmigratedAncestorsPooled
round ImmigratedAncestorsPooled
0
1
11

SWITCH
425
315
515
348
Trace
Trace
0
1
-1000

TEXTBOX
570
910
820
1041
TO to the left is a topological mapping of E&W. Each white patch is a parish with one or more ancestors. Black patches are parishes without any ancestors. Green patches are uninhabited space between parishes, which increases as the #parishes decreases in earlier Gens
13
104.0
1

TEXTBOX
5
40
875
190
Switch on \"BornAfter1966?\" if you were.\nThe \"%MarriagesInfertile\" slider doesn't make much difference until pre-Tudor times. \nThe #Gens default goes back to 557AD, the point at which Ancestors have flattened out. 31 Gens would go back to 1067.\nThe \"Trace\" switch controls whether to the trace key values.  \n\nClick \"Setup\" and \"go\" buttons to plot the how many Ancestors you had in each generation.\nOR\nClick \"SetupPooled\" and \"goPooled\" buttons to plot the equivalent when taking pooling by Parish into account. As this is stochastic, AncestorsPooled varies very slightly each time, as can be seen by clicking \"SetupPooled\" and recurring \"goPooled\" again (BEWARE: this gets slow after the first 10 Gens)
13
0.0
1

TEXTBOX
115
10
665
30
To model how many Ancestors you have in England & Wales through History
16
104.0
1

MONITOR
615
310
820
355
% of Parishes with any Ancestors
round %ParishesWithAncestors
0
1
11

MONITOR
540
310
597
355
secs
precision secs 3
3
1
11

MONITOR
685
210
820
255
%ParentFromSameParish
precision %ParentHere 1
1
1
11

SWITCH
125
315
250
348
BornAfter1966?
BornAfter1966?
0
1
-1000

@#$#@#$#@
## Purpose

The purpose of this model is to estimate the number of your ancestors in each previous generation, where "you" are taken to represent a typical person that is indigenous to England and Wales.

For an overview of the reason for this model, how it works and how it advances the state of the art in ancestor estimation, see the accompanying paper to this model here 
https://www.researchgate.net/publication/323757098

## Generations

In order to date Generations we use a generation interval of GenYears, which we assume is 30 years.  This generation length of 30 years is based on an average maternity age of 29 and paternity age of 33, rounded to 30 to fit in with our quinquennial marriage data source. 

The generations are indexed by the global variable Gen, counting backwards through time, starting with Gen0 for the generation born post-1966. Gen 1 is the generation born 1937-1966, and so on. If "You" declare you were born post-1966, then we trace back through the generations starting at Gen0, else we start at Gen1. The reason for starting generations 6 years after the start of a decade is because childbirth occurs 6 years after marriage, on average, and we use marriage data to determine the ceiling on the number of parents in each generation. 

## Estimating the # Ancestors by iteratively doubling lineages and calculating their confluence

We estimate #(short-hand for "the number of") Ancestors by tracing back through the generations. Each Ancestor has exactly 2 parents in its parental generation.  However, as we proceed further back through the generations we inevitably encounter "genealogical coalescence" (aka "pedigree collapse"). The whole point of this model is estimate the distinct coalesced ancestors.  The key to this estimation is the realisation that the # Ancestors in each generation is limited by the # Parents in that generation. 

## Estimating #Parents in each generation

We can assume that the #Parents is approximately double the #Marriages, as there were few unmarried parents in earlier more-religious generations (and their incidence in recent generations is insignificant anyway, as there is almost zero lineage confluence until we get back to the 18th Century).

We then adjust for childless marriages and remarriages, to get:

    #Parents = 2*(# fertile marriages - % of remarriages)


## Formula for #Ancestors based on the # in the children's generation

Clearly,

    #Ancestors in any generation = # distinct parents of Ancestors in the children's generation

Assuming that that these parents are randomly picked from all parents in this generation, we deduce:

    #Ancestors = #Parents*(1-(1-1/#Parents)^Picks)
       where Picks = 2*(#Ancestors in the children's generation)

This is based on a general formula for # distinct random Picks from a finite pool.

## NonExtincts

We also estimate all non-extinct parents in each generation, i.e. all those who have at least one descendant in E&W today. We call these the NonExtincts and calculate them  using a similar formula for the # distinct random picks, as follows:

    #NonExtincts = #Parents*(1-(1-1/#Parents)^Picks) 
       where Picks = 2*(#NonExtincts in the children's generation)

## Immigration

Finally, we adjust for Immigration.  This reduces the Picks in the formula for Ancestors to:

    NonImmigrantPicks = 2*Ancestors*(1-Immigrant%/100)
       where Immigrant% = % of new Immigrants in the children's generation

and reduces the Picks in the formula for #NonExtincts to:

    NonImmigrantPicks = 2*NonExtincts*(1-Immigrant%/100)



# Pooling

So far we have assumed that Picks are entirely random across Parents. However, in reality, Picks are proximity-weighted, as parents are closer than random to their children.  We implement this as an alternative method alongside pure randomisation for comparison.  

This method is labelled "Pooled" in the Interface and is a stochastic simulation, unlike the deterministic calculation of pure randomisation for "unPooled". The "Pooled" simulation gives slightly varying plots, which can be easily seen by repeating the Pooled plots by clicking SetPooled and goPooled again.    

We represent this proximity gravitation in a Netlogo World with patches representing parishes, or the uninhabited green space between parishes. We assume that 50% of parents come from the same parish (based on demographic data for recent generations), and that the other 50% are scattered using proximity-weighted randomisation. The parish of each parent is selected by a proximity-weighted random pick, which is implemented by randomly picking a parish from within a proximity-weighted random radius of the child's parish.

## Patch data

As just described, patches represent Parishes and the uninhabited green space between them. Parishes are assumed to contain a # Parents (= the total # Parents divided equally between the parishes).  

Patches have own-variables Hits and Picks, where:

   * Hits  holds the simulated # distinct Ancestors in that patch, and
   * Picks holds the simulated # non-distinct Parent picks scattered to that patch, 2 for every Hit in the children's Gen.

Color coding of patches:

  * Uninhabited green space between parishes are green 
  * Parishes with no Hits are Black
  * Parishes with some Hits are White


# Data Sources

**GenYears:**   
[International Society of Genetic Genealogy](https://isogg.org/wiki/Generation_length)
Conclusions show GenYears has remained remarkably constant, as far back as the medieval ages.  
Marital ages in [ONS - Marriages in England and  Wales](https://www.ons.gov.uk/peoplepopulationandcommunity/birthsdeathsandmarriages/marriagecohabitationandcivilpartnerships/datasets/marriagesinenglandandwales2013)

**Marriages**:  
"C:\Users\ianjh\NetLogo\Ancestors\Marriages by Gen.xlsx"  
1871-1961: [ONS - Annual UK figures for births, deaths, marriages etc](https://www.ons.gov.uk/peoplepopulationandcommunity/populationandmigration/populationestimates/datasets/vitalstatisticspopulationandhealthreferencetables)  
1541-1871: "CAMPOP PopEsts.EPHFR.xlsx" data extracted from Wrigley et al (1997), 
kindly supplied by The Cambridge Group for the History of Population and Social Structure.

**Remarriage%:**  
1841-1951: [ONS - Age and previous marital status at marriage](https://www.ons.gov.uk/peoplepopulationandcommunity/birthsdeathsandmarriages/marriagecohabitationandcivilpartnerships/datasets/ageandpreviousmaritalstatusatmarriage)  
1541: Wrigley & Schofield, The Population History of England 1541-1871, p 259 

**%MarriagesInfertile:**  
English Population History from Family Reconstitution 1580-1837, Wrigley at al,  
p 384, table 7.11 "Entry sterility: batchelor/spinster completed marriages".

**Immigrant%**:  
1851-1911: [VisionofBritain - Persons Born in the several parts of the UK and elsewhere](http://www.visionofbritain.org.uk/census/table_page.jsp?tab_id=EW1911GEN_M95&show=DB)

**#Parishes:**  
1831-present: "C:\Users\ianjh\NetLogo\Ancestors\VoB\#Parishes from VoB.xlsx"
[VisionofBritain nCube : Total Population](http://www.visionofbritain.org.uk/data/dds_entity_page.jsp?ent=N_TOT_POP)   
1560-1820: [TheClergyDatabase Advanced Search](http://db.theclergydatabase.org.uk/jsp/search/index.jsp)

**%ParentHere**:  
[Migration and Mobility in Britain from the Eighteenth to the Twentieth Centuries, Pooley and Turnbull (1996)](http://www.localpopulationstudies.org.uk/pdf/lps57/lps57_1996_50-71.pdf)

**%ParentInSameCounty**:  
"C:\Users\ianjh\NetLogo\Ancestors\1881 BornInCounty%.xlsx"  
Southall, H.R. and Ell, P. [Great Britain Historical Database: Census Data: Migration Statistics, 1851-1951](http://dx.doi.org/10.5255/UKDA-SN-4558-1)



## HOW TO CITE
If you mention this model in a publication, please attribute it as follows.

<a rel="license" href="http://creativecommons.org/licenses/by-nc/4.0/"><img alt="Creative Commons Licence" style="border-width:0" src="https://i.creativecommons.org/l/by-nc/4.0/88x31.png" /></a><br /><span xmlns:dct="http://purl.org/dc/terms/" property="dct:title">How Many Ancestors Do You Have?</span> by <span xmlns:cc="http://creativecommons.org/ns#" property="cc:attributionName">Ian J Heath</span>
is licensed under a Creative Commons Attribution-NonCommercial 4.0 International License http://creativecommons.org/licenses/by-nc/4.0
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

outline
false
0
Rectangle -2674135 true false 0 -15 315 15
Rectangle -2674135 true false -15 0 15 375
Rectangle -2674135 true false 0 285 360 330
Rectangle -7500403 true true 300 -90 360 300
Rectangle -2674135 true false 285 -60 345 285

outline1
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 129 129 42

outline2
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 69 189 42
Circle -2674135 true false 189 69 42

outline3
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 69 189 42
Circle -2674135 true false 129 129 42
Circle -2674135 true false 189 69 42

outline4
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 69 69 42
Circle -2674135 true false 189 69 42
Circle -2674135 true false 69 189 42
Circle -2674135 true false 189 189 42

outline5
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 69 69 42
Circle -2674135 true false 189 69 42
Circle -2674135 true false 69 189 42
Circle -2674135 true false 189 189 42
Circle -2674135 true false 129 129 42

outline6
false
0
Rectangle -2674135 true false 0 0 30 300
Rectangle -2674135 true false 30 270 360 360
Rectangle -2674135 true false 270 -15 300 270
Rectangle -2674135 true false 15 0 270 30
Circle -2674135 true false 69 69 42
Circle -2674135 true false 189 69 42
Circle -2674135 true false 69 189 42
Circle -2674135 true false 189 189 42
Circle -2674135 true false 69 129 42
Circle -2674135 true false 189 129 42

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
need-to-manually-make-preview-for-this-model
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
