;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;                                             OVERVIEW OF TERMS                                             ;;
;;                                                                                                           ;;
;;   In this model, the 'breeds' or AGENTS are the things you see within the garden interface that can move, ;;
;;     change size, appear, or dissapear.                                                                    ;;
;;   The breeds are collectively called TURTLES, and most of them are living things.                         ;;
;;   The code tells you what to call them in plural or singular terms.                                       ;;
;;   PATCHES are the background squares of the model, in this case, they represent the soil of the garden.   ;;
;;   GLOBALS are variables in the model that are determined by buttons or actions on the interface.          ;;
;;                                                                                                           ;;
;;   Together these three items (TURTLES, PATCHES, GLOBALS) make up all that is defined by the model.        ;;
;;   Everything else in the code defines the PROCEDURES that define the rules of the model                   ;;
;;   PROCEDURES come after the word "to" and are followed by "end"                                           ;;
;;   Some procedures might be defined on the interface page (inside a button) as opposed to the code         ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

breed [farmers farmer]
breed [plants1 plant1]
breed [plants2 plant2]
breed [weeds weed]
breed [drops drop]
breed [fungi fungus]
breed [bees bee]

turtles-own [
  energy            ; the life force of the creatures
  my-heading        ; which direction it is facing
  ]

patches-own [       ; Each patch of soil has these qualities.
  nutrients         ; determines the soil quality
  hydration         ; how much water is in the soil
  ]

globals [
total-harvested     ; The total pounds of produce harvested by the farmer
money               ; Starts off as your initial budget, and is updated based on the actions you take as the farmer.
crop                ; When plants are harvested, they are then called crop.
pesticide-ppm       ; Pesticide ppm = parts per million. This number increases when you add herbicide
]

to setup            ; The actions that happen when you press the setup button
  clear-all
  setup-patches
  setup-turtles
  make-farmer
  reset-ticks
  set money (budget - ((count plants1) + (count plants2)) * plant-cost)
end

to setup-patches
  ask patches [
 set nutrients (random 15)         ; Each patch has a random amount of nutrients between 1 and 15 at first
   set hydration (random 10)]      ; Each patch has a random amount of hydration between 1 and 10 at first
  repeat 5 [diffuse nutrients 1]   ; The diffuse command 'smoothes out' the random numbers, so that a patch with 15 isn't next to a patch with 1
  repeat 2 [diffuse hydration 1]
  ask patches [
   set pcolor scale-color brown nutrients  10 0   ; How dark or light brown the patches are relates to the level of nutrients
  ]
  end


to setup-turtles
  create-weeds 0
  create-drops 0
  create-fungi 0
  create-bees 0
end

to make-farmer
  create-farmers 1 [
    set color cyan             ; Change colors using the color swatch (see 'tips and tricks' sheet)
    set size 3                 ; Change size
    set shape "farmer"         ; You can find more possible shapes online (http://ccl.northwestern.edu/netlogo/docs/shapes.html)
    setxy 0 0
    set heading 90
    set my-heading 270
    run "seed-1"
    run "seed-2"
  ]
end

to seed-1                           ; This procedure sets up all of the first variety of plant
  repeat (num-plants-1)
[
 ask farmers [
  (ask patch-here [
      if not any? plants1-here [    ; No two crops can grow in the same space
        sprout-plants1 1 [
          set shape "plant small"
          set size 0.3
          set color 55
          set energy 2
        ]]])
  fd (spacing-1)
  if not can-move? spacing-1 [
    set heading 0
    fd spacing-1
    set heading my-heading
    ifelse my-heading = 90 [set my-heading 270] [set my-heading 90]
  ]]]
end


to seed-2                           ; This procedure sets up the second variety of plant
  setxy 0 4                         ; They start planting at coordinate (0, 4) in the field (top half)
  set heading 90
  set my-heading 270
  repeat (num-plants-2)
[
 ask farmers [
  (ask patch-here [
      if not any? plants2-here [
        sprout-plants2 1 [
          set shape "plant small2"  ; The shape, size, and color of plant2 are different from plant1
          set size 0.2
          set color 68
          set energy 2
        ]]])
  fd (spacing-2)
  if not can-move? spacing-2 [
    set heading 0
    fd spacing-2
    set heading my-heading
    ifelse my-heading = 90 [set my-heading 270] [set my-heading 90]
  ]]]
end

to re-seed                         ; How you re-plant once the model is running
run "seed-1"
run "seed-2"
set money (money - ((count plants1) + (count plants2)) * plant-cost)
end


to go
      animate               ; The main sequence of procedures for the garden to grow
      harvest-plants        ; This + harvest-weeds define actions you take with the mouse
      harvest-weeds
       tick
end

to animate                                                     ; Happens every 'tick' while the model is running
 if drought [ask patches [set hydration (hydration - 0.1)]]    ; This will happen constantly in drought mode
 if flood [ask patches [set hydration (hydration + 0.2)]]      ; This will happen constantly in flood mode
 ask turtles [                                                 ; All turtles run through these procedures
   run word breed "-grow"
   run word breed "-check-if-dead"
   run word breed "-eat" ]


 if any? patches with [                                 ; Conditions that would enable weeds to grow
   not any? plants1-here and
   not any? plants2-here and
   nutrients > 5 and                                    ; Weeds are more likely to appear in spaces with high nutrient levels
   hydration > 2]
 [
   ask one-of patches with [                            ; The growth is random to one of those patches
     not any? plants1-here and
     not any? plants2-here and
     nutrients > 5 and
     hydration > 2]
   [
    sprout-weeds 1
    [
      set breed weeds
      set shape "flower budding"
      set color lime
      set energy 1
      set size 0.03
    ]


if (count (patches with [any? plants1-here with [size > 3.5]]) > 5)     ; Conditions that determine if bees show up
  or (count (patches with [any? plants2-here with [size > 3]]) > 5)
 [ ask one-of patches with [any? plants1-here with [size > 3.5]]        ; Can you find out what is significant about these size numbers?
   [sprout-bees 1
     [set shape "bee 2"
       set color 4
       set energy 5
       set size 1]
   ]]]

    ]

end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;               The following lines indicate how the particular agents in the model GROW.                   ;;
;;                                                                                                           ;;
;;   Note that some agents don't grow at all.                                                                ;;
;;   You can try changing the RATE that agents grow by increasing or decreasing the number following         ;;
;;     the "set size +/-" command                                                                            ;;
;;   You can also try changing the CONDITIONS that determine when an agent grows by changing the numbers     ;;
;;     before or after the inequalities (< or >)                                                             ;;
;;   How do the two types of plant grow differently from each other?                                         ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to plants1-grow
  if (energy > 3) [set size size + 0.02]
  if (size > 0.8) [set shape "plant medium"]
  if (size > 1.6) [set shape "plant"]
  if (size > 3.5) [set shape "flower 1"]
end

to plants2-grow
  if (energy > 3) [set size size + 0.01]
  if (size > 0.7) [set shape "plant medium 2"]
  if (size > 1.4) [set shape "tomato plant"]
  if (size > 3) [set shape "flower2"]
end

to weeds-grow;;
    if (energy > 2) [set size size + 0.04]     ; How do these numbers compare to the conditions for plant growth?
end

to farmers-grow
end

to drops-grow
end

 to fungi-grow
if (hydration > 7) [set size size + 0.01]
if (hydration < 7) [set size size - 0.2]
 end

 to bees-grow            ; Bees don't actually grow in size in the model, so this shows how they look for food.
rt random 50             ; rt = "right turn", and a random number between 1 and 50
lt random 50             ; lt = "left turn". This positions the bees in a random direction before they move.
fd spacing-1             ; You could change this to any number.
set energy energy - 2.5  ; How much energy they lose each time they fly
 end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;               The following lines indicate how the particular agents in the model EAT.                    ;;
;;  Note that some agents don't have code to eat because it isn't necessary to their function in the model.  ;;
;;  How do the agents interact differently with nutrients and hydration of the soil?                         ;;
;;  What impacts the gain or loss of energy?                                                                 ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to plants1-eat
  if (nutrients >= 1 and (hydration >= 1  and hydration < 10)) ; Conditions that let the plant eat
  [
      set energy energy + 1
      set nutrients (nutrients - 0.2)
      set hydration (hydration - 0.2)
      ask neighbors4 [set nutrients (nutrients - 0.05)        ; "neighbors4" are the four patches with sides directly next to the patch
        set hydration (hydration - 0.05)                      ; How does the impact on the neighbor patches differ from the main patch?
        ask neighbors [set nutrients (nutrients - .02)        ; "neighbors" are the 8 patches adjacent/ in a circle around the patch
        set pcolor scale-color brown nutrients 10 0]          ; The color of the patches updates based on their new nutrient level
      set pcolor scale-color brown nutrients 10 0]]

  if (nutrients < 1  or hydration < 1)                        ; Conditions where the plant is unable to eat
  [
      set energy energy - 0.1]                                ; If no nutrients in the soil, this rate is how fast they will die

  if (hydration > 12) [                                       ; Conditions that lead to fungus also impact soil quality
    set nutrients (nutrients - 2)
 ask neighbors4
 [set nutrients (nutrients - .8)]
 ask neighbors4 [
    sprout-fungi 1
  [
  set shape "mushroom"
   set color white
   set size 0.3]]]
end


;;  This breed of plant (2) is more drought resilient, so it needs less water to be able to survive.      ;;
;;        But, it also grows slower.                                                                      ;;
;;  Try to change some of its properties (numbers written in brown) to make it behave differently.        ;;
;;  What other plants could you try to program based on how they interact with water and the soil?        ;;


to plants2-eat
  if (nutrients >= 1 and (hydration >= -5  and hydration < 15))
  [
      set energy energy + 1
      set nutrients (nutrients - 0.2)
      set hydration (hydration - 0.2)
      ask neighbors4 [set nutrients (nutrients - 0.05)
        set hydration (hydration - 0.05)
        ask neighbors [set nutrients (nutrients - .02)
        set pcolor scale-color brown nutrients 10 0]
      set pcolor scale-color brown nutrients 10 0]]
  if (nutrients < 1  or hydration < 1)
  [
      set energy energy - 0.1]

  if (hydration > 20) [
    set nutrients (nutrients - 2)
 ask neighbors4
 [set nutrients (nutrients - .8)]
 ask neighbors4 [
    sprout-fungi 1
  [
  set shape "mushroom"
   set color white
   set size 0.3]]]
end

to weeds-eat
  if (nutrients >= 1) [                               ; Do weeds have more or less restrictive conditions than other plants?
  set energy energy + 1
  set nutrients (nutrients - 0.1)                     ; How do weeds affect the soil quality, compared to other plants?
   set hydration (hydration - 0.1)
  ask neighbors4 [
     set hydration (hydration - 0.05)
    set nutrients (nutrients - 0.08)
    set pcolor scale-color brown nutrients 10 0]
  set pcolor scale-color brown nutrients 10 0]

  if (nutrients < 1) [                                ; Conditions where the weed is unable to eat
  set energy energy - 0.1]
end

to farmers-eat
end

to drops-eat
end

to fungi-eat
  set nutrients (nutrients - 0.1)
  set pcolor scale-color brown nutrients 10 0
end

to bees-eat                                        ; How could you change this procedure to better represent pollination?
if any? plants1-on neighbors [
 set energy energy + 5                             ; Bees get energy from being near plants
 ask plants1-on neighbors [set energy energy + 3]  ; Bees also give energy to the plants they visit
   ]
if any? plants2-on neighbors [
 set energy energy + 7
 ask plants2-on neighbors [set energy energy + 5]  ; Plant-2 gives the bees more energy
   ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;               The following lines indicate how the particular agents in the model DIE.                    ;;
;;             Note that some agents don't have code to die because they are always in the model.            ;;
;;                                                                                                           ;;                                                                                         ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to plants1-check-if-dead
  if (energy <= 0) [
    set nutrients (nutrients + 0.5)               ; Plants add nutrients to the soil when they die
    set pcolor scale-color brown nutrients 10 0
    die
    ]
end

to plants2-check-if-dead
  if (energy <= 0) [
    set nutrients (nutrients + 0.4)
    set pcolor scale-color brown nutrients 10 0
    die
    ]
end

to weeds-check-if-dead
  if (energy <= 0) [ die ]                       ; Weeds don't produce compost when they die
end

to farmers-check-if-dead
end

to drops-check-if-dead                           ; Even though water isn't technically living, this removes them from the model
  set size size - 0.2
  die
end

to fungi-check-if-dead
if (hydration < 5) [die]
 end

to bees-check-if-dead
  if (energy <= 0) [die]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;            The code below describes how you can harvest plants and weeds from the model                   ;;
;;                                                                                                           ;;
;;   Some things can be harvested, but don't count towards your 'total harvested' button.                    ;;
;;   Do you think this is fair? Which things should or should not be able to be sold for money?              ;;
;;   What else would you want to harvest? What else would you want to do with crops or weeds you pull up?    ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to harvest-plants
  ask farmers [
    set hidden? not mouse-inside?
    setxy mouse-xcor mouse-ycor                 ; Farmer follows your mouse
  ]
  if mouse-inside? and mouse-down? [            ; Mouse-down means you are clicking/ holding click button
    set crop plants1 with                       ; Determining what counts as 'crop'
    [distance one-of farmers < (size / 2) and   ; Closeness to farmer
      size < 3.5]                               ; Size restrictions
    if any? crop [
      let weight sum [size] of crop             ; 'Weight' is a new global being defined here in this sequence ('let' does this)
      set total-harvested (                     ; It uses the size of the plant to determine how to increase your total harvest amount
        total-harvested +
        (ceiling (1.5 * weight * count crop)))  ; 'Ceiling' rounds up to the nearest whole number
ask crop [ die ]
    ]
    let flowers plants1 with [distance one-of farmers < (size / 2) and size > 3.5]
    if any? flowers [
      ask flowers [die]]
  ]



   ask farmers [                                ; Harvesting the second breed of plant
    set hidden? not mouse-inside?
    setxy mouse-xcor mouse-ycor
  ]
  if mouse-inside? and mouse-down? [
    let crop2 plants2 with
    [distance one-of farmers < (size / 2) and
      size < 3.5]
    if any? crop2 [
      let weight sum [size * 1.5] of crop2      ; This crop has a higher value when it is harvested
      set total-harvested (total-harvested +
        (ceiling (1.5 * weight * count crop2)))
ask crop2 [ die ]
    ]
    let flowers2 plants2 with [distance one-of farmers < (size / 2) and size > 3]
    if any? flowers2 [
      ask flowers2 [die]]
  ]
end

to harvest-weeds
  ask farmers [
    set hidden? not mouse-inside?
    setxy mouse-xcor mouse-ycor
  ]
  if mouse-inside? and mouse-down? [
    let weedable weeds with [distance one-of farmers < (size * 2)]
    if any? weedable [

ask weedable [ die ]
    ]
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                                                           ;;
;;             The code below describes the procedures for adding inputs to your garden                      ;;
;;                                                                                                           ;;
;;   Currently it contains compost, water, and herbicide (fungicide is defined inside the button             ;;
;;       on the interface.)                                                                                  ;;
;;   Do you think the way they function is realistic? What else should adding these things do?               ;;
;;   Is there anything else you would want to add to the garden? What about where these things get added?    ;;
;;   Currently these inputs all cost money. Do the prices seem realistic? Should they ever change?           ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to add-compost
  ask patches [                                     ; All patches get compost
    set nutrients (nutrients + 5)
    set pcolor scale-color brown nutrients 10 0 ]   ; The color updates after based on new nutrient level
set money (money - 20)
end


to add-water
if not drought [                                    ; This procedure won't run during drought mode
ask patches [ set hydration (hydration + 5)
  sprout-drops 1
  [ set breed drops
    set shape "drop"
    set color sky
    set size 0.6]
    set pcolor scale-color blue hydration 10 0]     ; The color updates to show you the new hydration level
set money (money - 20)]
end

to add-herbicide
  ask weeds [die]
  ask plants1 [
    set energy energy - 2]
  ask patches [set nutrients (nutrients - 2)
    set pcolor scale-color brown nutrients 10 0 ]
set money (money - 30)
set pesticide-ppm (pesticide-ppm + 1)               ; This keeps track of how many times you apply chemicals
end
@#$#@#$#@
GRAPHICS-WINDOW
577
29
1262
397
-1
-1
37.53
1
10
1
1
1
0
0
0
1
0
17
0
8
1
1
1
ticks
30.0

SLIDER
430
300
564
333
num-plants-1
num-plants-1
0
60
42
1
1
NIL
HORIZONTAL

BUTTON
9
58
107
105
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
8
118
110
171
start / stop
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
236
558
341
598
apply compost
add-compost
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
386
438
740
713
Populations
time
population
0.0
100.0
0.0
80.0
true
true
"" ""
PENS
"Plants1" 1.0 0 -15040220 true "plot count plants1" "plot count plants1"
"Weeds" 1.0 0 -5825686 true "" "plot count weeds"
"Fungi" 1.0 0 -9276814 true "" "plot count fungi"
"Bees" 1.0 0 -4079321 true "" "plot count bees"
"Plants2" 1.0 0 -5509967 true "" "plot count plants2"

MONITOR
110
407
190
452
units of crop harvested
total-harvested
2
1
11

SLIDER
450
340
562
373
spacing-1
spacing-1
1
3
1
1
1
NIL
HORIZONTAL

MONITOR
19
407
101
452
money
money
2
1
11

SLIDER
158
57
280
90
plant-cost
plant-cost
0
10.00
5
.05
1
NIL
HORIZONTAL

SLIDER
288
58
410
91
budget
budget
0
500
500
10
1
NIL
HORIZONTAL

BUTTON
17
558
111
598
re-seed
ask farmers [\nsetxy 0 0 \nset heading 90\nset my-heading 270 \nseed-1\nseed-2]\n\nset money (money - ((count plants1) + (count plants2)) * plant-cost)
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
17
607
111
644
sell crop
set money (ceiling money + (total-harvested * plant-cost) )\nset total-harvested 0
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
235
608
338
644
apply herbicide
add-herbicide
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
118
558
230
598
water garden
add-water\n;if drought? = true [set money (money - 80)]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
201
407
297
452
pesticide count
pesticide-ppm
17
1
11

SLIDER
434
131
562
164
num-plants-2
num-plants-2
1
60
10
1
1
NIL
HORIZONTAL

SLIDER
464
169
561
202
spacing-2
spacing-2
1
3
1
1
1
NIL
HORIZONTAL

TEXTBOX
181
259
331
277
NIL
11
0.0
1

SWITCH
8
215
111
248
drought
drought
1
1
-1000

SWITCH
9
261
112
294
flood
flood
1
1
-1000

BUTTON
116
608
229
644
apply fungicide
ask fungi [die]\nset money (money - 50)\nset pesticide-ppm (pesticide-ppm + 1)
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
910
438
1258
715
Average energy of plants and animals
Time
Creature energy
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Plants1" 1.0 0 -15040220 true "" "ifelse any? plants1 [plot mean [energy] of plants1] [plot 0]"
"Plants2" 1.0 0 -6565750 true "" "ifelse any? plants2 [plot mean [energy] of plants2] [plot 0]"
"Weeds" 1.0 0 -5825686 true "" "ifelse any? weeds [plot mean [energy] of weeds] [plot 0]"
"Bees" 1.0 0 -4079321 true "" "ifelse any? bees [plot mean [energy] of bees] [plot 0]"

TEXTBOX
164
107
423
359
First start by selecting how many of each type of plant you want, and how far apart they should be spaced, by adjusting the sliders (num-plants-1 is the number of the 'plant 1' type.) \n\nThen, adjust the plant-cost and budget sliders. \n\nWhen you've made your selections, press the Setup button, followed by Start. You can always press Start again to pause the model. \n\nTo start over any time, set new conditions and press Setup again. \n\n
12
0.0
1

TEXTBOX
31
665
343
729
The buttons above can be pressed as many times as you want once the model is running.
13
115.0
1

TEXTBOX
19
466
286
531
The boxes above are outputs that will update automatically based on choices you make in the model.
13
43.0
1

TEXTBOX
464
10
1035
28
You can change the speed of the model by dragging the slider above to the left or right.
11
0.0
1

TEXTBOX
14
300
120
372
Toggling one of these switches on will change how the model behaves.
11
76.0
1

TEXTBOX
626
401
1254
428
The farmer will follow your mouse around the field. Click or hold down the mouse button to harvest plants. You can only harvest when the model is running. Right click on anything in the field to see more about it.
11
0.0
1

TEXTBOX
752
500
902
665
These graphs are line graphs, that show the total populations of each species (left) and the average energy of all members of that species (right) over time.
13
43.0
1

@#$#@#$#@
## WHAT IS IT?

This view of a garden bed shows what happens when a farmer tries to plant and grow a variety of crops. Balance to the ecosystem involves careful planning and regular intervention from the farmer, so that the garden doesn't get overrun by weeds, develop fungus, or deplete the soil nutrients.

## HOW IT WORKS

You act as the farmer, and can select how many seeds to plant, how far apart they should be spaced, and what additives you'd like to put into the soil. You can also harvest your plants, sell them, and seed again.


## HOW TO USE IT
In this model, you have two types of plants you can grow in the same plot of land. For each, select how many you would like on the sliders ('Num-plants-1' and 'Num-plants-2'), as well as how far apart they should be spaced ('spacing-1' and 'spacing-2'). You can try out multiple layouts until you find one you are happy with.

When you're ready to start, press 'Start/Stop', and then the farmer will follow your mouse. Things will begin to grow in the garden, hold down the mouse button to harvest a plant. You can click the interface buttons (light purple) at any time while the model is running, or press 'Start/Stop' again to pause it.

## THINGS TO NOTICE

The graphs will constantly update based on what is happening in the model. Watch to see how they change directly after pressing a button like "Apply Compost", "Apply herbicide", or "Water Garden".

What happens to the color of the soil as you let the model run?


## THINGS TO TRY

Toggle on the drought mode or flood mode and see how the model behaves differently.

You can approach the model with many goals in mind:
   - Make the most money
   - Grow organically
   - Support pollinator populations
   - Strategize for the effects of climate change

## EXTENDING THE MODEL

Try changing the rules related to the plant varieties so that they are more or less drought tolerant. (For example, the level of hydration the soil must have for them to eat, the amount of energy they get from eating)

Currently the only way to add nutrients to the soil is to either make a purchase by Aplying compost, or to let plants die and recycle themselves. Can you create another animal in the ecosystem that might add nutrients to the soil, given certain conditions?

You can also change the shape and appearance of the plants and weeds. Can you design a crop you know? How would it behave? What does it add or take away from the soil? What are the stages of its growth? What does it look like from seed to fruit to flower?


## RELATED MODELS

Garden Ecosystem

## CREDITS AND REFERENCES
Developed by Rebecca Shareff, UC Berkeley (2019)
https://ocf.berkeley.edu/~beccashareff

Michelle Wilkerson and Dor Abrahamson supported the development of this model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee 2
true
0
Polygon -1184463 true false 195 150 105 150 90 165 90 225 105 270 135 300 165 300 195 270 210 225 210 165 195 150
Rectangle -16777216 true false 90 165 212 185
Polygon -16777216 true false 90 207 90 226 210 226 210 207
Polygon -16777216 true false 103 266 198 266 203 246 96 246
Polygon -16777216 true false 120 150 105 135 105 75 120 60 180 60 195 75 195 135 180 150
Polygon -16777216 true false 150 15 120 30 120 60 180 60 180 30
Circle -1184463 true false 105 30 30
Circle -1184463 true false 165 30 30
Polygon -7500403 true true 120 90 75 105 15 90 30 75 120 75
Polygon -16777216 false false 120 75 30 75 15 90 75 105 120 90
Polygon -7500403 true true 180 75 180 90 225 105 285 90 270 75
Polygon -16777216 false false 180 75 270 75 285 90 225 105 180 90
Polygon -7500403 true true 180 75 180 90 195 105 240 195 270 210 285 210 285 150 255 105
Polygon -16777216 false false 180 75 255 105 285 150 285 210 270 210 240 195 195 105 180 90
Polygon -7500403 true true 120 75 45 105 15 150 15 210 30 210 60 195 105 105 120 90
Polygon -16777216 false false 120 75 45 105 15 150 15 210 30 210 60 195 105 105 120 90
Polygon -16777216 true false 135 300 165 300 180 285 120 285

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

caterpillar
true
0
Polygon -7500403 true true 165 210 165 225 135 255 105 270 90 270 75 255 75 240 90 210 120 195 135 165 165 135 165 105 150 75 150 60 135 60 120 45 120 30 135 15 150 15 180 30 180 45 195 45 210 60 225 105 225 135 210 150 210 165 195 195 180 210
Line -16777216 false 135 255 90 210
Line -16777216 false 165 225 120 195
Line -16777216 false 135 165 180 210
Line -16777216 false 150 150 201 186
Line -16777216 false 165 135 210 150
Line -16777216 false 165 120 225 120
Line -16777216 false 165 106 221 90
Line -16777216 false 157 91 210 60
Line -16777216 false 150 60 180 45
Line -16777216 false 120 30 96 26
Line -16777216 false 124 0 135 15

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

drop
false
0
Circle -7500403 true true 73 133 152
Polygon -7500403 true true 219 181 205 152 185 120 174 95 163 64 156 37 149 7 147 166
Polygon -7500403 true true 79 182 95 152 115 120 126 95 137 64 144 37 150 6 154 165

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

farmer
false
2
Circle -955883 true true 110 5 80
Polygon -13345367 true false 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -955883 true true 127 79 172 94
Polygon -955883 true true 195 90 240 150 225 180 195 120
Polygon -955883 true true 105 90 60 150 75 180 105 120
Polygon -1184463 true false 75 45 225 45 195 30 180 0 120 0 105 30 75 45
Polygon -6459832 true false 225 240 240 255 255 105 240 105 225 240
Polygon -7500403 true false 270 105 225 105 225 60 240 45 255 45 270 60 270 105
Line -16777216 false 120 90 135 120
Polygon -955883 true true 120 90 135 120 165 120 180 90 120 90

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower 1
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -5825686 true false 85 132 38
Circle -5825686 true false 130 147 38
Circle -5825686 true false 192 85 38
Circle -5825686 true false 85 40 38
Circle -5825686 true false 177 40 38
Circle -5825686 true false 177 132 38
Circle -5825686 true false 70 85 38
Circle -5825686 true false 130 25 38
Circle -5825686 true false 96 51 108
Circle -1184463 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

flower budding
false
0
Polygon -7500403 true true 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Polygon -7500403 true true 189 233 219 188 249 173 279 188 234 218
Polygon -7500403 true true 180 255 150 210 105 210 75 240 135 240
Polygon -7500403 true true 180 150 180 120 165 97 135 84 128 121 147 148 165 165
Polygon -7500403 true true 170 155 131 163 175 167 196 136

flower2
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -2064490 true false 85 132 38
Circle -2064490 true false 130 147 38
Circle -2064490 true false 192 85 38
Circle -2064490 true false 85 40 38
Circle -2064490 true false 177 40 38
Circle -2064490 true false 177 132 38
Circle -2064490 true false 70 85 38
Circle -2064490 true false 130 25 38
Circle -13345367 true false 96 51 108
Circle -1184463 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

leaf 2
false
0
Rectangle -7500403 true true 144 218 156 298
Polygon -7500403 true true 150 263 133 276 102 276 58 242 35 176 33 139 43 114 54 123 62 87 75 53 94 30 104 39 120 9 155 31 180 68 191 56 216 85 235 125 240 173 250 165 248 205 225 247 200 271 176 275

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

mouse side
false
0
Polygon -7500403 true true 38 162 24 165 19 174 22 192 47 213 90 225 135 230 161 240 178 262 150 246 117 238 73 232 36 220 11 196 7 171 15 153 37 146 46 145
Polygon -7500403 true true 289 142 271 165 237 164 217 185 235 192 254 192 259 199 245 200 248 203 226 199 200 194 155 195 122 185 84 187 91 195 82 192 83 201 72 190 67 199 62 185 46 183 36 165 40 134 57 115 74 106 60 109 90 97 112 94 92 93 130 86 154 88 134 81 183 90 197 94 183 86 212 95 211 88 224 83 235 88 248 97 246 90 257 107 255 97 270 120
Polygon -16777216 true false 234 100 220 96 210 100 214 111 228 116 239 115
Circle -16777216 true false 246 117 20
Line -7500403 true 270 153 282 174
Line -7500403 true 272 153 255 173
Line -7500403 true 269 156 268 177

mushroom
true
14
Rectangle -1 true false 120 195 180 270
Polygon -7500403 true false 118 197 78 196 65 164 88 122 136 107 196 108 240 143 246 188 233 198 179 196 118 197
Rectangle -7500403 true false 165 195 180 270
Rectangle -7500403 true false 120 195 135 270
Rectangle -7500403 false false 120 270 180 285
Rectangle -7500403 true false 120 270 180 285
Polygon -1 true false 180 135 210 165 225 150 180 135 225 150 195 150 210 165 195 150 210 150 180 135 210 150 210 165 195 150 210 150 195 150 195 135 225 150 195 135 180 135 210 150 210 165 210 150 195 150 180 135 195 150 195 135 180 135 195 135 210 150 225 150 210 150 195 150 210 150 225 150 180 135
Polygon -1 true false 180 135 225 150 195 135 180 135 210 165

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

plant medium
false
0
Rectangle -7500403 true true 135 165 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 165 120 120 150 90 180 120 165 165

plant medium 2
false
0
Rectangle -7500403 true true 135 165 165 300
Polygon -7500403 true true 135 255 90 210 60 225 75 270 135 285
Polygon -7500403 true true 165 255 210 210 240 225 225 270 165 285
Polygon -7500403 true true 135 180 90 135 60 150 75 180 135 225
Polygon -7500403 true true 165 180 165 225 225 180 240 150 210 135
Polygon -7500403 true true 135 165 120 120 150 105 180 120 165 165

plant small
false
0
Rectangle -7500403 true true 135 240 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 240 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 240 165 285
Polygon -7500403 true true 135 255 120 210 150 135 180 210 165 255

plant small2
false
0
Rectangle -7500403 true true 135 240 165 300
Polygon -7500403 true true 135 255 90 210 60 225 60 255 135 285
Polygon -7500403 true true 165 255 210 210 240 225 240 255 165 285
Polygon -7500403 true true 135 255 120 210 150 180 180 210 165 255

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

strawberry
false
0
Polygon -7500403 false true 149 47 103 36 72 45 58 62 37 88 35 114 34 141 84 243 122 290 151 280 162 288 194 287 239 227 284 122 267 64 224 45 194 38
Polygon -7500403 true true 72 47 38 88 34 139 85 245 122 289 150 281 164 288 194 288 239 228 284 123 267 65 225 46 193 39 149 48 104 38
Polygon -10899396 true false 136 62 91 62 136 77 136 92 151 122 166 107 166 77 196 92 241 92 226 77 196 62 226 62 241 47 166 57 136 32
Polygon -16777216 false false 135 62 90 62 135 75 135 90 150 120 166 107 165 75 196 92 240 92 225 75 195 61 226 62 239 47 165 56 135 30
Line -16777216 false 105 120 90 135
Line -16777216 false 75 120 90 135
Line -16777216 false 75 150 60 165
Line -16777216 false 45 150 60 165
Line -16777216 false 90 180 105 195
Line -16777216 false 120 180 105 195
Line -16777216 false 120 225 105 240
Line -16777216 false 90 225 105 240
Line -16777216 false 120 255 135 270
Line -16777216 false 120 135 135 150
Line -16777216 false 135 210 150 225
Line -16777216 false 165 180 180 195

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tomato plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 60 210 75 255 135 285
Polygon -7500403 true true 165 255 210 210 240 210 225 255 165 285
Polygon -7500403 true true 135 180 90 135 60 135 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 240 135 210 135
Polygon -7500403 true true 135 105 90 60 60 60 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 240 60 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
Circle -2674135 true false 90 90 30
Circle -2674135 true false 174 84 42
Circle -2674135 true false 60 150 30
Circle -2674135 true false 180 165 30
Circle -2674135 true false 150 30 30
Circle -2674135 true false 210 135 30
Circle -2674135 true false 60 60 30

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
