;;
;; This model is developed by Dr. Lin Xiang at the University of Kentucky. Contact: lin.xiang@uky.edu
;;
;; If you see this page rather than a "download" button when downloading the model, click the "download" icon
;; in your browser to download this model file.
;;
;; This model is free for classroom or virtual learning or teacher professional development related to STEM education.
;; You are asked to provide citations when using this model in any published work.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


breed [insulins insulin]
breed [glucose a-glucose]
breed [gluts glut]
breed [I-receptors I-receptor]
breed [G-receptors G-receptor]
breed [lipids lipid]
breed [glucagons glucagon]
breed [legends legend]

insulins-own [bonding]
patches-own[protein]
gluts-own[bonding]
glucose-own [bonding cyto glycogen]
I-receptors-own [bonding]
G-receptors-own [bonding]
glucagons-own [bonding]

globals[clock]

to-report blood-sugar
  report count glucose with [ycor > 2 and color = white and bonding = 0]
end

to-report cyto-glucose
  report count glucose with [ycor < 2 and color = white and bonding = 0 and glycogen = 0]
end

to-report glycogen-level
  report count glucose with [glycogen = 1 and color = 138]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Main Procedures (SETUP and GO)                                    ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
 ;clear-turtles
 ;clear-patches
ca
 setup-legend     ; must be the first line
 setup-membrane-IR
 setup-glucose
 check-glucose-location
 setup-glut
 setup-g-receptors
 set-insulin-sensitivity

reset-ticks
end


to go
  if blood-sugar < 54 [user-message "The blood sugar level is dangerously low." Stop]


check-glucose-location
food-intake
set-insulin-sensitivity
setup-insulin
setup-glucagon
glucose-movement
insulin-movement
glucagon-movement
glycolysis
glycogenesis-glycogenolysis
plot-levels

tick

end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Sub-procedures of SETUP                                           ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup-membrane-IR

;make the membrane
  ask patches with [pycor = 2] [set pcolor 49 ]

; make boundary for the system
  ask patches with [abs pxcor = max-pxcor or abs pycor = max-pycor] [set pcolor 2]
  ask patches with [pycor <= min-pycor + 4 ] [set pcolor 2]

;add Insulin receptors (I-R) into the membrane
  ask patches with [pcolor = 49] [
    if  pxcor = -2 or
        pxcor = 6
    [set pcolor 123]
  ]

 ask patches with [pcolor = 123]
     [sprout-I-receptors 1
        [set color yellow - 2
        set size 2.5
        set shape "I-R"
        set bonding 0 ]
     ]

  ; add bilayer
   ask patches with [ pycor = 2 and pcolor = 49] [sprout-lipids 1 [set color 7 set shape "lipid-1" set size 1 set heading 0 set ycor 2]]

 end

to setup-glucose
  create-glucose Starting-Blood-sugar-level
  [set color white
    set shape "glucose"
    set size 0.75
    set bonding 0
    set glycogen 0
    setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) 2.5 + random-float 5]

  create-glucose Starting-cytoplasmic-glucose
  [set color white
    set shape "glucose"
    set size 0.75
    set bonding 0
    set glycogen 0
    setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) 1 + random-float -4]

  create-glucose 200        ;set glycogen
  [set color 138
    set shape "glucose"
    set size 0.75
    set bonding 0
    set glycogen 1
    setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) min-pycor + 5 + random-float 0.5]
end

to setup-glut
 ask patches with [(pxcor = 2 or pxcor = -10 or pxcor = 10)  and pycor = 2 and pcolor = 49]
[if any? lipids-here
[ask lipids-here [die]
sprout-gluts 1 [set shape "glut" set color 86.5 set size 1.25 set heading 0 set bonding 0]
]]
end

to setup-insulin

 if blood-sugar > 140 and can-produce-insulin?     ;only produce insulin when blood sugar level is high and capable of producing insulin
  [if not any? insulins with [ycor > 2] [
    create-insulins 10
    [ set bonding 0
    set shape "insulin"
    set size 2
    set color 95
      setxy -6 + random-float 16 2.5 + random-float 5]]]

  if blood-sugar < 120
  [if count insulins != 0
    [ask insulins with [ycor > 2] [die]]
  ]

end

to setup-g-receptors
 ask patches with [pxcor = -6 and pycor = 2 and pcolor = 49]
  [if any? lipids-here
    [ask lipids-here [die]
   sprout-G-receptors 1 [set shape "g-r" set color 26 set size 3 set heading 0 set bonding 0 setxy -6 2.35]
]]
end

to setup-glucagon

 if blood-sugar < 98
  [if not any? glucagons with [ycor > 1] [
    create-glucagons 10
    [ set bonding 0
    set shape "glucagon"
    set size 1.75
    set color 64
      setxy -10 + random 5 2.5 + random-float 5]]]

  if blood-sugar > 103
  [if count glucagons != 0
    [ask glucagons with [ycor > 0] [die]]
  ]

end

to setup-legend
 create-legends 10
  [set shape "legend"]

 ask legend 0
  [set label "lipids"
    setxy -8 -4.75 ]

  create-lipids 1
  [set color 7
    set shape "lipid-1"
    set size 0.75
    set heading 0
    setxy -8 -4.25 ]

  ask legend 1
  [set label "Glucose"
    setxy -2.75 -4.75]

  Create-glucose 1
  [set color white
    set shape "glucose"
    set size 1.25
    set heading 0
    set glycogen 2
    setxy -3 -4.25]

  ask legend 2
  [set label "Glycogen"
    setxy 2.75 -4.75 ]

  Create-glucose 2
  [set color 138
    set shape "glucose"
    set size 1.25
    set glycogen 2
    setxy 2 + random-float 0.5 -4.25]

  ask legend 3
  [ set label "Glucose Transporter 2"
    setxy 10 -4.75 ]

  create-gluts 1
  [set shape "glut"
    set color 86.5
    set size 0.75
    set heading 0
    set bonding 2
    setxy 8 -4.25]

 ask legend 4
  [set label "Insulin"
    setxy -7.8 -7.25 ]

 create-insulins 1
  [ set bonding 0
    set shape "insulin"
    set size 1.25
    set color 95
    setxy -8 -6.75]

  ask legend 5
  [set label "Insulin"
    setxy  -2.8 -7.15 ]

ask legend 6
  [set label "Receptor"
    setxy  -2.5 -7.65 ]

  create-I-receptors 1
       [set color yellow
        set size 0.75
        set shape "I-R"
        setxy -3 -6.75]

ask legend 7
  [set label "Glucagon"
    setxy  2.75 -7.25 ]

create-Glucagons 1
  [set color 64
    set shape "Glucagon"
    set size 1.25
    set heading 90
    setxy 2.25 -6.75
  ]

 ask legend 8
  [set label "Glucagon"
    setxy  8.5 -7.15 ]

ask legend 9
  [set label "Receptor"
    setxy  8.5 -7.65 ]

 create-G-receptors 1
  [set color 26
    set shape "g-r"
    set size 0.75
    set heading 90
    setxy 8 -6.75
  ]

  create-lipids 1
  [set color 7
    set shape "line"
    set size 22
    set heading 90
    setxy 0 -5.75
  ]

end
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Sub-procedures of GO                                              ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



to glucose-movement
  ask glucose with [glycogen = 0] [
  ifelse bonding = 0
  [ if pcolor = 0
      [right random 90 left random 90   ;wiggle
         if [pcolor] of patch-at dx dy = 0 [fd 0.2]   ;If nothing is ahead, move forward
    ]
    if any? gluts-on patch-ahead 1 [
      let my-channel one-of gluts-on patch-ahead 1
      if [bonding] of my-channel = 0 [
       set color red move-to my-channel set bonding 1

        ask my-channel [set bonding 1 set color 85]]]]
  [
    ifelse blood-sugar > cyto-glucose
      [set heading 180 fd 0.1] [set heading 0 fd 0.1]
      if pcolor = 0 [set bonding 0 set color white]
        ]
  ]

  ask gluts with [bonding = 1] [if count glucose-here = 0 [set bonding 0 set color 86.5]]
end


to check-glucose-location ;detemine glucose location
  ask glucose [ifelse ycor > 2 [set cyto 0][set cyto 1]]
end


to insulin-movement
  ask insulins with [ycor > 2]
  [if bonding = 0
    [
      if pcolor = 0
      [right random 90 left random 90   ;wiggle
         if [pcolor] of patch-at dx dy = 0 [fd 0.2]
      ]

      if any? I-receptors in-radius 2 with [bonding = 0] [
      let my-receptor one-of I-receptors in-radius 2 with [bonding = 0]
        set heading 0 move-to my-receptor set ycor ycor + 1.2  set bonding 1

        ask my-receptor [set bonding 1 set color yellow]]]
      ]

  ask I-receptors with [bonding = 1]
  [if count insulins with [ycor > 2] = 0 [set bonding 0 set color yellow - 2]]
end

to glucagon-movement
  ask glucagons with [ycor > 2]
  [if bonding = 0
    [
      if pcolor = 0
      [right random 90 left random 90   ;wiggle
         if [pcolor] of patch-at dx dy = 0 [fd 0.2]
      ]

      if any? G-receptors in-radius 2 with [bonding = 0] [
      let my-receptor one-of G-receptors in-radius 2 with [bonding = 0]
        set heading 0 move-to my-receptor set xcor xcor - 0.3 set ycor ycor + 0.95 set bonding 1

        ask my-receptor [set bonding 1 set color 27]]]
      ]

  ask G-receptors with
    [bonding = 1] [if count glucagons with [ycor > 2] = 0 [set bonding 0 set color 26]]
end

to glycolysis
  ask glucose with [glycogen = 0 and cyto = 1] [
    if random 1000000 < metabolic-rate  [die]
  ]
end

to glycogenesis-glycogenolysis

  if glycogen-level = 0
   [if count glucose with [cyto = 1] != nobody
     [ask one-of glucose with [cyto = 1]    ;set a starter
        [set glycogen 1
         set color 138
         setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) -2 + random-float -1 ]]]

  if glycogen-level < 800 [       ;glycogenesis set a max glycogen
  ask glucose with [cyto = 1 and glycogen = 0]
     [if any? other glucose-here with [glycogen = 1]
      [if random 100 < (1 + (count insulins with [bonding = 1])* 45)
        [ set glycogen 1 set color 138]
  ]]]

  if any? G-receptors with [bonding = 1]              ;glycogenolysis
    [ask glucose with [cyto = 1 and glycogen = 1]
     [if random 1000 < 5
      [set glycogen 0 set color white
       set heading 45 - random 90
          fd 0.5]]
    ]
end

to set-insulin-sensitivity
  ask I-receptors with [ycor > 0]
    [ifelse Insulin-resistance?
      [set bonding 2 set color yellow - 2]
      [ifelse color = yellow [set bonding 1][set bonding 0]]
      ]
end

to food-intake
  set clock clock + 1
  if clock = 1200 [
    if eat-every-4-hrs?
  [create-glucose 100
  [set color white
    set shape "glucose"
    set size 1
    set bonding 0
    set glycogen 0
    setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) 2.5 + random-float 5]

 ask glucose with [ycor > -4]
  [if pcolor = 2 [ifelse random 2 = 0 [set xcor xcor + 1 ][set xcor xcor - 1 ]]]
  ]
    set clock 0]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;         Supporting-procedures                                             ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to plot-levels ;; this creates creates the bar graph
    set-current-plot "Glucose Gradient"
    clear-plot
    plot-pen-down
    set-current-plot-pen "Blood stream glucose"
    plotxy 1 blood-sugar
    set-current-plot-pen "Cytoplasm glucose"
    plotxy 2 cyto-glucose
    set-current-plot-pen "Glycogen"
    plotxy 3 glycogen-level

end
@#$#@#$#@
GRAPHICS-WINDOW
19
10
621
393
-1
-1
22.0
1
10
1
1
1
0
1
1
1
-13
13
-8
8
0
0
1
ticks
10.0

BUTTON
633
244
754
280
Setup/Reset
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
759
244
854
280
Run/Pause
go\n\nif ticks > Investigation-Time * 300 [user-message \"The simulation has run the defined hours. You may increase the time to continue running the simulation.\"  stop]\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
759
286
854
319
Run a tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
278
396
499
588
Blood Sugar level
Time
Glucose
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Blood sugar  " 1.0 0 -2674135 true "" "plotxy ticks blood-sugar"

PLOT
19
396
271
587
Glucose Gradient
NIL
NIL
0.0
5.0
0.0
100.0
true
true
"" ""
PENS
"Blood Stream glucose" 1.0 1 -16777216 true "" "plotxy 1 blood-sugar"
"Cytoplasm glucose" 1.0 1 -7500403 true "" "plotxy 2 cyto-glucose"
"Glycogen" 1.0 1 -2064490 true "" "plotxy 3 glycogen-level"

TEXTBOX
20
593
269
626
Tip: Run the model at a slower speed to get a better visual result.
11
105.0
1

MONITOR
717
440
826
485
Blood Sugar Level
count glucose with [ycor > 2]
17
1
11

BUTTON
759
321
854
354
Snack
 create-glucose 50\n  [set color white\n    set shape \"glucose\"\n    set size 1\n    set bonding 0\n    set glycogen 0\n    setxy (max-pxcor - 1) - random-float ((max-pxcor - 1) * 2) 2.5 + random-float 5]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
717
396
826
437
Time (Hours)
ticks / 300
1
1
10

SLIDER
634
10
854
43
Starting-Blood-sugar-level
Starting-Blood-sugar-level
60
250
250.0
10
1
NIL
HORIZONTAL

SLIDER
634
88
855
121
Metabolic-rate
Metabolic-rate
0
100
30.0
10
1
NIL
HORIZONTAL

INPUTBOX
633
286
753
354
Investigation-Time
1.0
1
0
Number

TEXTBOX
711
325
753
343
Hours
11
125.0
1

PLOT
505
396
705
588
Glycogen
Time
Glycogen
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Glycogen   " 1.0 0 -13345367 true "" "plotxy ticks glycogen-level"

MONITOR
717
489
826
534
Glycogen
glycogen-level
17
1
11

BUTTON
633
358
744
391
Watch a glucose
watch one-of glucose with [ycor > -4 and color = white]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
747
358
854
391
Stop watching
rp
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
634
49
854
82
Starting-cytoplasmic-glucose
Starting-cytoplasmic-glucose
60
200
100.0
1
1
NIL
HORIZONTAL

SWITCH
633
128
854
161
Can-produce-insulin?
Can-produce-insulin?
0
1
-1000

SWITCH
633
166
854
199
Insulin-resistance?
Insulin-resistance?
1
1
-1000

SWITCH
633
202
855
235
Eat-every-4-hrs?
Eat-every-4-hrs?
1
1
-1000

@#$#@#$#@
## WHAT IS IT?

This model simulates the processes of glucose regulation in liver cells. Students may observe the changes in blood sugar level regulated by glucose transportation, glycogenesis, and glycogenolysis.   

## HOW IT WORKS

This model represents the following ideas:

In the model, the upper area is the bloodstream or extracellular space, and the lower area is the liver. The following ideas are represented in the model:

* The glucose transporters (GLUT-2) permanently exist in the cytomembrane of liver cells.
* Glucose passes the cytomembrane of liver cells through facilitated diffusion.
* Insulin stimulates glycolysis and glycogenesis in liver cells.   
* Glucagon stimulates glycogenolysis.
* A higher Metabolic rate enables the glucose to be broken down faster.


## THINGS TO NOTICE AND TRY

* Where do the glucose molecules move? Out of the liver cell or into the liver cell?

* When is insulin produced?

* How quickly does the blood sugar level change with and without insulin resistance?

* How does insulin resistance affect glycogenesis, i.e., glycogen synthesis? 

* What happens to the blood sugar level and glycogen level when "you" keep eating?

* What happens to the blood sugar level when glycogen reaches the max level? 

* What are the real-life examples of the simulated scenarios above?

* What process is not represented in the model?

## KEEP IN MIND

This model only shows the blood sugar regulation in the liver. When you turn on the switch "eat-every-4-hrs?", you should see that blood sugar will build up over time as the liver can only store a certain amount of glucose as glycogen. Other cells, such as muscle and adipose cells, also participate in regulating blood sugar. 

## HOW TO USE IT

1. Click on the "setup/reset" button to initiate or reset the model. 
2. Put down the time you want to model to run. For a preliminary investigation, put down a large number (e.g., 1000) to get a sufficient observation time. To set experiments, set time as 1 ~4 hours to get the results close to the real-life. 
3. Click on the "Run/pause" button to run or pause the model. Click on the "Run a tick" button to run the model for just one tick. 
4. Use switches to explore/test the impacts of insulin production and resistance.
5. Use the "snack" button to simulate a small food intake. Use the "eat-every-4-hrs?" switch to add glucose into the bloodstream regularly.
6. Use sliders to adjust the initial blood sugar level, cytoplasmic glucose level, and metabolic rate.
7. Use the "watch a glucose" and "stop watching" buttons to track the behaviors of an individual glucose molecule. 
8. Examine results in the simulation window, the numeric monitors, and plots.

**Run the model at a lower speed to get a better visual effect.**



## RELATED MODELS

Check [3dsciencemodeling.com](http://3dsciencemodeling.com) for related models and other resources.


## CREDITS AND REFERENCES

Copyright 2021 by Lin Xiang.  All rights reserved.

This module is made by Dr. Lin Xiang at the University of Kentucky. If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2021). Blood sugar regulation by the liver. Department of STEM Education, University of Kentucky, Lexington, KY.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

adp
true
0
Circle -7500403 true true 201 127 48
Circle -7500403 true true 152 127 48
Polygon -7500403 true true 150 150 120 195 60 195 30 150 90 105

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

atp
true
0
Circle -7500403 true true 250 126 48
Circle -7500403 true true 201 127 48
Circle -7500403 true true 152 127 48
Polygon -7500403 true true 150 150 120 195 60 195 30 150 90 105

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

enzyme1
true
0
Rectangle -7500403 true true 75 30 225 270

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

g-r
false
0
Polygon -7500403 true true 105 0 105 15 135 30 135 75 105 90 105 180 105 225 90 240 90 285 105 300 195 300 210 285 210 240 195 225 195 0 105 0

glucagon
true
0
Polygon -7500403 true true 150 90 195 120 195 180 150 210 120 195 105 195 105 180 105 165 120 150 105 135 105 120 105 105 120 105

glucose
true
0
Polygon -7500403 true true 150 90 105 120 105 180 150 210 195 180 195 120

glut
true
0
Polygon -7500403 true true 225 30 285 90 285 210 210 285 165 285 165 210 195 180 195 120 165 90 165 15 210 15
Polygon -7500403 true true 75 30 15 90 15 210 90 285 135 285 135 210 105 180 105 120 135 90 135 15 90 15

glut-bonded
true
0
Polygon -7500403 true true 225 30 285 90 285 210 210 285 165 285 165 210 210 180 210 120 165 90 165 15 210 15
Polygon -7500403 true true 75 30 15 90 15 210 90 285 135 285 135 210 90 180 90 120 135 90 135 15 90 15
Polygon -1 true false 150 90 105 120 105 180 150 210 195 180 195 120

hex
true
0
Polygon -7500403 true true 0 150 75 30 225 30 300 150 225 270 75 270

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

i-r
false
0
Polygon -7500403 true true 210 0 150 45 150 105 150 120 180 150 180 210 165 240 150 255 150 285 165 300 225 300 240 285 240 255 225 240 210 210
Polygon -7500403 true true 90 0 150 45 150 105 150 120 120 150 120 210 135 240 150 255 150 285 135 300 75 300 60 285 60 255 75 240 90 210

insulin
true
0
Polygon -7500403 true true 150 195 210 150 210 105 90 105 90 150
Line -2064490 false 90 120 210 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

legend
false
0

legend-1
false
0
Rectangle -7500403 true true 120 120 180 180

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

lipid
true
0
Circle -7500403 false true 113 1 74
Line -7500403 true 165 75 165 105
Line -7500403 true 135 75 135 105
Line -7500403 true 135 105 150 120
Line -7500403 true 150 120 135 135
Line -7500403 true 135 135 150 150
Line -7500403 true 135 165 150 180
Line -7500403 true 135 195 150 210
Line -7500403 true 135 225 150 240
Line -7500403 true 135 255 150 270
Line -7500403 true 135 285 150 300
Line -7500403 true 165 105 180 120
Line -7500403 true 165 135 180 150
Line -7500403 true 165 165 180 180
Line -7500403 true 165 195 180 210
Line -7500403 true 165 225 180 240
Line -7500403 true 165 255 180 270
Line -7500403 true 150 150 135 165
Line -7500403 true 180 120 165 135
Line -7500403 true 180 150 165 165
Line -7500403 true 180 180 165 195
Line -7500403 true 150 180 135 195
Line -7500403 true 180 210 165 225
Line -7500403 true 150 210 135 225
Line -7500403 true 150 240 135 255
Line -7500403 true 150 270 135 285
Line -7500403 true 180 240 165 255

lipid-1
false
0
Circle -7500403 false true 240 239 60
Circle -7500403 false true 0 239 60
Circle -7500403 false true 179 240 60
Circle -7500403 false true 120 240 60
Circle -7500403 false true 60 240 60
Circle -7500403 false true 0 1 60
Circle -7500403 false true 60 0 60
Circle -7500403 false true 120 0 60
Circle -7500403 false true 179 0 60
Circle -7500403 false true 240 1 60
Line -7500403 true 30 240 30 165
Line -7500403 true 30 60 30 135
Line -7500403 true 90 240 90 165
Line -7500403 true 90 60 90 135
Line -7500403 true 150 60 150 135
Line -7500403 true 150 240 150 165
Line -7500403 true 210 240 210 165
Line -7500403 true 210 60 210 135
Line -7500403 true 270 60 270 135
Line -7500403 true 270 240 270 165

lipid-2
true
0
Line -7500403 true 150 120 150 149
Line -7500403 true 120 120 120 149
Line -7500403 true 150 180 165 195
Line -7500403 true 120 150 135 165
Line -7500403 true 135 165 120 180
Line -7500403 true 120 180 135 195
Line -7500403 true 135 195 120 210
Line -7500403 true 120 210 135 225
Line -7500403 true 135 225 120 240
Line -7500403 true 120 240 135 255
Line -7500403 true 135 255 120 270
Line -7500403 true 120 270 135 285
Line -7500403 true 150 150 165 165
Line -7500403 true 165 165 150 180
Line -7500403 true 165 195 150 210
Line -7500403 true 150 210 165 225
Line -7500403 true 165 225 150 240
Line -7500403 true 150 240 165 255
Circle -7500403 false true 75 0 120

lipid-3
true
0
Line -7500403 true 165 120 165 151
Line -7500403 true 120 114 120 149
Line -7500403 true 165 180 180 195
Line -7500403 true 120 150 135 165
Line -7500403 true 135 165 120 180
Line -7500403 true 120 180 135 195
Line -7500403 true 135 195 120 210
Line -7500403 true 120 210 135 225
Line -7500403 true 135 225 120 240
Line -7500403 true 120 240 135 255
Line -7500403 true 135 255 120 270
Line -7500403 true 120 270 135 285
Line -7500403 true 165 150 180 165
Line -7500403 true 180 165 165 180
Line -7500403 true 180 195 165 210
Line -7500403 true 165 210 180 225
Line -7500403 true 180 225 165 240
Line -7500403 true 165 240 180 255
Circle -7500403 false true 90 0 120

lipids
true
0
Circle -7500403 false true 128 1 44
Line -7500403 true 145 45 145 236
Line -7500403 true 156 45 156 286

lipids-1
true
0
Circle -7500403 false true 241 1 58
Line -7500403 true 270 60 270 283
Circle -7500403 false true 181 1 58
Circle -7500403 false true 121 0 58
Circle -7500403 false true 61 1 58
Circle -7500403 false true 1 1 58
Line -7500403 true 210 60 210 283
Line -7500403 true 90 60 90 283
Line -7500403 true 30 60 30 283
Line -7500403 true 150 59 150 282

lipids-2
true
0
Circle -7500403 false true 241 1 58
Line -7500403 true 270 60 270 283
Circle -7500403 false true 121 0 58
Circle -7500403 false true 1 1 58
Line -7500403 true 30 60 30 283
Line -7500403 true 150 59 150 282

nad
true
0
Polygon -7500403 false true 30 150 75 90 165 90 210 150 165 210 75 210

nadh
true
0
Polygon -7500403 true true 75 90 30 150 75 210 165 210 210 150 165 90
Circle -16777216 false false 117 117 66
Line -16777216 false 135 150 165 150
Line -16777216 false 150 135 150 165

nadh-1
true
0
Polygon -7500403 true true 75 90 30 150 75 210 165 210 210 150 165 90
Circle -7500403 false true 207 117 66
Line -7500403 true 225 150 255 150
Line -7500403 true 240 135 240 165

oxygen
true
0
Circle -7500403 true true 135 90 120
Circle -7500403 true true 45 90 120

p-1
true
0
Rectangle -7500403 true true 15 120 285 180
Circle -7500403 true true 75 30 150

p-2
true
0
Polygon -7500403 true true 0 135 45 90 105 135 150 90 195 135 240 90 300 135 300 165 240 120 195 165 150 120 105 165 45 120 0 165

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

proton
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Line -7500403 true 150 75 150 225
Line -7500403 true 75 150 225 150

proton1
true
0
Circle -16777216 true false 13 13 272
Circle -7500403 false true 15 15 270
Line -7500403 true 150 90 150 210
Line -7500403 true 90 150 210 150

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

water
true
0
Circle -1 true false 65 35 170
Circle -7500403 true true 207 131 67
Circle -7500403 true true 26 131 67

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
