breed [persons person]
breed [stores store]
breed [houses house]
;; the start-time (unhappiness start-time) is the first time when each customer have visited the target store at least once
;; the unhappiness-values-list keeps track of the unhappiness values of the population of the customers for calculation of standard-deviation
globals [start-time unhappiness-values-list]

persons-own [house-id home-xcor home-ycor
  ;; who-number of the home belonging to person, the x-coordinate and y-coordinate of the home
  store-travel-time-list store-visit-count-list store-total-waiting-time-list
  ;; store-travel-time-list contains the list of travel-distance from home to each store, store-visit-count-list contains the total number of visits by the person to each store
  ;; store-waiting-time-list contains the total waiting time for each store in the past visits of the person
  rest-time wait-time demand target-store-id
  ;; how much time the person will rest at home after returning from store
  ;; what is the quantity of the item that person requires in current visit to store
  ;; in the beginning of journey, the person decides which store to visit, the who of target store is saved in target-store-id
  at-home? at-store? travelling-to-store? travelling-to-home?]
  ;; the state variables, at-home is true if the person is at home, travelling-to-store if on way to store, travelling-to-home if on way to home, at-store if at store currently
stores-own [quantity]
;; every store keeps track of its current quantity

;; Reports the current average-unhappiness of the population
;; A person's unhappiness for a given store = Avg. Waiting Time at the store + Travel Time for the store (proportional to distance of store from home)
;; A person's unhappiness is counted towards average unhappiness, only when the person has visited the current targeted store at least once.
;; ^ This is to avoid bad unhappiness values when the person is yet to experience the target store atleast once
to-report average-unhappiness
  ;; Initialize the list for the unhappiness values of the customers
  set unhappiness-values-list []
  ;; The number of customers which have actually experienced the target-store at least once
  let actual-num-consumers 0
  ask persons [
    ;; If the person has visited the target store at least once
    if (item target-store-id store-visit-count-list) > 0
    [
      ;; insert its unhappiness value in the unhappiness list for the population
      set unhappiness-values-list lput store-unhappiness target-store-id unhappiness-values-list
      set actual-num-consumers actual-num-consumers + 1 ]
  ]
  ;; If all of the customers have atleast experienced their current target store at least once, then set the current time as the start-time
  if actual-num-consumers = num-consumers and start-time = 0 [set start-time ticks]

  ;; color the customers and houses as per the unhappiness distribution
  ;; report the standard-deviation and mean of the unhappiness if the actual numbers of customers are greater than 2
  if (actual-num-consumers >= 2)
  [
    let std-dev-unhappiness standard-deviation unhappiness-values-list
    let mean-unhappiness mean unhappiness-values-list
    ;;set the colors of the customer and house agents
    set-customer-unhappiness-colors std-dev-unhappiness mean-unhappiness
    ;; If the user has turned ON the visual tool to show the unhappiness distribution over the patches, then color the patches accordingly
    ;; otherwise set all the patches to turquoise color
    ifelse (show-unhappiness-terrain?) [show-unhappiness-terrain] [ ask patches [set pcolor turquoise] ]
  ]
  ;; report the average unhappiness value
  ifelse actual-num-consumers > 0
  [report mean unhappiness-values-list]
  [report 0]
end

;; sets the color of the customers and houses as per the current unhappiness value in the population distribution
to set-customer-unhappiness-colors [std-dev mean-val]
  ask persons [
    ;; get the unhappiness value of the customer
    let person-unhappiness store-unhappiness target-store-id
    ;; red color for person with unhappiness more than mean value + standard-deviation
    (ifelse (person-unhappiness >= mean-val + std-dev) [set color red]
      ;; orange color for person with unhappiness above mean
    (person-unhappiness >= mean-val) [set color orange]
      ;; sky color for unhappiness below the mean-value - standard deviation, the happiest of the customers
    (person-unhappiness < mean-val - std-dev) [set color sky]
      ;; rest green color for below mean
    [ set color green])
    ;; set the same color for the house as of the associated customer
    ask house house-id [set color [color] of myself]
  ]
end

;; set the color of the patches as per the color of the nearest house (denoting the unhappiness value of customer residing in the region)
to show-unhappiness-terrain
  ask patches [
    ;; local variable to store the new color value of the patch
    let my-new-color 0
    ;; get the color of the nearest house to the patch
    ask houses with-min [distance myself]
    [
      set my-new-color [color] of self
    ]
    ;; set a lighter tone for the patch color
    set pcolor my-new-color + 2
  ]
end

;; reports the first time, when all of the customers have experienced their target store atleast once
to-report unhappiness-start-time
  report start-time
end


to setup
  clear-all
  ask patches [set pcolor turquoise]
  ;; initialize the unhappiness start-time to 0
  set start-time 0
  ;; setup the stores in the grid
  setup-stores-initial

  ;; setup the persons in the grid
  setup-homes-and-consumers

  ;; initialize the lists and initial state variables for the persons
  ask persons [ initialize-person-lists-and-state ]
  reset-ticks
end

;; if we want keep the location of stores same, but want to try a new location of consumers
to setup-consumers
  clear-all-plots
  ask patches [set pcolor turquoise]
  ;; initialize the unhappiness start-time to 0
  set start-time 0
  ;; kill all the persons and houses and generate a new set of customers and houses with different location
  ask persons [die]
  ask houses [die]
  ;; reset the quantity in the stores
  ask stores [set quantity store-capacity]
  ;; generate the customer and house pairs in the grid
  setup-homes-and-consumers
  ;; intialize the lists for each consumer
  ask persons [initialize-person-lists-and-state]
  reset-ticks
end

;; if we want to keep the same location of consumers, but want to try a new location for shops
to setup-stores
  clear-all-plots
  ;; initialize the unhappiness start-time to 0
  set start-time 0
  ;; set a unifrom random location of the stores
  ;; intially the inventory (quantity) at each store is full
  ask stores [
    setxy random-xcor random-ycor
    set quantity store-capacity
  ]
  ;; initialize the list and state variables of the customers
  ask persons [initialize-person-lists-and-state]
  reset-ticks
end

;; create the stores in the grid
to setup-stores-initial
    create-stores num-stores [
    setxy random-xcor random-ycor
    set shape "building store"
    set color yellow
    set size 1.5
    ;; intially the inventory (quantity) at each store is full
    set quantity store-capacity
  ]
end

;; move each of the store to the average coordinate of its current customers
to move-shops-to-customers
  ask stores [
    let num-customers count link-neighbors
    let mean-x-customer mean [xcor] of link-neighbors
    let mean-y-customer mean [ycor] of link-neighbors
    ;; move the store to the mean coordian
    setxy mean-x-customer mean-y-customer
  ]
  reset-consumers
end

;; move the store to the unhappiness weighted mean coordinate of its current customers
to move-shops-to-unhappiness-customers
  ask stores [
    let current-store-id [who] of self
    ;; the total unhappiness of current store customers
    let total-customer-unhappiness 0
    let mean-unhappiness-x-customer 0
    let mean-unhappiness-y-customer 0
    ;; ask the houses of its customers
    ask link-neighbors[
      let current-house-id [who] of self
      ;; get the customer who is associated with this house
      let person-here persons with [house-id = current-house-id]
      ;; ask the customer associated with this house
      ask person-here
      [
        let unhappiness-value store-unhappiness current-store-id
        ;; take the weigted x and y coordinate of the customer
        set mean-unhappiness-x-customer  mean-unhappiness-x-customer + (xcor * unhappiness-value)
        set mean-unhappiness-y-customer  mean-unhappiness-y-customer + (ycor * unhappiness-value)
        set total-customer-unhappiness total-customer-unhappiness + unhappiness-value
      ]
    ]
    ;; finally get the unhappiness weighted customer coordinates
    set mean-unhappiness-x-customer mean-unhappiness-x-customer / total-customer-unhappiness
    set mean-unhappiness-y-customer mean-unhappiness-y-customer / total-customer-unhappiness

    setxy mean-unhappiness-x-customer mean-unhappiness-y-customer
  ]
  reset-consumers
end

;; reset the customer variables and set the location to its home location
;; reset the quantity of the store as well
to reset-consumers
  clear-all-plots
  set start-time 0
  ;; reset the quantity in the stores
  ask stores [set quantity store-capacity]
  ;; intialize the lists for each consumer
  ask persons [initialize-person-lists-and-state]
  reset-ticks
end

;; create the customer and the associated house pairs on the grid
to setup-homes-and-consumers
    ;; create persons/consumers
    create-persons num-consumers [
    setxy random-xcor random-ycor
    ;; the initial location will be the location of person's home as well
    set home-xcor xcor
    set home-ycor ycor
    set shape "person"
    set color blue
    ;; store the current person in local variable parent
    let parent self
    ;; ask the patch on the where the person is standing to create an agent of breed house
    ask patch-here[
      sprout-houses 1 [
        set shape "house"
        set color blue
        ;; store the who number of house in local variable of the associated customer
        let current-house-id who
        ;; ask the customer stored as parent, to set its house-id to the who number of this house
        ask parent [
          set house-id current-house-id
        ]
      ]
    ]
  ]
  ;; reset the unhappiness values of the population list
  set unhappiness-values-list []
end

;; initializes the calling customer's lists and initial-state
to initialize-person-lists-and-state
    ;; calculate the travelling distance to each store
    set store-travel-time-list n-values num-stores distance-from-store
    ;; initialize visit count for each store to zero
    set store-visit-count-list n-values num-stores [i -> 0]
    ;; initialize the total waiting time for each store to zero
    set store-total-waiting-time-list n-values num-stores [i -> 0]
    ;; set the initial state of person as at home
    set at-home? true
    ;; initialize the rest-time of the person
    set rest-time random person-max-rest-time
    ;; place the person at associated house
    setxy home-xcor home-ycor
end

;; reports the euclidean distance from calling customer's house to the store with given who number as store-id
to-report distance-from-store [store-id]
  let distance-to-store 0
  ;; store the distance from home to store in distance-to-store variable
  ask house house-id [ set distance-to-store distance store store-id]
  ;; resport the distance
  report distance-to-store
end

;; reports the best store option for the calling customer depending on the unhappiness values
;; unhappiness for a store = distance-from-home + (average waiting time at the store)
to-report best-store-option
  ;; prepare the unhappiness value for each store and store in a local variable - list
  let store-unhappiness-list n-values num-stores store-unhappiness
  ;; find the minimum unhappiness value
  let min-unhappiness min store-unhappiness-list
  ;; report the index (who of the store in this case) with the given minimum unhappiness-value
  report position min-unhappiness store-unhappiness-list
end

;; reports the unhappiness for a particular store for the calling customer
;; unhappiness for a store = distance-from-home + (average waiting time at the store)
to-report store-unhappiness [store-id]
  ;; initally unhappiness = distance-from-home
  let unhappiness item store-id store-travel-time-list
  ;; add average waiting time for the store to the unhappiness value, if the number of visits to the store > 0
  if (item store-id store-visit-count-list) > 0 [
    set unhappiness unhappiness + (item store-id store-total-waiting-time-list) / (item store-id store-visit-count-list)
  ]
  ;; report the unhappiness value
  report unhappiness
end

;; display the inner influence circles for each store
;; every house strictly inside this radius is a customer to calling store only, not to any other store
to-report inner-influence-radius
  let current-store self
  ;; the list of the houses of the current customers of the store
  let current-store-link-neighbors link-neighbors
  ;; report the distance of the nearest home which is a customer of another store
  report min [distance current-store] of houses with [not member? self current-store-link-neighbors]
end

;; the distance of the home of the farthest customer for the store
;; this is the radius of the star network of this
to-report outer-influence-radius
  let current-store self
  ;; report the distance of the home of the farthest customer
  report max [distance current-store] of link-neighbors
end

to go
  ;; perform the following actions for each customer
  ask persons[run-consumer-finite-state-logic]
  ;; increase the quantity in each store by the supply-rate
  ask stores [
    set quantity quantity + supply-rate
    ;; if the quantity of the store goes above its capacity, set quantity to exactly its capacity
    if quantity > store-capacity [set quantity store-capacity]
  ]
  ;; clear all the previous drawings
  clear-drawing
  ;; display inner-circles if inner-influence-circles? switch is ON
  if inner-influence-circles? [display-inner-influence-circles]
  ;; display outer-circles if outer-influence-circles? switch is ON
  if outer-influence-circles? [display-outer-influence-circles]
  ;; hide links if show-links? switch if OFF
  ask links [set hidden? not show-links?]
  tick
end

;; contains the logic of customer for the go procedure
to run-consumer-finite-state-logic
  ;; fetch the target store via the target-store-id
    let target-store store target-store-id
    (ifelse
      ;; if the person is currently inside home
      at-home? = true [
        ;; if the person at home doesn't need rest
        ifelse rest-time = 0 [
          ;; set demand to a random value over the range
          set demand random person-max-demand

          let person-home house house-id
          ;; kill the link with previous target-store
          ask target-store[
            let link-with-person-home link-with person-home
            if link-with-person-home != nobody [ ask link-with-person-home [die] ]
          ]
          ;; calculate the best store, as per the unhappiness values
          set target-store-id best-store-option
          ;; update the target-store local variable as well
          set target-store store target-store-id
          ;; create a link between updated target-store and home of the person
          ask target-store[create-link-with  person-home ask link-with person-home [set color red]]

          ;; set at-home to false, set travelling-to-store to true
          set at-home? false
          set travelling-to-store? true]
        ;; otherwise if the person requires rest, then person stays at home and rest-time is decreased by 1
        [set rest-time rest-time - 1]
      ]
      ;; if the person is on its way to store
      travelling-to-store? = true[
          ;; if reached the store (that is when distance from store is less or equal to one)
          ifelse distance target-store <= 1[
          ;; move inside the shop
          setxy [xcor] of target-store [ycor] of target-store
          ;; travelling-to-store is set to false and at-store set to true
            set travelling-to-store? false
            set at-store? true ]
        ;; if still far from store, then face towards the store and move 1 step forward
            [ face target-store fd 1]
       ]
      ;; if the person is inside store
      at-store? = true [
        ;; try to get the demand fulfilled
        ifelse demand > 0 [
         ;; initialize the wait-time if the quantity at store is less than the demand of person
        if [quantity] of target-store < demand [ set wait-time floor (( demand - [quantity] of target-store ) / supply-rate) ]
          ;; increase the store visit count and total waiting time for the store accordingly
          set store-visit-count-list replace-item target-store-id store-visit-count-list (item target-store-id store-visit-count-list + 1)
          set store-total-waiting-time-list replace-item target-store-id store-total-waiting-time-list (item target-store-id store-total-waiting-time-list + wait-time)
          let person-demand demand
          ;; update the quantity at the store and the demand is fulfilled, set it to zero
          ask target-store[set quantity quantity - person-demand]
          set demand 0]
        [
        ;; if waiting is over
        ifelse wait-time = 0 [
            ;; move outside the store and get on way back to home
          set at-store? false
          set travelling-to-home? true]
          ;; if still needs to wait, decrease wait-time by 1
          [set wait-time wait-time - 1]]
      ]
      ;; if is travelling to home
      travelling-to-home? = true [
        ;; if reached near home
        ifelse (distancexy home-xcor home-ycor) <= 1 [
          ;; move inside home and set travelling to home as false
          setxy home-xcor home-ycor
          set travelling-to-home? false
          ;; set a rest-time uniform random in range 1 to person-max-rest-time
          set rest-time random person-max-rest-time
          set at-home? true ]
        ;; if far from home, face towards home and move forward one step
        [ facexy home-xcor home-ycor fd 1]]
     )
end

;; display the inner influence circle for the calling store-agent
to display-inner-influence-circles
  ask stores [
    let distance-influence inner-influence-radius
    stamp-circle "circle" distance-influence
  ]
end

;; display the outer influence circles for the calling store-agent
to display-outer-influence-circles
    ask stores [
    let distance-influence outer-influence-radius
    stamp-circle "circle 3" distance-influence
  ]
end

;; to draw the influence circles on the grid
;; it stamps a shape of given size as "distance-influence * 2"
;; parameters are drawing-shape, the shape of the stamp, distance-influence, the radius in case of circles
to stamp-circle [drawing-shape distance-influence]
  ;; store-xcor and store-ycor stores the coordinates of the current store
  let store-xcor xcor
  let store-ycor ycor
  ;; ask the patch-here to create a circle turle, then stamp and then die
  ask patch-here[
    sprout 1[
      ;; set the center of the turtle at the store
      setxy store-xcor store-ycor
      set shape drawing-shape
      ;; lighten the shade of the color
      set color color + 1
      ;; make the turtle half transparent
      set color lput 125 extract-rgb color
      ;; set appropriate turtle-size for the
      set size 2 * ( distance-influence )
      ;; mark the color impression of the turtle and make it die
      stamp
      die
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
299
24
832
558
-1
-1
25.0
1
10
1
1
1
0
0
0
1
-10
10
-10
10
0
0
1
ticks
30.0

SLIDER
56
25
228
58
num-consumers
num-consumers
1
300
260.0
1
1
NIL
HORIZONTAL

SLIDER
55
63
227
96
num-stores
num-stores
1
10
3.0
1
1
NIL
HORIZONTAL

BUTTON
11
334
74
367
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
94
381
157
414
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
55
172
227
205
supply-rate
supply-rate
1
100
5.0
1
1
NIL
HORIZONTAL

SLIDER
55
133
227
166
store-capacity
store-capacity
1
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
54
232
226
265
person-max-demand
person-max-demand
1
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
53
274
226
307
person-max-rest-time
person-max-rest-time
0
50
10.0
1
1
NIL
HORIZONTAL

SWITCH
39
446
220
479
inner-influence-circles?
inner-influence-circles?
1
1
-1000

SWITCH
39
483
220
516
outer-influence-circles?
outer-influence-circles?
1
1
-1000

SWITCH
72
520
190
553
show-links?
show-links?
0
1
-1000

PLOT
851
39
1174
233
num-customers
store-id
number-customers
0.0
1.0
0.0
10.0
true
false
"set-plot-x-range 0 (num-stores)" ""
PENS
"default" 1.0 1 -16777216 true "" "set-plot-x-range 0 (num-stores)\nhistogram [target-store-id] of persons"

PLOT
850
262
1314
496
average-unhappiness
ticks
average-unhappiness
0.0
10.0
0.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot average-unhappiness"

BUTTON
16
380
83
413
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
1325
416
1466
461
unhappiness-start-time
unhappiness-start-time
17
1
11

BUTTON
80
334
185
367
setup-consumers
setup-consumers
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
191
335
293
368
setup-stores
setup-stores
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
1194
36
1492
233
num-customers-wating
store-id
num-customers-waiting
0.0
10.0
0.0
10.0
true
false
"set-plot-x-range 0 (num-stores)" ""
PENS
"default" 1.0 1 -16777216 true "" "set-plot-x-range 0 (num-stores)\nhistogram [target-store-id] of persons with [at-store? = true]"

SWITCH
33
556
238
589
show-unhappiness-terrain?
show-unhappiness-terrain?
0
1
-1000

BUTTON
166
381
290
414
reset-consumers
reset-consumers
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
1322
266
1455
311
average-unhappiness
average-unhappiness
2
1
11

BUTTON
28
623
237
656
move-shops-to-customers
move-shops-to-customers
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
493
587
672
632
std-dev inner-influence-radius
standard-deviation [inner-influence-radius] of stores
2
1
11

MONITOR
305
587
471
632
mean inner-influence-radius
mean [inner-influence-radius] of stores
2
1
11

MONITOR
304
641
473
686
mean outer-influence-radius
mean [outer-influence-radius] of stores
2
1
11

MONITOR
493
642
674
687
std-dev outer-influence-radius
standard-deviation [outer-influence-radius] of stores
2
1
11

TEXTBOX
31
603
198
623
Store Location Optimization
11
104.0
1

TEXTBOX
41
427
191
445
Visual Tools
11
104.0
1

TEXTBOX
59
10
209
28
Agent Population\n
11
104.0
1

TEXTBOX
55
116
205
134
Store Parameters
11
104.0
1

TEXTBOX
55
217
205
235
Consumer Parameters
11
104.0
1

TEXTBOX
14
318
164
336
Procedure Buttons
11
104.0
1

TEXTBOX
306
569
488
597
Inner and Outer Influence Circles
11
104.0
1

TEXTBOX
851
18
1044
46
Current Consumers of Each Store
11
114.0
1

TEXTBOX
1197
15
1482
43
Number of Consumers Waiting at each Store
11
114.0
1

TEXTBOX
852
242
1123
270
Average Unhappiness over the time
11
114.0
1

TEXTBOX
1328
367
1478
409
The First Time at which each consumer has experienced its target shop atleast once
11
114.0
1

BUTTON
6
660
288
693
move-shops-to-unhappiness-customers
move-shops-to-unhappiness-customers
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

@#$#@#$#@
# Overview
## Purpose and Patterns
The model is designed to explore questions about how customers choose which stores to visit given a setting of the location of the stores and customers in a grid. A customer is assumed to favor the store which is near to home location and the waiting time at the store should also be low. Using this, we assume that the "unhappiness" of a customer depends on a linear combination of the travelling time from home to the store and average waiting time at the store over the past visits. At first the customer prefer the store nearest, but as the waiting time grows due to crowd, he/she looks for alternatives and then makes the best decision based on it. We can think of the store selection as a form of "Minority Game", but the with a location factor involved for the agents. 

In patterns, we hope to see the trend of the average "unhappiness" of the population over the time, as the customers regurarly make decisions to choose the best store based on the past experiences and distance to store's location. The distribution of the customers at each store and "unhappiness" distribution of the population is also to be observed to draw conclusions about how biased the distribution is on the location of stores. Finally, we observer the pattern of unhappiness over the grid terrain, to conclude the effect of proximity of competing stores to uneven unhappiness distribution around the stores location.

## Entities, State variables and scales
The model has mainly the two entities Stores and Customers; the other minor being House entities.
### Customers
- ID and location of the home on grid (Each customer is associated with a unique home on the grid))
- Store visit count list (A list which maintains that how many times the customer have visited each store in the past)
- Store total waiting time list (A list which maintains that how much total time customer have spent waiting at each store)
- Rest time (A random value in range(1, MaxRestTime) to decide for how much time, a customer should rest at home before next visit)
- Wait time (Based on the current crowd and quantity, the time for which the customer must wait at store to get the request fulfilled)
- Demand (A random value in range(1, MaxDemanTime) to decide what is the required quantity by the customer while visit to a store)
- Target store ID (On starting a visit to store, the customer decides the best available store)
- A state variable or set of state variable to store the current state of person whose value can be one of ["at home", "travelling to store", "at store", "travelling to home"]
### Stores
- Quantity (The current quantity available at the store, can be negative if the crowd at the store is already waiting at the store for a placed order)
### Houses 
Marks the home location of the unique customer associated with it.

### Scales and Limits
- The landscape is 21 X 21 patches in size with no wrapping at its edges.
- There is no time limit to the simulation. The user has to manually stop the go forever button.
### Global Variables
- Supply rate (same for each store)
- Max quantity of store (same for each store)
- Max Demand of customers for a visit
- Max Rest time of customers after a visit

## How to Use (Model Interface)
### Model Parameters
**Agent Population**   
- num-consumers: The number of customers in the grid
- num-stores: The number of stores in the grid   
    
**Store Parameters**
- store-capacity: The maximum quantity any store can hold
- supply-rate: The rate at which the quantity is replenished    
 
**Customer Parameters**
- person-max-demand (The maximum required quantity by the customer on a visit)
- person-max-rest-time (The maximum time a customer rests at home before next visit)

### Procedure Buttons
- Setup: Setup and initialise the whole with the new location of stores and customers
- setup-consumers: Vary the location of customers keeping the location of stores same and reset the time and all entities variables
- setup-stores: Vary the location of stores keeping the location of customers same and reset the customer variables
- go-once: Run the model for a single time step
- go: Run the model forever
- reset-consumers: Reset the customer variables with the time and plots. The location of stores and customers do not change

### Visual Tools
- inner-influence-circles? : Whether to display the inner influence circles for each store
- outer-influence-circles? : Whether to diplay the outer influence circles for each store
- show-links? : Whether to display the links between stores and its current customer home locations
- show-unhappiness-terrain? : Whether to color the patches as per the unhappiness distribution over the grid

### Plots
- num-customers(Histogram): It shows the current number of customers for each store at the current time
- num-customers-waiting(Histogram): It shows the number of customers at each store which are currently waiting at the store for their demand to be fulfilled
- average-unhappiness(Line plot): Shows the average unhappiness of the customers value over the time
- unhappiness-distribution(Histogram): Show the distribution of the unhappiness over the customers population

### Monitors
- average-unhappiness: Shows the current average unhappiness of the customer population
- unhappiness-start-time: The first time when each customer has visited its target store at least once
- std-dev unhappiness: The standard deviation of the unhappiness values of the customer population
- mean inner-influence-radius: The mean of the inner influence radius of the stores
- std-dev inner-influence-radius: Standard deviation of the inner influence radius of the stores
- mean outer-influence-radius: The mean of the outer influence radius of the stores
- std-dev outer-influence-radius: Standard deviation of the outer influence radius of the stores

### Store Location Optimization
- move-shops-towards-customers: Moves the stores to the mean coordinate of their current customers home location and reset the customers and plots
- move-shops-towards-unhappiness-customers: Moves the stores the unhappiness weighted mean of their current customers home location and reset the customers and plots

## Process Overview and Scheduling
## Processes
### Logic for Customer
In a single discrete time unit, the execution in the flow chart of customer reaches a "flowchart state" (can be same as previous) from the previous "flowchart state".
The figure below explains the cycle of actions for the customer.
![](customer_logic.png)
- The customer performs actions as per the current state in the flowchart. Accordingly, it moves or sets some state variables such as restTime, waitTime or completes an order transaction at a store in a single time unit.
- A customer is initially at the home location with a random RestTime between 1 and Max Rest time. 
- With every tick, the rest time of the customer decreases by 1, if the customer is at home.
- When the rest time becomes zero, the customer makes a decision to which store to visit based on the store with least unhappiness value. (Unhappiness of a customer with respect to a store = Travelling distance from store + Average waiting time at the store)
- The state of the customer changes from "at home" to "travelling to store" and starts moving towards target-store.
- On reaching the target store, the state of customer is changed to "at store". On visit, the customer updates the quantity at store, the demand value. In case the quantity at the store is less the customer's demand, then the customer has to wait for the given time.
WaitTime = (Customer's demand - Quantity at Store) / Supply Rate
- Once the waiting time is over, the customer starts to move towards home location. 
- Once reaching the home, the restTime is set in Random(1 , max-restTime).
- When the restTime becomes zero, the customer repeats the above steps starting from making a decision to which store to visit based on unhappiness value for each store for the given customer.

### Logic for Store
- At each time unit, the quantity of each store increases by the supply rate. 
- If the quantity is more than the capacity of the store, quantity is set to the capacity.
![](store_logic.png)

## Schedules
- The order in which the customers and stores respectively perform actions is unimportant because there are no direct interactions.
- In our model implementation, first the customer agentset perform the action described in Customer Flowchart. The order of customers execution is random.
- After the customers have performed their respective actions, the stores update the quantity as per the supply rate. The order of stores execution is random and is unimportant.
- In each time unit, customers and stores execute one step in the flowchart moving to a new state (which may be the same as previous state).
- The visual outputs use synchronous updating such as the influence circles are updated at the end of the time step.
- There is no time limit for a run, so the user has to manually stop the go procedure if required.

# Design Concepts
### Basic Principles
**How Customers decide the Stores**   
There are multiple factors by which a customer decide which store to visit such as travelling distance to the store, waiting time at the store, price, customer service, personal preferences and many more. Here in our models we have taken into consideration the first two factors that are the travelling distance to the store and waiting time at the store.

**'Unhappiness' value for a store of a customer**
Here we define the unhappiness value, as the unsatisfaction of the customer for the specific store. We take it as a linear combination of the travelling distance and average waiting time at the store. The customer keeps track of the average waiting at the store of his/her past visits to it. 
So we define unhappiness value of a store for a customer as:
unhappiness  value = Travelling  Distance + Average  Waiting  Time  for  the   customer  in   past   visits
In case the customer has not visited a store once, the average waiting time for that store is taken to be 0. THe unhappiness value of the customer for calculating the average unhappiness and unhappiness distribution is taken to be the unappiness associated with its current target-store.

**Deciding target store for a customer**
The store that the customer will be visiting, will be the store which gives the minimum unhappiness value for the customer.

### Emergence
The key outcome of the model are the distribution of customers over the stores and the avaerage unhappiness value over the time period. Over the simulation, the customers would try to avoid the crowded stores with a preference of the location. The choices of the customers made over the time is based on the prediction from past experiences. The run of the model can be generally classified into three time phases - the initial phase, decision phase and the stable phase. In the initial phase, the customers are expected to prefer just the nearby store. But as certain store gets crowded, the customers of the specific store look for better alternatives stores. In the decision phase, the customer distribution keeps on changing as per the decision and crowd at the store and the average unhappiness values goes over a period of ups and downs. Finally after a long run, in stable phase, the average unhappiness value is expected to stabilise as the customers have made choice preferences based on the past visits to each store.

### Adaptation
**Deciding the target store**
When the customer wants to visit a store, it has to make the decision that which store it must visit. The choices available to each customer are all the stores on the grid. The decision is modeled as direct objective seeking, explained below.

### Objectives
**Direct Objective Seeking**
When deciding which store to visit, the customer chooses the store which gives the minimum unhappiness value to the customer. The unhappiness value as mentioned before is the sum of travelling time (taken as travelling distance here) and average waiting time at the store in past visits. In case the person has not visited the store before, then the waiting time for that store is taken to be zero. Each customer maintains the visit counts for each store, hence gets the average waiting time as (Total waiting time for the customer in past visits to the store) / Number of visits.

**To minimize the unhappiness value**
Each customer is making choices seeking to reduce its unhappiness value over the time. Due to each customer continuosly looking for better store alternatives, some stores over time get crowded forcing its existing customers to prefer other store alternatives. 

### Learning
The average waiting time and thus the decision to choose the store based on minimum unhappiness value depends on the past experience of the customer at the store. As part of past experience, a customer maintains visit count to each store and the total waiting for the customer at the store in past visits, using this memory the customer calculates the average waiting time factor in unhappiness.
In case the customer has not visited the store at least once, the initial waiting time is assumed to be 0 for the customer. 

### Prediction
**Implicit Prediction**
The adaptive behaviour of the customers is based on implicit prediction that choosing the store with minimum unhappiness value is likely to minimize the unhappiness value of the customer. 

### Sensing
**Customers**
- Customers store the location and ID of the unique house agent associated with it.
- When a customer reaches it target store, it will know the current quantity (inventory) available at the store. It updates its waiting time at the store based on the available quantity at the store and its demand.
- Customer knows the location of each store and its distance from its home location.
- Customers sense the crowding at a store indirectly via the calculated waiting time at the store.
- Customer update the quantity of the target store upon reaching as per the its current demand.
- Customer all keeps track of its past experiences for each store.

**Stores**
- Stores can sense the number of customers associated with the store using the links between the stores and houses of current customers.
- Using the links connecting store and house locations of its current customers, the store can detect the distances of the current customers to its location.

**Houses**
- Each House is associated with a unique customer.

### Interaction
**Mediated Interaction between customers**
The customers interaction is mediated by the resource that they compete for, each customer is seeking to minimize its unhappiness value. A customer thus prefer less crowded stores over the more crowded ones, as it has to wait longer due to crowding at a store. As the number of stores and supply-rate is limited, the customers compete over the choices of the stores to get better experience.

**Direct Interaction between stores and customers**
When a customer reaches its target store, it updates the quantity available at the store as per its current demand. When the quantity available at the store is less than the demand, then the customer has to wait at the store for time = (demand - quantity at store) / supply-rate.

### Stochasticity
1. Uniform Random location of the customers and stores: In the beginning the customers at their associated home and the stores are located randomly over the grid.
2. RestTime: The time for which a customer must rest at home before going for next visit to store is random uniform in range 1 to person-max-rest-time.
3. Demand: The quantity required by a customer during a store visit is random uniform in range 1 to person-max-demand.
4. The order in which customers execute is random, to avoid bias on different customers arriving at same time at a store.

### Collectives
There are no major collectives in the model. 
A minor case of collective in the model is the group of customers that belong to the same store. Their behaviour such as the demand value, rest time and distance from the store affects the other individual customers of the same store. Based on the arrival time and waiting at the store of the customer group, the individual customers may choose alternative stores.

### Observation
The models purpose is study the customer distribution and average unhappiness over the time. Also, the unhappiness distribution over the terrain and customer population. We also observe the influence of different stores based on the current setting of their location and proximity to other competing stores. 

**Graphical Output**
- The color of the customer agent and its associated house shows where does its current unhappiness value lie in the unhappiness distribution of the population.
    - Blue (Happiest ones) -- if the unhappiness value is below the mean - standard deviation of the distribution
    - Green (Happy) -- else if the unhappiness value is below the mean of the distribution
    - Orange (Unhappy) -- else if the unhappiness value is above the mean of the distribution
    - Red (Unhappiest ones) -- if the unhappiness value is above the mean + standard deviation of the distribution
- The color of the patch shows a lighter tone of the color of the nearest house to the patch.
- Links between the stores and houses, shows the location of the house of the current customers of the store.
- The inner influence circles (every house inside the circle is the customer of the specified store only) and outer influence cirlces (show the distance from which the farthest customer is coming to the store)   

 
**Plots**
- The distribution of customers over the stores (Histogram): Shows how many customers each store has at the current time
- The distribution of waiting customers over the stores (Histogram): Shows how many customers are currently waiting at each store
- Average Unhappiness (Line): The line plot shows how the average unhappiness varies over the complete simulation
- Unhappiness distribution over population (Histogram): Shows the count distribution of the unhappiness values over the population
**Output Monitors**
- Average Unhappiness value: The mean value of unhappiness value over the population (The unhappiness value for a customer is its unhappiness associated with its current target store)
- Standard Deviation of Unhappiness in Population: The standard deviation of the unhappiness values over the population
- Mean of inner influence radius: The mean of the inner influence radius of the stores
- Unhappiness Start Time: The first time at which each customers has visited at least one of the stores since the beginning of the simulation
- Standard Deviation of inner influence radius: The standard deviation of the radius of inner influence of the stores
- Mean of outer influence radius
- Standard Deviation of outer influence radius
   
# Details
## Initialization
- The location of the stores and customers is random over the grid.
- The intital quantity at each store is the store-capacity. The inventory at stores are assumed to be full in the beginning.
- The customers are located at the same location as the associated unique house.
- The visit count for each customer to each store is initialised to zero and the total waiting time for each store is zero as well.
- All the customers at the start are in "at home" state with a uniform random rest time in range 1 to max-person-rest-time.
- The color of the patches is initialized to some same color.
- The house id and house coordinates of the customer is set according to the house uniquely associated with the customer.

## Input data
The model does not use input data to represent the time-varying processes.

## Submodels
**Parameters to the Model**
- Supply rate (same for each store)
- Max quantity of store (same for each store)
- Max Demand of customers for a visit
- Max Rest time of customers after a visit

**Generating the house and customer pairs**
- For generating the house and customer pairs, first the customers are generated uniform random in the grid.
- Then each customer sprouts a house and the customer then stores its ID and location.

**Average Unhappiness Reporter**
- For calculating the average unhappiness, only those customers were considered which have at least visited the target store once.
- This was done to avoid unexpected trends in average unhappiness, due to many customers which are visiting the store for first time and the average waiting for that visit was thus taken as zero.
- Thus the mean is taken for the unhappiness values of the customers who have visited the current target store at least once.

**Calculating the Inner and Outer Influence circles**
- For the inner influence radius, the distance of the closest customer to the given store who is currently customer of any other store is taken.
- The outer influence raius, is the distance of the farthest current customers of the given store.

**Moving stores to Mean Location of current customers with and without unhappiness weights**
- We used these procedures to see if the store location can be optmized in the given distribution of customers over the grid.
- In attempt 1, after running the model for a long time, we can change the location of each stores as the mean coordinate of its current customers.
- In attempt 2, we move the stores to the unhappiness weighted mean coordinate of its current customers. That is x-coordinate = ( Sum(Unhappines of customer * x-coordinate of the customer) )/ (Number of customers of the given store). Similarly y-coordinate = (Sum(Unhappines of customer * y-coordinate of the customer)) / (Number of customers of the given store).

## Extending the Model
- In our model, we have only assumed a single type of commodity being sold by the stores. We can extend the model by having more than one commodities with different supply-rates for each commodity.
- Further the stores are assumed to have same suppy-rate, we can have different supply-rate for different stores. Different supply rates would signify the different production or procurement rate at the stores. Such as restaurent with larger cooking staff will have higher supply-rate of the quantities.
- The customer in our model, keeps track of all the past visits to each stores. We can implement a limit to the memory of the customer, such that the customer only remembers about the past 'n' visits.
- There can be customers influencing the customers that reside near their home location about the preference for the stores.

## Related Models
**Minority Game**
- At each step the minority agent group wins and their strategies are rewarded.
- We can think of our model as minority game on different stores, each person tries to avoid crowded stores but there is a cost factor of distance involved.

**El Farol**
- The problem also deals with reduced customer satisfaction when the customer finds the bar crowded during a visit.
- The agents here work on strategies seeking to visit the bar when it is uncrowded.

**Kolkata Paise Restaurant Problem**
- It is a variant of El Farol problem where customers have to choose between many restaurants.
- Each restaurant serves a limited number of customers and rest are not serviced.
- In our model as well, each customer has to decide which store to visit, but crowded stores result in long waiting times rather than denial of service.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

building store
false
0
Rectangle -7500403 true true 30 45 45 240
Rectangle -16777216 false false 30 45 45 165
Rectangle -7500403 true true 15 165 285 255
Rectangle -16777216 true false 120 195 180 255
Line -7500403 true 150 195 150 255
Rectangle -16777216 true false 30 180 105 240
Rectangle -16777216 true false 195 180 270 240
Line -16777216 false 0 165 300 165
Polygon -7500403 true true 0 165 45 135 60 90 240 90 255 135 300 165
Rectangle -7500403 true true 0 0 75 45
Rectangle -16777216 false false 0 0 75 45

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

circle 3
true
0
Circle -7500403 false true 0 0 300

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="unhappiness vs supply-rate and demand" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1300"/>
    <metric>average-unhappiness</metric>
    <enumeratedValueSet variable="person-max-rest-time">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="person-max-demand" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-stores">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="supply-rate" first="1" step="2" last="21"/>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="std-dev customers vs supply-rate and num-stores" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1300"/>
    <metric>standard-deviation [count link-neighbors] of stores</metric>
    <enumeratedValueSet variable="person-max-rest-time">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="person-max-demand">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="100"/>
    </enumeratedValueSet>
    <steppedValueSet variable="num-stores" first="2" step="1" last="10"/>
    <steppedValueSet variable="supply-rate" first="1" step="2" last="25"/>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="unhappiness vs max demand and max rest time" repetitions="15" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1500"/>
    <metric>average-unhappiness</metric>
    <steppedValueSet variable="person-max-rest-time" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="person-max-demand" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-stores">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="std-dev vs max demand and max rest time" repetitions="15" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1500"/>
    <metric>standard-deviation [count link-neighbors] of stores</metric>
    <steppedValueSet variable="person-max-rest-time" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="person-max-demand" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-stores">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply-rate">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="unhappiness vs max demand and supply" repetitions="8" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1300"/>
    <metric>average-unhappiness</metric>
    <enumeratedValueSet variable="person-max-rest-time">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="person-max-demand" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-stores">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="supply-rate" first="1" step="2" last="21"/>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="std dev vs max demand and supply" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1300"/>
    <metric>standard-deviation [count link-neighbors] of stores</metric>
    <enumeratedValueSet variable="person-max-rest-time">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="show-unhappiness-terrain?">
      <value value="false"/>
    </enumeratedValueSet>
    <steppedValueSet variable="person-max-demand" first="1" step="4" last="40"/>
    <enumeratedValueSet variable="inner-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-consumers">
      <value value="260"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="outer-influence-circles?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="store-capacity">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-stores">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="supply-rate" first="1" step="2" last="15"/>
    <enumeratedValueSet variable="show-links?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
