patches-own[
  mu ; Mean of the logisitic production function
  beta ; Steepness of the production function
  contribution-level ;; How much has been contributed
  progress ;; Ratio of project complete
  topics ;; Distribution of topics that a patch is "about"
  pindex ;; Index of the patch
  ]
globals [
  multiplier ; How much the parent patch multiplies the interest
  patch-gini ; Stores the gini of patch contributions
  turtle-gini ; Stores the gini of turtles
  patch-progress-gini ; Gini of patch progress
  ]
turtles-own[
  interest ;; Baseline interest in projects
  resources ;; How much they have to contribute each tick
  contributions-made ;; How much they have contributed in total
  top-project ;; Current patch with highest utility
  topic-interests ;; Distribution of topic interests
  similarity-list ;; List of cosine similarity to each patch
  ]

to setup
  ca
  ;; Set the number of topics to use for topic-based
  let num-topics 20
  set-multiplier 1

  ;; Create the production functions for the patches
  ask patches[
    set mu random-normal global-mu 1
    set beta abs random-normal global-beta 1
    update-patch-progress
  ]
  ;; If topic-based, then create topics
  if topic-based? [
    setup-patch-topics num-topics
    ]
  ;; Make the new turtles
  make-new-turtles num-turtles num-topics
  reset-ticks
end

to set-multiplier [x]
  set multiplier x
end

to make-new-turtles [n num-topics]
  repeat n [
    crt 1 [
      ;; Place turtle randomly
       setxy random-xcor random-ycor
       ;; Give it resources
       set resources abs random-normal .2 .05
       ;set resources random-float 3
       ;; And interest
       set interest random-float max-interest
       ;; Create topics, if necessary
       if topic-based? [
         set topic-interests make-topics num-topics
         make-similarity-list
       ]
    ]
  ]
end


to go
  ask turtles[
    ; Find the best project
    find-top-project
    ; Figure out whether to contribute to it
    contribute-to-project
    ;ask-to-die
  ]
  ask patches[
    recolor-patches
  ]
  update-ginis
  tick
end

; Creates topics for the patches, and creates an index to access the similarity matrix
to setup-patch-topics [num-topics]
  ask patches [
    ; Create random distribution of topics
    set topics make-topics num-topics
  ]
  ;; Create an index so that we can access the similarity list
  let n 0
  foreach sort patches [
    ask ? [
      set pindex n
      set n n + 1
    ]
  ]
end

; Turtle function; creates a turtle-patch similarity matrix
to make-similarity-list
  set similarity-list []
  foreach sort patches [
    set similarity-list lput cosine-similarity ? similarity-list
  ]
end

;; turtle function; reports the cosine similarity of a turtle and a patch's topics
to-report cosine-similarity [curr-patch]
  let patch-topics [topics] of curr-patch
  ; Dot product of topics
  let dot-product sum (map * topic-interests patch-topics)
  ; Then normalize by the magnitude of the vectors
  report dot-product / (make-magnitude topic-interests * make-magnitude patch-topics)
end

to-report make-magnitude [vector]
  ; Magnitude of a vector
  report sqrt sum map [? * ?] vector
end

;; Patch and turtle function
to-report make-topics [num-topics]
  let topic-list []
  repeat num-topics [
    ; For each topic, randomly choose how much of that topic
    set topic-list lput random 3 topic-list
  ]
  report topic-list
end

; turtle method
to find-top-project
  ifelse topic-based? [
    ; if it's topic-based, then look at all patches
    set top-project max-one-of patches [patch-utility]
  ][
  ; Otherwise, only look at neighbors
  set top-project max-one-of patches in-radius neighborhood-size [patch-utility]
  ]
end

to contribute-to-project
  ; Find the top utility of the top project
  let top-utility [patch-utility] of top-project
  ; If the benefit is greater than the cost, then contribute resources
  ifelse top-utility > resources [
    ask top-project[
      ; update the contribution level and progress of the project
      set contribution-level contribution-level + [resources] of myself
      update-patch-progress
      set pcolor red
    ]
    ; Record that this turtle has made a contribution
    set contributions-made contributions-made + resources
    ; If they contributed, they are more likely to contribute in the future
    set interest interest * (1 + positive-reinforcement)
    ; But, they can't go above the max interest
    if interest > max-interest [
      set interest max-interest
    ]
    set color red
  ][
  ;; If they didn't contribute, then decrease their interest
  set interest interest * (1 - negative-reinforcement)
  set color blue
  ]
end

to update-patch-progress
  set progress get-patch-progress contribution-level
end

to-report patch-utility
  let x contribution-level + [resources] of myself ; What would the total contribution be?
  let weight [interest] of myself * multiplier / distance myself ; The utility is multiplied by the overall multiplier times my interest
  if topic-based? [
    ; Multiply the weight by how similar the patch topic distribution is to my interest
    set weight weight * item pindex [similarity-list] of myself
  ]
  let orig_utility weight * progress ; Current utility based on logisitic
  let total_utility weight * get-patch-progress x
  report total_utility - orig_utility ; Return the amount of utility added
end

;; Patch reporter; gives the progress at a given x
to-report get-patch-progress [x]
  let result 0
  carefully[
    ; Update the patch progress based on the current contribution level.
  set result 1 / (e ^ ((mu - x) / beta) + 1)
  ][
  ;If the exponential is too lrge, it means we're really close to 0
  set result 0
  ]
  report result
end

to recolor-patches
  set pcolor scale-color orange progress -1 2
end

; Turtle function
to ask-to-die
  if interest < .05 [
    die
  ]
end

to update-ginis
  ; Code basically stolen from the wealth distribution model
  set patch-gini gini [contribution-level] of patches
  set turtle-gini gini [contributions-made] of turtles
  set patch-progress-gini gini [progress] of patches
end

;; this procedure recomputes the value of gini-index-reserve
;; and the points in lorenz-points for the Lorenz and Gini-Index plots
to-report gini [contributions]
  let sorted-contribs sort contributions
  let total-contribs sum sorted-contribs
  if total-contribs = 0 [
    report 0
  ]
  let contrib-sum-so-far 0
  let index 0
  let gini-index-reserve 0
  let lorenz-points []
  let num-contribs length contributions

  ;; now actually plot the Lorenz curve -- along the way, we also
  ;; calculate the Gini index.
  ;; (see the Info tab for a description of the curve and measure)
  repeat num-contribs [
    set contrib-sum-so-far (contrib-sum-so-far + item index sorted-contribs)
    set lorenz-points lput ((contrib-sum-so-far / total-contribs) * 100) lorenz-points
    set index (index + 1)
    set gini-index-reserve
    gini-index-reserve +
    (index / num-contribs) -
    (contrib-sum-so-far / total-contribs)
  ]
  report (gini-index-reserve / num-contribs) / 0.5
end
@#$#@#$#@
GRAPHICS-WINDOW
273
30
518
282
8
8
13.0
1
10
1
1
1
0
1
1
1
-8
8
-8
8
0
0
1
ticks
30.0

BUTTON
31
52
104
85
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
41
104
104
137
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
25
6
197
39
num-turtles
num-turtles
0
500
200
10
1
NIL
HORIZONTAL

BUTTON
38
153
128
186
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
25
219
197
252
max-interest
max-interest
0
200
100
1
1
NIL
HORIZONTAL

MONITOR
802
22
991
67
NIL
sum [progress] of patches
1
1
11

PLOT
802
91
1002
241
Patch contributions
Contributions
Count
0.0
20.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "set-histogram-num-bars 10" "set-plot-x-range 0 floor max [contribution-level] of patches + 10\nhistogram [contribution-level] of patches"

PLOT
805
275
1005
425
Turtle Contributions
Contributions
Count
0.0
100.0
0.0
240.0
false
false
"" ""
PENS
"default" 0.5 1 -16777216 true "set-plot-y-range 0 count turtles" "set-histogram-num-bars 10\nset-plot-x-range 0 floor max [contributions-made] of turtles + 1\nhistogram [contributions-made] of turtles"

SLIDER
24
273
196
306
global-mu
global-mu
0
10
10
.1
1
NIL
HORIZONTAL

SLIDER
31
320
203
353
global-beta
global-beta
.1
10
10
.1
1
NIL
HORIZONTAL

PLOT
233
373
433
523
Turtle Gini over time
Time
Gini
0.0
50.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot turtle-gini"

PLOT
445
373
645
523
Patch Gini over time
Time
Gini
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot patch-gini"

SWITCH
119
52
268
85
topic-based?
topic-based?
1
1
-1000

PLOT
21
372
221
522
Interest * Contributions
Interest
Contributions
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 2 -16777216 true "" "clear-plot\n;ask patches [\n;plotxy mu * beta contribution-level\n;]\nask turtles [\nplotxy interest contributions-made\n]"

SLIDER
533
47
751
80
positive-reinforcement
positive-reinforcement
0
.6
0.5
.01
1
NIL
HORIZONTAL

SLIDER
533
94
758
127
negative-reinforcement
negative-reinforcement
0
1
0.5
.01
1
NIL
HORIZONTAL

SLIDER
534
143
723
176
neighborhood-size
neighborhood-size
0
5
3
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This is a model of how people make decisions about what collective goods to contribute to.

## HOW IT WORKS

Turtles have interest in projects, and resources to contribute to projects, each of which are randomly distributed. They look at the projects near them, and choose to contribute to whichever one (if any) maximizes their utility.

## HOW TO USE IT

The interest slider affects how much utility turtles get from each of the projects near them. The GLOBAL-MU and GLOBAL-BETA parameters affect the shape of the production / utility functions for each project.

## THINGS TO NOTICE

The goal is to try to get the distributions on the right to match up with empricial patterns of contributions to collective projects, such as wikis, where both distributions follow power-law distributions.

## THINGS TO TRY

Try to find settings that lead to the reference pattern

## RELATED MODELS

This is related to and builds on formal models of collective action, particularly the work by Marwell and Oliver (e.g., Marwell, G., & Oliver, P. (1993). The critical mass in collective action. Cambridge University Press.)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 6.0-M5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>patch-gini</metric>
    <metric>turtle-gini</metric>
    <enumeratedValueSet variable="max-interest">
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="0.1"/>
      <value value="1"/>
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="0.5"/>
      <value value="1"/>
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
      <value value="200"/>
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="3" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="40"/>
    <metric>patch-gini</metric>
    <metric>turtle-gini</metric>
    <metric>sum [contribution-level] of patches</metric>
    <metric>sum [progress] of patches</metric>
    <metric>sum [contributions-made] of turtles</metric>
    <enumeratedValueSet variable="max-interest">
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
      <value value="250"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_2" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="300"/>
    <metric>patch-gini</metric>
    <metric>patch-progress-gini</metric>
    <metric>turtle-gini</metric>
    <metric>sum [contribution-level] of patches</metric>
    <metric>sum [progress] of patches</metric>
    <enumeratedValueSet variable="max-interest">
      <value value="20"/>
      <value value="50"/>
      <value value="80"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neighborhood-size">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="max_interest_experiment" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>sum [contribution-level] of patches</metric>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-interest">
      <value value="100"/>
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neighborhood-size">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="4.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="5.1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="max_interest" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>patch-gini</metric>
    <metric>patch-progress-gini</metric>
    <metric>turtle-gini</metric>
    <metric>sum [contribution-level] of patches</metric>
    <metric>sum [progress] of patches</metric>
    <enumeratedValueSet variable="max-interest">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="0.5"/>
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neighborhood-size">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_2_1" repetitions="2" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="300"/>
    <metric>patch-gini</metric>
    <metric>patch-progress-gini</metric>
    <metric>turtle-gini</metric>
    <metric>sum [contribution-level] of patches</metric>
    <metric>sum [progress] of patches</metric>
    <enumeratedValueSet variable="max-interest">
      <value value="20"/>
      <value value="50"/>
      <value value="80"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="negative-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="topic-based?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-mu">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-beta">
      <value value="2"/>
      <value value="5"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-turtles">
      <value value="100"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="positive-reinforcement">
      <value value="0"/>
      <value value="0.2"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="neighborhood-size">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
