extensions [ vrml ]
globals [ 
  headlist 
  jigglers
  jiggler-patches
  first-mover ; for keeping track of which snakesegment initiates motion, 
              ; whenever a snake movement occurs
              
  colors
  color-names
  ]
breed [snakesegments snakesegment]
snakesegments-own [ 
  prev
  next
  snake-id 
  
  tempdx
  tempdy
  tempdz 
]
; for hubnet handling
breed [students student]
students-own
[
  my-hubnet-id
  my-snake-id
  move-snake?
]
to startup
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
  setup-vars
end
;; initialize global variables
to setup-vars
  set colors      (list red brown green yellow
                        violet (sky + 1) pink (blue - 1))
  set color-names ["red" "brown" "green" "yellow"
                   "purple" "light blue" "pink" "dark blue"]
end
to-report get-color [ id ]
  report (item (id mod (length colors)) colors)
end
to-report get-color-name [ id ]
  report (item (id mod (length color-names)) color-names)
end

to setup
  clear-patches
  reset-perspective
  set headlist []
  set first-mover nobody
  ask patch 0 0 (max-pzcor - 2) [ set plabel "UP" ]
  ask patch 0 0 (min-pzcor + 2) [ set plabel "DOWN" ]
  ask patch 0 (max-pycor - 2) 0 [ set plabel "NORTH" ]
  ask patch 0 (min-pycor + 2) 0 [ set plabel "SOUTH" ]
  ask patch (max-pxcor - 2) 0 0 [ set plabel "EAST" ]
  ask patch (min-pxcor + 2) 0 0 [ set plabel "WEST" ]
  orbit-down 45
  orbit-left 30
  zoom 20
end
to go
  listen-clients
  forward-all 0.1
end
to reset-snakes
  ask snakes [ 
    repeat 39 [ snake-shrink ]
    setxyz 0 0 0
    set heading 360 * snake-id / (count snakes)
  ]
end
to update-appearance
  ifelse (is-head? and not looped?)
  [
    set shape "default"
    set size 2
  ]
  [
    set shape "circle"
    set size 1
  ]
  set color (get-color snake-id) - 1
  if (is-head?)
    [ set color color + 2 ]
end

to-report snake [ n ]
 if (n < 0 or n >= length headlist or (item n headlist) = nobody )
 [ 
   user-message (word "Snake " n " doesn't exist!")  
   report nobody
 ] 
 report item n headlist 
end
to-report snakes
  report turtle-set headlist
end
to-report is-head?
  report self = snake snake-id 
end
to-report is-tail?
 report (next = nobody) or ([is-head?] of next)
end
to-report get-head
  report (snake snake-id)
end
to-report get-tail
  ifelse (is-tail?)
  [ report self ]
  [ report [get-tail] of next ]
end
to-report make-snake [ x y z ]
  if (not is-list? headlist )
    [  set headlist [] ]
  let nextid position nobody headlist   ; reuse snake-ids
  if (nextid = false)
    [ set nextid length headlist ]
  let temp nobody
  create-snakesegments 1 
  [
    setxyz x y z
    set pitch random 360
    set prev nobody
    set next nobody
    set temp self
    set snake-id nextid
  ]
  ifelse (nextid = length headlist)
  [ set headlist lput temp headlist ]
  [ set headlist replace-item nextid headlist temp ]
  ask temp [ update-appearance ]
  report temp
end
to snake-grow
  ; impose a limit of 40 segments, to prevent any student
  ; from going too crazy about snake-length.
  if (count snakesegments with [ snake-id = [snake-id] of myself ] < 40)
  [
    let temp next
    hatch 1 [
      set next temp
      set prev myself
      set temp self
      update-appearance
    ]
    if (not is-tail?)
    [ 
      ask next
        [ set prev temp ]
    ]
    set next temp
    fd 0.55
  ]
end
to snake-shrink
  ask get-head [
    if (not is-tail?)
    [
      let temp self
      face next
      move-to next
      rt 180
      ask next
      [ 
        if (next != nobody)
        [
          ask next
          [ 
            set prev temp
            set temp self
          ]
        ]
        die
      ]
      set next temp
    ]
  ]
end
to snake-die
   ask snakesegments with [ snake-id = [snake-id] of myself ]
   [
     die 
   ]
end
to-report looped?
  report [ prev != nobody ] of get-head
end
to snake-loop
  ask get-head
  [
    let mytail get-tail
    set prev mytail
    ask mytail [ set next myself ]
    face mytail
    update-appearance
  ]
end
to snake-unloop
  ask get-head
  [
    let mytail get-tail
    set prev nobody
    ask mytail [ set next nobody ]
    face next
    rt 180
    update-appearance
  ]
end
to snake-reverse
 ask get-head 
 [
   snake-reverse-helper
   update-appearance
 ]
end
to snake-reverse-helper
  let temp next
  set next prev
  set prev temp 
  ifelse (prev = nobody or ([is-head?] of prev))
  [ 
    set headlist replace-item snake-id headlist self
    update-appearance
    if (not is-tail?)
    [ face next ]
    right 180
  ]
  [
    ask prev [ snake-reverse-helper]
  ]
  
end
; neat little utility function, to get every snakesegment
; in the snake to run some code.
to snake-map [ str ]
  run str
  if (not is-tail?)
  [
    ask next [ snake-map str ]
  ]
end
to snake-move [ dist ]
  let distprev world-width ; big number
  let mystep dist
  
  ifelse (self = first-mover)
  [
    fd dist
  ]
  [ 
    face prev
    set distprev (distance prev)
    ; we allow them to step up to 25% farther than the
    ; head moved, to allow them to "catch up" if necessary
    ; we don't allow them to move *too* close to
    ; the snakesegment ahead of them.
    set mystep min list (dist * 1.25)  (distprev - 0.55)
    
    if (mystep != 0 )
    [
      fd mystep
    ]
  ]
  if (next != nobody and next != first-mover)
  [
    ask next [ snake-move dist ]  
  ]
end
to snake-move-bk [ dist ]
  let distnext world-width ; big number
  let mystep dist
 
  ifelse (self = first-mover)
  [
      fd dist
  ]
  [ 
    face next
    set distnext (distance next)
    ; we allow them to step up to 25% farther than the
    ; head moved, to allow them to "catch up" if necessary
    ; we don't allow them to move *too* close to
    ; the snakesegment ahead of them.
    set mystep min list (dist * 1.25)  (distnext - 0.55)
    
    if (mystep != 0 )
    [
      fd mystep
    ]
  ]
  if (prev != nobody and prev != first-mover)
  [
    ask prev [ snake-move-bk dist ]  
  ]
end
to snake-forward [ dist ]
  if ([move-snake?] of one-of students with [ my-snake-id = [snake-id] of myself ])
  [
    repeat round (dist / 0.1)
    [
      ; even if you ask a snakesegment in the middle
      ; of the snake to move, we move the head anyway
      ask get-head
      [ 
        update-appearance
        set first-mover self
        ifelse (looped?)
        [
            face prev
            snake-move 0.1
        ]
        [
          snake-move 0.1
        ]
      ]
      collision-detection
      display
    ]
  ]
end
to snake-step
  snake-forward 1
end
to forward-all [ dist ] 
  repeat round (dist / 0.1)
  [
    ask snakes 
    [ 
      if ([move-snake?] of one-of students with [ my-snake-id = [snake-id] of myself ])
      [
        set first-mover self
        ifelse (looped?)
        [
          face prev
          snake-move 0.1
        ]
        [
          snake-move 0.1
        ]
      ]
    ]
    collision-detection
    display
  ]
end

to collision-detection
  let clist []
  ask snakesegments
  [
    set tempdx 0
    set tempdy 0
    set tempdz 0
    let pushx 0
    let pushy 0
    let pushz 0
    let myx xcor
    let myy ycor
    let myz zcor
    let nearbys snakesegments in-radius 1.0
    set nearbys nearbys with [prev != myself and next != myself and self != myself]
;    print count nearbys
    if (any? nearbys)
    [
      ask nearbys [
        let dist distance myself
        if (dist = 0)
          [ fd 0.001 set dist 0.001  ]
        let push (1.0 - dist) / 2 + 0.02
        let ddx myx - xcor
        if (ddx < 0 - world-width / 2)
         [ set ddx ddx + world-width ]
        if (ddx > world-width / 2)
         [ set ddx ddx - world-width ]
        let ddy myy - ycor
        if (ddy < 0 - world-height / 2)
         [ set ddy ddy + world-height ]
        if (ddy > world-height / 2)
         [ set ddy ddy - world-height ]
        let ddz myz - zcor
        if (ddz < 0 - world-depth / 2)
         [ set ddz ddz + world-depth ]
        if (ddz > world-depth / 2)
         [ set ddz ddz - world-depth ]
        set pushx pushx + ddx / dist * push
        set pushy pushy + ddy / dist * push
        set pushz pushz + ddz / dist * push
      ]
      set tempdx pushx / count nearbys + random-float 0.01
      set tempdy pushy / count nearbys + random-float 0.01
      set tempdz pushz / count nearbys + random-float 0.01
      set clist fput self clist
      ;show (list tempdx tempdy tempdz)
      
;      show (list tempdx tempdy tempdz)
;    if (tempdx != 0)
;    [ print word "moo: " (list tempdx tempdy tempdz) ]
    ]          
  ]
  foreach clist [
    ask ? [
      set xcor xcor + tempdx
      set ycor ycor + tempdy
      set zcor zcor + tempdz
      
    ]
  ]
  
end
to tighten-all
  ask snakes [ snake-reverse ]
  forward-all 0.5
  ask snakes [ snake-reverse ]
  forward-all 0.5
end
to snake-tighten
  snake-reverse
  snake-forward 0.5
  snake-reverse
  snake-forward 0.5
end

to jiggle-snakes
  if (not is-list? jigglers or member? nobody jigglers or random 100 < 1)
  [ set jigglers [self] of (n-of 2 snakesegments)
    set jiggler-patches [self] of (n-of 2 patches ) 
  ]
  (foreach jigglers jiggler-patches
  [
    let j ?1
    let jp ?2
    ask j
    [
      ;let nearbys (snakesegments in-radius 2.0) with [prev != myself and next != myself and self != myself]
      ;ifelse any? nearbys
      ;[ face one-of nearbys ]
      ;[ face one-of patches ]
      face jp
      set first-mover self
      forward 0.05      
      if (next != nobody)
      [ ask next [ snake-move 0.1 ] ]
      if (prev != nobody)
      [ ask prev [ snake-move-bk 0.1 ] ]
    ]
  ])
  collision-detection
  display
end
to save-to-vrml 
  let filename user-new-file 
  if (filename != false)
  [
    vrml:clear-scene
    ask snakesegments [
      let mycolor extract-rgb color
      ; make the head the same color as the body
      ; in the vrml export
      if (is-head?)
        [ set mycolor extract-rgb (color - 2) ]
      vrml:set-color (item 0 mycolor) (item 1 mycolor) (item 2 mycolor)
      vrml:add-sphere xcor ycor zcor 0.5
    ]
    vrml:save-scene filename
  ]
end
; returns the head of the most recently created snake
to-report newest-snake
  report snake ((length headlist) - 1)
end

;;; HUBNET procedures
;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-student ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [
        ask students with [ my-hubnet-id = hubnet-message-source ]
          [ execute-command hubnet-message-tag ]
      ]
    ]
  ]
end
to execute-command [command]
  if command = "move-snake?"
  [
    set move-snake? hubnet-message
    stop
  ]
  if command = "north"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch 0 set heading 0 ] stop ]
  if command = "south"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch 0 set heading 180 ] stop ]
  if command = "east"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch 0 set heading 90 ] stop ]
  if command = "west"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch 0 set heading 270 ] stop ]
  if command = "up"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch 90 ] stop ]
  if command = "down"
  [ ask snake my-snake-id [ rt 90 fd 0.03 set pitch -90 ] stop ]
  ask snake my-snake-id [ 
    run command 
    stop
  ]
end
to create-new-student
  let newsnake (make-snake random-pxcor random-pycor random-pzcor)
  create-students 1
  [
    ht
    set my-hubnet-id hubnet-message-source
    set my-snake-id [snake-id] of newsnake
    set move-snake? false
    hubnet-send my-hubnet-id "my-snake-color" get-color-name my-snake-id
  ]
end

to remove-student
  ask students with [my-hubnet-id = hubnet-message-source]
  [
    ask snake my-snake-id [ snake-die ]
    die
  ]
end

@#$#@#$#@
GRAPHICS-WINDOW
0
0
855
876
32
32
13.0
1
24
1
1
1
0
1
1
1
-32
32
-32
32
-32
32
1
1
1
ticks
CC-WINDOW
5
247
119
342
Command Center
0
BUTTON
10
10
80
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
10
155
110
188
camera-orbit
every 0.01 [ orbit-right 0.2 ]\\ndisplay
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
10
200
110
233
NIL
save-to-vrml
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
10
55
80
88
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
10
104
110
137
NIL
reset-snakes
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
@#$#@#$#@
VERSION
-------
$Id: KnotLogo-Hubnet.nlogo 40114 2008-06-12 23:22:32Z tisue $

WHAT IS IT?
-----------
This Hubnet model is based off of the KnotOrNot3D NetLogo model, which is a microworld for playing with rope, thinking and learning about knots and knot theory, as well as creating virtual rope-based art and artifacts.  This Hubnet version of it provides a shared virtual space where a class can interact together to engage in these activities.

HOW IT WORKS
------------
Students have more limited control than they do in the "solo" KnotOrNot environment.  They also have a different method of navigating their snakes.  In the "solo" world, everything was snake-centric.  In the Hubnet version, students choose absolute directions for their snake's movement (North, South, East, West, Up, and Down), and are limited to orthogonal movement (although the snake's body will still curve nicely around behind).  The reason for this change is twofold.  
1) I think it's good for students to have to think in multiple representations -- both from the perspective of individual agents, and from the perspective of the environment.  Or, from a totally mathematical perspective, we are making them think in two different coordinate spaces. 
2) In the "solo" model, each student could move the camera and position it such that they had a good view of their turtle, and could figure out which direction it was facing.  In this Hubnet version, there is only one global camera, controlled by the teacher, and so viewing your turtle's heading is more difficult.  It should be easier to spotting your turtle's location, and using the direction labels on the boundaries of the world, choose an appropriate direction to move.  
HOW TO USE IT
-------------
The teacher should push SETUP, then GO.  DON'T push SETUP again after the model is running!
Students should log in.  They will each be assigned a color of snake based on their order of logging in, and they will be told (on their client screens) what color they are.  Currently there are only different colors, so more than 8 clients will cause repeated colors -- and probably some confusion.  More colors can be added easily by changing the COLORS and COLOR-NAMES lists that are initialized in the SETUP procedure.  The trick is in choosing lots of good colors that contrast with each other.
The RESET-SNAKES button makes all the snakes only one segment long, and moves them all to the center of the world.
If desired, the teacher can also set the CAMERA-ORBIT button going, so that the viewing angle on the world will slowly rotate around.  This may be useful for making the world seem more 3D, as well as moving closer and farther away from various students.
If the class creates a neat rope configuration that they'd like to save, the teacher can use the SAVE-TO-VRML button.
THINGS TO NOTICE
-------------
The student's snake bodies are limited to 40 units in length in this Hubnet version, to prevent any student from getting too carried away.  Furthermore, the model runs more slowly the more segments there are in the world, and the Hubnet version in particular can really come to a crawl when the world is crowded.  Thus limiting the length of snakes should be helpful for that.
THINGS TO TRY
-------------
This is a fairly open-ended participatory simulation.  In general, it might work to let the class choose the direction of the activity.  That said, sometimes they need more structure and guidance, so here are some ideas:
1) Can the class pair up and tie simple knots with each other?
2) Can the class form a long chain, by having each snake bite its own tail through the rings created by two other snakes?
3) Can the class cooperatively spell out a message using the snake bodies?

EXTENDING THE MODEL
-------------------
Make a slider for the teacher that controls the maximum segment length for any snake.
Try adding support for teacher-controlled snakes.  Or give the teacher more controls for manipulating the student's snakes.
What about giving client interfaces inputboxes, so students can ask their snakes to run any code they want?  Is this safe?  Probably not, students who are sufficiently savvy with NetLogo can find ways to break the model.  Even so, it still could be a lot of fun, and the benefits might outweigh the drawbacks.
RELATED MODELS
--------------
KnotOrNot3D -- the solo version.
CREDITS AND REFERENCES
----------------------
Model authored by Forrest Stonedahl, for an LS 426 Final Project at Northwestern University. 
The basic rope simulation is adapted from the "follow the leader" algorithm proposed in this paper:
Brown, J. Latombe, J. & Montgomery, K. (2004), 'Real-time knot-tying simulation', The Visual Computer 20(2), 165--179.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250
airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15
arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150
box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75
bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30
butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60
car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58
circle
false
0
Circle -7500403 true true 0 0 300
circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123
cylinder
false
0
Circle -7500403 true true 0 0 300
dot
false
0
Circle -7500403 true true 90 90 120
face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240
face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225
face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183
fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30
flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45
flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240
house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120
leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195
line
true
0
Line -7500403 true 150 0 150 300
line half
true
0
Line -7500403 true 150 0 150 150
pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120
person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
square
false
0
Rectangle -7500403 true true 30 30 270 270
square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240
star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108
target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60
tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152
triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255
triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224
truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42
turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99
wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269
x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 3D Preview 5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
BUTTON
80
60
143
93
north
NIL
NIL
1
T
OBSERVER
NIL
W
BUTTON
79
120
142
153
south
NIL
NIL
1
T
OBSERVER
NIL
S
BUTTON
144
90
207
123
east
NIL
NIL
1
T
OBSERVER
NIL
D
BUTTON
14
90
77
123
west
NIL
NIL
1
T
OBSERVER
NIL
A
BUTTON
223
84
286
117
up
NIL
NIL
1
T
OBSERVER
NIL
U
BUTTON
223
119
286
152
down
NIL
NIL
1
T
OBSERVER
NIL
J
MONITOR
225
180
322
229
my-snake-color
NIL
3
1
SWITCH
10
10
137
43
move-snake?
move-snake?
1
1
-1000
BUTTON
15
170
110
203
snake-grow
NIL
NIL
1
T
OBSERVER
NIL
NIL
BUTTON
15
205
110
238
snake-shrink
NIL
NIL
1
T
OBSERVER
NIL
NIL
BUTTON
115
170
215
203
snake-loop
NIL
NIL
1
T
OBSERVER
NIL
NIL
BUTTON
115
205
215
238
snake-unloop
NIL
NIL
1
T
OBSERVER
NIL
NIL
BUTTON
145
10
252
43
snake-reverse
NIL
NIL
1
T
OBSERVER
NIL
NIL
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
