;; --------------------------------------------------------------------------
;; COVID-19 mitigation with personal habits and intermittent short lockdowns
;;
;; A model by Sergio Rojas-Galeano and Lindsay Alvarez
;; v1.27 Copyright (c) September 2020 The authors
;; Correspondance email: srojas@udistrital.edu.co
;; Universidad Distrital Francisco Jose de Caldas, Bogota, Colombia
;;
;; This is an extended version of the of the COVID-19 epidemic simulation
;; with Non-Pharmaceutical Interventions and zonal restraints model
;;
;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License (GPLv3)
;; (see license at: https://www.gnu.org/licenses/gpl-3.0.txt)
;;
;; The model is made publicly available in the hope that it will be useful
;; to modelers, but WITHOUT ANY WARRANTY whatsoever (see license for details).
;; --------------------------------------------------------------------------

;; Define globals
globals [
  daytime?      ; is it day or night? ( 1 day + 1 night = ticks-day)
  sort-codes    ; a list of codes to identify each zone of a city
  shapes-list   ; a list of shapes to identify agents of the same zone
  colors-list   ; a list of colors to identify different zones
  day hour      ; the current day and hour since simulation started
  sicks-day     ; list of daily count of sicks
  n-cases       ; number of overall COVID-19 sick people
  n-confirmed   ; number of confirmed COVID-19 sick people
  n-tested      ; number of overall people tested by sentinel
  n-positives   ; number of people that tested COVID-19 positive
  doubling-time ; estimated days to double today's number of sicks
  R0 S0 R0_     ; reproduction number and initial Susceptibles
  p-contagion   ; probability of contagion on a single encounter
  intermittent? ; is it intermittent lockdown on or off (i.e. "accordion strategy")?
]

;; Define breeds for healthy and infected people
breed[ healthies healthy ]
breed[ sicks sick ]
breed[ deaths death ]
breed[ houses house ]
breed[ ambulances ambulance ]

;; Define attributes for patches
patches-own [
  zone                   ; the zone code of this patch
]

;; Define attributes of all the people
turtles-own [
  homebase               ; the home patch of this person
  scope                  ; the reach of distance away from home for this person
  speed                  ; how fast does he/she move? A speed of 0 is still
  unlocked?              ; is the person free from lockdown?
  tested?                ; was the person tested by sentinel test?
  contagions             ; how many people this person infected if he/she ever got sick
  mask-habit?            ; is the person willing to wear protection mask?
  wash-habit?            ; is the person willing to wash his hands regularly?
]

;; Define attributes for sick people only
sicks-own [
  days-to-cure           ; the actual duration of the illness for this person
  recovery               ; counter of days remaining to recovery
  confirmed?             ; is the person isolated at homebase?
  asymptomatic?          ; is the person asymptomatic?
  risky?                 ; is the person high-risk due to existing factor (obese, diabetes, older, etc)
  severe?                ; is the person in severe condition?
  deadly?                ; is the person in deadly (critical) condition?
  hospitalised?          ; has he/she been admitted to hospital?
  ICU-admitted?          ; has he/she been admitted to ICU?
]

;; Define attributes for healthy people only
healthies-own [
  immune?                ; has the person recovered and acquired immunity?
]

;; Define attributes for ambulances
ambulances-own [
  tests-stock            ; how many tests in stock
  tests-supply           ; numbers of tests to top-up (depends on amount of zone residents)
]

;; One iteration of simulation's steps
to go
  if ticks = ticks-day / 2 [ outbreak ] ; Epidemic begins with outbreak at 0d:12h (enable this line only for BehaviourSpace experiments)
  ifelse day < end-day [ tick ] [ stop ] ;screenshot stop ]
  lifestyle
  epidemic
  isolation
  quarantine
  distancing
  sentinel
  lockdown
  illness
  clock
  indicators
end

;; Simulate everyday routine of people
to lifestyle
    ask (turtle-set healthies sicks with [not confirmed?] ) [
    if zone-enforcing? [
      if zone != [zone] of homebase [ face homebase fd 0.5 ]  ; if away from resident zone, head back
    ]
    forward speed                                             ; move ahead
    if distance homebase > scope [ face homebase ]            ; if too far from home, head back
    set heading heading + (random-float 3 - random-float 3)   ; change direction swiftly
  ]
end

;; Spread the infection from sicks not yet isolated, to healthies
to epidemic
  ask sicks with [not confirmed?] [
    let counter 0                     ; number of contagions made by this sick
    let sick-masked? mask-habit?      ; is the sick wearing mask?

    ask healthies with [not immune?] in-radius .5 [
      let healthy-masked? mask-habit? ; is the healthy wearing mask?

      ;; Set contagion risk depending on mask-wearing habits
      set p-contagion contagion-risk sick-masked? healthy-masked?

      ;; Contagion risk may decrease further depending on hand-washing habits of healthies
      if wash-habit? [ set p-contagion (p-contagion * 0.3) ]  ; P(Contagion | p-contagion, healthy-washer?)

      if random-float 1 < p-contagion [
        get-sick
        set counter counter + 1
      ]
    ]
    set contagions contagions + counter
  ]
end

;; Probablity of contagion depending on mask wearing combination
to-report contagion-risk [sick-masked? healthy-masked?]
  let p 0
  if not sick-masked? and not healthy-masked? [ set p 0.9 ]  ; P(Contagion | False, False)
  if not sick-masked? and     healthy-masked? [ set p 0.5 ]  ; P(Contagion | False, True )
  if     sick-masked? and not healthy-masked? [ set p 0.3 ]  ; P(Contagion | True,  False)
  if     sick-masked? and     healthy-masked? [ set p 0.1 ]  ; P(Contagion | True,  True)
  report p
end


;; Make a single person ill
to get-sick
  ;; S->I (set the initial properties of a sick person)
  set breed sicks
  set shape item ([zone] of homebase) shapes-list
  set confirmed? false
  set tested? false
  set severe? false
  set deadly? false
  set hospitalised? false
  set ICU-admitted? false

  ;; Set stochastic attributes
  set days-to-cure abs int random-normal avg-duration 4         ; duration of illness for this sick
  set recovery 0                                                ; initial count of days until recovery
  set asymptomatic? (random 100) < %-asymptomatics              ; is this an asymptomatic patient?
  set risky? (random 100) < %-high-risk  and not asymptomatic?  ; is it high-risk (co-morbidities) given not asymptomatic?
  set color ifelse-value asymptomatic? [ yellow ] [ red ]

  ;; Update how many cases so far
  set n-cases n-cases + 1
end

;; Ask confirmed sick people to stay home
to isolation
  ;; If quarantine on, enforce isolation of symptomatic
  ifelse case-isolation? [
    ask sicks with [ not asymptomatic? and not confirmed? ] [
      move-to homebase
      set confirmed? true
      set n-confirmed n-confirmed + 1         ; these sicks are now confirmed cases
    ]
  ]
  ;; Else lift isolation, except if was sent sentinel testing
  [
    ask sicks with [ not tested? and confirmed? ] [
      set confirmed? false
      set n-confirmed n-confirmed - 1         ; these sicks are set back to not confirmed cases
    ]
  ]

  ;; Decorate houses with color of isolated people
  ask houses [
    set color ifelse-value any? sicks-here with [ not asymptomatic? ] [ red ]
            [ ifelse-value any? sicks-here with [ asymptomatic? ] [ yellow ] [ white ] ]
  ]
end

;; Ask housemates of isolated  people to stay home
to quarantine
  if home-quarantine? [
    ask sicks with [ confirmed? ] [
      let sick-home homebase
      ask other turtles with [ homebase = sick-home ] [ move-to homebase ]
    ]
  ]

  ;; Decorate houses with color of quarantined people
  ask houses [
    set color ifelse-value any? sicks-here with [ not asymptomatic? ] [ red ]
            [ ifelse-value any? sicks-here with [ asymptomatic? ] [ yellow ] [ white ] ]
  ]
end

;; Avoid contacting nearby people (closer than than min-dist)
to distancing
  if social-distancing?
  [ let people (turtle-set healthies sicks)
    let willings (count people * %-willings / 100)
    ask n-of willings people [
      if any? other people in-radius 0.6 [  ; a radius of 0.6 is slightly higher than the 0.5 threshold to contagion
        let closest min-one-of other people [distance myself]
        face closest rt 180                 ; head in the opposite direction of closest person
      ]
    ]
  ]
end

;; Ask ambulances to catch sicks by mass testing
to sentinel
  ifelse sentinel-testing? [
    ask ambulances [
      st
      if tests-stock > 0 [
        ;; Move ambulances around
        forward speed
        if zonal? [ if zone != [zone] of homebase [ face homebase fd 0.5 ] ]  ; keep ambulance in-zone
        set heading heading + (random-float 10 - random-float 10)             ; change ambulance's heading randomly

        ;; Pick nearbys and perform tests
        let target-people ((turtle-set healthies sicks) in-radius 5) with [not tested?]
        ask target-people  [
          if is-sick? self and not confirmed? [                               ; if sick and not isolated, target is positive
            move-to homebase set confirmed? true
            set n-positives n-positives + 1                                   ; update count positives tests so far
            set n-confirmed n-confirmed + 1                                   ; update count of confirmed cases so far
          ]
          set tested? true                                                    ; mark this person as tested
        ]
        set n-tested n-tested + (count target-people)                         ; update count of tests made so far
        set tests-stock tests-stock - (count target-people)                   ; update stock of tests available
      ]
    ]
  ]
  ;; Else, stop and hide ambulances
  [ ask ambulances [ ht ] ]
end

;; Enforce everyone staying home, except for a few permits
to lockdown
  if (hour mod 24) = 0 [                                            ; lockdown starts or ends at 00:00 of next day
    ifelse total-lockdown? [
      let people (turtle-set healthies sicks) with [not unlocked?]  ; for people not in an unlocked zone
      ask people [ move-to homebase set speed 0 ]                   ; sent everyone home, not confirmed
      foreach sort-codes [
        z -> let residents people with [ zone = z ]
        let permits (count residents * %-permits / 100)
        ask n-of permits residents [ set speed step-size ]
      ]
      if intermittent? and (day mod 7) >= 3 [                            ; apply flexible lockdown: 3 days out of 7 days period
        ask people [ set speed step-size ]
      ]
    ]
    ;; Else unlock everyone
    [
      ask (turtle-set healthies sicks) [ set speed step-size set unlocked? false ]
      if intermittent? [ lock3X4 ]
    ]
  ]
end

;; Allow to lift total lockdown by unlocking zones gradually
to unlock
  ;let pick-zone random zones
  ask (turtle-set healthies sicks) with [ zone = (pick-zone - 1) ] [ set speed step-size set unlocked? true ] ;not unlocked? ]
end

;; Enable/disable 3x4 flexible lockdown i.e. "accordion rule": 3 days lockdown by 4 days free movement
to lock3X4
  set intermittent? (not intermittent?)                   ; flip to the opposite value
  ask patch (min-pxcor + 20) (min-pycor + 2) [  ; show/hide indicator label at bottom-left corner
    set plabel-color black
    set plabel ifelse-value intermittent? [ "3x4 ON" ] [""]
  ]
end

;; Simulate illness evolution in each person
to illness
  if (ticks mod ticks-day) = 0 [                        ; illness unfolds in a daily basis

    ;; I->R (sicks recovering after days-to cure)
    ask sicks [
      set recovery recovery + 1                       ; countdown of days left to recover
      if recovery >= days-to-cure [ get-well ]        ; if recovery time over, the person is cured
    ]

    ;; I->E (sicks getting extinct, that is, dead)
    ask sicks with [not asymptomatic?] [
      ;; Define probability of death
      let p 0.005                                     ; P(Extinct | Symptomatic)
      if deadly? and not ICU-admitted? [ set p 1.0  ] ; P(Extinct | Deadly, no ICU)
      if deadly? and     ICU-admitted? [ set p 0.5  ] ; P(Extinct | Deadly, ICU)
      if severe? and not hospitalised? [ set p 0.2  ] ; P(Extinct | Severe, Hospitalised)
      if severe? and     hospitalised? [ set p 0.05 ] ; P(Extinct | Severe, no Hospitalised)

      ;; Death may occur any day within days-to-cure, with triangular probability distribution peaking at half the days-to-cure
      let day-chance cdf-triangle recovery days-to-cure ; CDF of triangle distribution

      ;; Simulate death
      let r random-float 1.0
      if r < (p * day-chance) [ pass-away ]
    ]

    ;; I+Severe
    ask sicks with [not asymptomatic? and not (severe? or deadly?)] [
      let p 0.0
      if not risky? [ set p 0.05 ]                    ; P(Severe | Symptomatic, no Risky)
      if risky?     [ set p 0.25 ]                    ; P(Severe | Symptomatic, Risky)

      ;; Severity may arise any day within days-to-cure, following an uniform distribution
      let day-chance recovery / days-to-cure          ; CDF of uniform distribution

      ;; Simulate severity
      let r random-float 1.0
      if r < (p * day-chance) [ set severe? true ]
    ]

    ;; I+Deadly
    ask sicks with [severe?] [
      let p 0.0
      if not risky? [ set p 0.05 ]                    ; P(Deadly | Severe, no Risky)
      if risky?     [ set p 0.25 ]                    ; P(Deadly | Severe, Risky)

      ;; Deadliness may arise any day within days-to-cure, following an uniform distribution
      let day-chance recovery / days-to-cure          ; CDF of uniform distribution

      ;; Simulate deadliness
      let r random-float 1.0
      if r < (p * day-chance) [ set deadly? true set severe? false ]
    ]

    ;; I+Hospitalised
    let occupied count sicks with [severe? and hospitalised?]
    let required count sicks with [severe? and not hospitalised?]
    let incoming min list (hospital-beds - occupied) required                   ; how many can be hospitalised?
    ask n-of incoming sicks with [severe? and not hospitalised?] [              ; hospitalised as many as possible
      set hospitalised? true
      set tested? true                                                          ; so they can't move even when isolation off
      if not confirmed? [ set confirmed? true set n-confirmed n-confirmed + 1 ] ; if wasn't confirmed yet, confirm and update count
    ]

    ;; I+ICU-admitted
    set occupied count sicks with [deadly? and ICU-admitted?]
    set required count sicks with [deadly? and not ICU-admitted?]
    set incoming min list (ICU-beds - occupied) required                        ; how many can be admitted to ICU?
    ask n-of incoming sicks with [deadly? and not ICU-admitted?] [              ; ICU-admit as many as possible
      set ICU-admitted? true
      set tested? true                                                          ; so they can't move even when isolation off
      if not confirmed? [ set confirmed? true set n-confirmed n-confirmed + 1 ] ; if wasn't confirmed yet, confirm and update count
    ]
  ]
end

;; Compute Cumulative Distribution Function (CDF) of triangular distribution at a given number of days elapsed (x) within illness period (width)
to-report cdf-triangle [x width]
  ;; Height of the inner triangle to obtain CDF, i.e. the value of the PDF on x
  let height 2 * (width - abs (width - (2 * x))) / (width ^ 2)

  ;; If x is before peak, CDF is the area of the inner triangle with base x
  ifelse x <= (width / 2) [
    let base x
    report height * base / 2
  ]
  ;; Else if x is after peak, CDF is to total area (1) minus area of inner triangle with base (width - x)
  [
    let base (width - x)
    report 1 - (height * base / 2)
  ]
end

;; Become healthy again
to get-well
  let old-shape shape
  set breed healthies
  set immune? true
  set shape old-shape
  set color white
  set speed step-size
end

;; Become dead
to pass-away
  if not confirmed? [ set confirmed? true set n-confirmed n-confirmed + 1 ]  ; some cases are confirmed only at death
  set breed deaths
  set shape "x"
  set speed 0
  set color black
end

;; Update counters of days and hours
to clock
  set day int (ticks / ticks-day)           ; track of number of days elapsed since beginning
  set hour int ((ticks / ticks-day) * 24)   ; track of number of hours elapsed since beginning
  sunset
end

;; Record epidemic indicators
to indicators
  ;; Daily statistics are sicks per day and doubling-time
  if ticks mod ticks-day = 0 [
    set sicks-day lput (count sicks) sicks-day

    ;; Compute doubling time
    ;; see (Bakir, 2016, "Compound Interest Doubling Time Rule: Extensions and Examples from Antiquities.")
    if any? sicks[
      ifelse (last sicks-day) > (item (day - 1) sicks-day)
      [ set doubling-time ln 2 / ln ((last sicks-day) / (item (day - 1) sicks-day + 1E-10)) ]
      [ set doubling-time "Inf" ]
    ]
  ]

  if ticks mod (ticks-day / 24) = 0 [
    ;; Estimate R0 based on the final size of susceptibles in population
    ;; see (Dietz, 1993, "The estimation of the basic reproduction number for infectious diseases.")
    if S0 != 0 [
      let Sn count healthies with [ not immune? ] / pop-size
      set R0 (ln (S0 + 1E-10) - ln (Sn + 1E-10)) / (S0 - Sn + 1E-10)
    ]
    ;; Estimate R0_ with an individual-based approach, as the average of contagions per sick
    if any? turtles with [contagions > 0] [
      set R0_ mean [contagions] of turtles with [contagions > 0]
    ]
  ]
end

;; Inoculate infection on one agent
to infect
  if any? healthies with [not immune?][
    ask one-of healthies with [not immune?] [ get-sick ]
  ]

  ;; Record the initial count of susceptibles for R0 computation
  set S0 (count healthies with [not immune?]) / pop-size
end

;; Contagion by an outbreak
to outbreak
  ;; Infect a percentage of population
  repeat pop-size * (%-spread / 100) [ infect ]

  ;; Record the initial count of susceptibles for R0 computation
  set S0 (count healthies with [not immune?]) / pop-size
end

;; Light up or down the city depending on daytime
to sunset
  if see-sunset? [
    if (ticks mod (ticks-day / 2)) = 0 [
      ifelse daytime?
      [ ask patches [ set pcolor pcolor + 4 ] ]
      [ ask patches [ set pcolor pcolor - 4 ] ]
      set daytime? not daytime?
    ]
  ]
end

;; Get ready to go
to setup
  clear-all
  reset-ticks
  setup-globals
  setup-city
  setup-people
  setup-ambulances
  set daytime? false
  set intermittent? false
end

;; Initialize global variables
to setup-globals
  ;; Codes to identify zones
  set sort-codes n-values zones [ i -> i ]

  ;; Residents of each zone are identified with different shapes and color
  set shapes-list [ "dog" "cat" "cow skull" "fish 2" "chess knight" "ant 2" "wolf 7" "bird"
    "butterfly 2" "rabbit" "bug" "spider" "ghost" "monster" "person" "fish" "sheep" "bee 2"]
  set colors-list [ cyan pink violet green brown turquoise gray sky lime violet blue
    brown gray sky pink turquoise ]

  ;; Initalise daily sicks, positives, total-tested, total and confirmed cases
  set sicks-day [0]
  set n-positives 0 set n-tested  0 set n-cases 0 set n-confirmed 0

  ;; Initial value for doubling time
  set doubling-time 0.0
end

;; Split the city into zones and setup facilities
to setup-city
  set-patch-size 1.9

  ;; Layout a grid of patch square zones by assingning auxiliary codes
  let step world-width / sqrt zones
  ask patches[
    let units int ((pxcor + max-pxcor) / step)
    let tens  int ((pycor + max-pycor) / step)
    set zone 10 * tens + units
  ]

  ;; Now map those auxiliary codes to consecutive sort-codes and corresponding colors
  let aux-codes sort remove-duplicates [zone] of patches
  (foreach aux-codes sort-codes [ [old new] -> ask patches with [zone = old] [set zone new] ])
  ask patches [ set pcolor (item zone colors-list) + 1 ]
  ;foreach sort-codes [ c -> ask one-of patches with [zone = c] [set plabel zone] ] ; display zone code on view area

  ;; If housing, create houses
  if housing? [
    create-houses pop-size / 4 [
      setxy random-xcor random-ycor
      set color white set shape "house" set size 9
    ]
  ]
end

;; Create initial population
to setup-people
  create-healthies pop-size [
    ;; Assign as homebase either a house if any, otherwise his initial position
    ifelse housing?
    [  set homebase one-of houses
       move-to homebase
    ][ setxy random-xcor random-ycor
       set homebase patch-here
    ]

    ;; Assign color and shape depending on homebase's zone
    set color item zone colors-list - 2
    set shape item zone shapes-list
    set size 9
    set heading random 360
    set scope one-of (list 25 50 100 200)      ; choose radomly movement range
    set immune? false
    set unlocked? false
    set tested? false
    set contagions 0
    set mask-habit? random 100 < %-mask-users  ; set face mask protection social habit
    set wash-habit? random 100 < %-handwashers ; set hand-washing social habit
    set speed step-size ;0.1 ;0.5
    forward 3                                  ; peek out of house to be seen at the beginning
  ]
end

;; Create ambulances
to setup-ambulances
  foreach range zones [ z ->
    create-ambulances ambulances-zone [
      move-to one-of patches with [ zone = z ]
      set homebase patch-here
      set shape "ambulance 2" set size 12 set color white
      set speed 4 * step-size        ; ambulances move faster than people
      let residents (turtle-set healthies sicks) with [ zone = z ]

      ;; The amount of tests supply depends on residents population
      set tests-supply round (count residents * (%-tests / 100) / ambulances-zone )
      set tests-stock tests-supply   ; initialise stock of tests
    ]
  ]
end

;; Replenish the stock of available tests in ambulances
to supply-ambulances
  ask ambulances [ set tests-stock tests-stock + tests-supply ]
end

;; Replenish an infinite stock of tests in ambulances
to supply-infinity
  ask ambulances [ set tests-stock 1000000 ]
end

;; Save a screenshot of the interface (view+plots, etc) in a file
to screenshot
  display
  ;export-interface (word "COVID19-NPI-ABM-" random 1000 random 1000 ".png")
end
@#$#@#$#@
GRAPHICS-WINDOW
478
10
1057
590
-1
-1
1.9
1
10
1
1
1
0
1
1
1
-150
150
-150
150
0
0
1
ticks
30.0

BUTTON
181
29
266
63
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
181
71
266
104
infect
infect
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
272
29
353
63
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1070
39
1394
159
SIRE plot
Hours since start
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Susceptible" 1.0 0 -14439633 true "" "plotxy hour count healthies with [not immune?]"
"Infected" 1.0 0 -5298144 true "" "plotxy hour count sicks"
"Recovered" 1.0 0 -13345367 true "" "plotxy hour count healthies with [immune?]"
"Extinct" 1.0 0 -16777216 true "" "plotxy hour count deaths"

SWITCH
160
395
306
428
case-isolation?
case-isolation?
1
1
-1000

SWITCH
161
476
306
509
sentinel-testing?
sentinel-testing?
1
1
-1000

SLIDER
9
215
142
248
pop-size
pop-size
10
1000
400.0
10
1
NIL
HORIZONTAL

SWITCH
161
354
305
387
total-lockdown?
total-lockdown?
1
1
-1000

BUTTON
360
29
452
64
step
go\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
10
165
142
210
zones
zones
1 4 9 16
2

SWITCH
4
106
117
139
see-sunset?
see-sunset?
1
1
-1000

SWITCH
8
408
141
441
housing?
housing?
0
1
-1000

SLIDER
4
28
117
61
ticks-day
ticks-day
600
2400
1200.0
600
1
NIL
HORIZONTAL

SLIDER
9
473
142
506
avg-duration
avg-duration
3
28
18.0
3
1
NIL
HORIZONTAL

MONITOR
479
10
559
55
Clock:
(word day \"d, \" (hour mod 24) \"h\")
17
1
11

PLOT
1070
173
1394
301
I+CARDS plot
Hours since start
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Infected" 1.0 0 -2674135 true "" "plotxy hour count sicks"
"Confirmed" 1.0 0 -14439633 true "" "plotxy hour count sicks with [confirmed?] "
"Asymptomatic" 1.0 0 -4079321 true "" "plotxy hour count sicks with [asymptomatic?] "
"Risky" 1.0 0 -955883 true "" "plotxy hour count sicks with [risky?] "
"Severe" 1.0 0 -13791810 true "" "plotxy hour count sicks with [severe?] "
"Deadly" 1.0 0 -4699768 true "" "plotxy hour count sicks with [deadly?] "

SWITCH
160
517
307
550
social-distancing?
social-distancing?
0
1
-1000

SLIDER
314
517
463
550
%-willings
%-willings
0
100
50.0
10
1
NIL
HORIZONTAL

SLIDER
313
354
405
387
%-permits
%-permits
5
95
5.0
5
1
NIL
HORIZONTAL

SLIDER
8
512
143
545
%-asymptomatics
%-asymptomatics
0
100
50.0
10
1
NIL
HORIZONTAL

SLIDER
360
114
452
147
%-tests
%-tests
10
100
100.0
10
1
NIL
HORIZONTAL

MONITOR
182
112
267
157
tests stock
sum [tests-stock] of ambulances
17
1
11

SWITCH
160
436
305
469
zone-enforcing?
zone-enforcing?
1
1
-1000

SLIDER
8
369
141
402
ambulances-zone
ambulances-zone
0
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
360
72
452
105
%-spread
%-spread
1
99
5.0
1
1
NIL
HORIZONTAL

BUTTON
273
72
353
105
NIL
outbreak
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
314
476
464
509
zonal?
zonal?
1
1
-1000

BUTTON
273
113
353
149
resupply
supply-ambulances
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
9
254
143
287
%-high-risk
%-high-risk
0
100
30.0
10
1
NIL
HORIZONTAL

SLIDER
8
330
142
363
ICU-beds
ICU-beds
0
.1 * pop-size
1.0
1
1
NIL
HORIZONTAL

MONITOR
305
178
405
223
confirmed (now)
(word n-confirmed \"   (\" count sicks with [confirmed?] \")\")
17
1
11

BUTTON
409
435
464
468
NIL
unlock\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1069
448
1229
568
R0
Hours
NIL
0.0
10.0
0.0
10.0
true
true
"" "set-plot-y-range 0 int min (list doubling-time R0 R0_) + 1"
PENS
"attack" 1.0 0 -2674135 true "" "plotxy hour R0"
"tracing" 1.0 0 -13345367 true "" "plotxy hour R0_"

MONITOR
160
281
269
326
R0 attack (tracing)
(word precision R0 2 \" (\" precision R0_ 2 \")\")
2
1
11

TEXTBOX
7
10
177
40
-- General settings --
12
54.0
1

TEXTBOX
7
455
167
485
-- Covid-19 settings --
12
13.0
1

TEXTBOX
184
10
451
40
----------- Action commands ----------
12
104.0
1

TEXTBOX
168
334
459
364
----- Non-Pharmaceutical Interventions -----
12
72.0
1

TEXTBOX
165
160
479
190
----------------- Monitors ----------------
12
124.0
1

MONITOR
157
229
263
274
deaths (mortality)
(word count deaths \" (\" precision (100 * (count deaths / pop-size)) 1 \"%)\")
17
1
11

MONITOR
152
178
202
223
healthy
count healthies
17
1
11

TEXTBOX
1071
20
1388
50
-------------------- Plots --------------------
12
0.0
1

SLIDER
4
67
117
100
step-size
step-size
0.1
1
0.1
.1
1
NIL
HORIZONTAL

SWITCH
313
395
463
428
home-quarantine?
home-quarantine?
1
1
-1000

MONITOR
408
178
473
223
new today
last sicks-day - item (day - 1) sicks-day
17
1
11

MONITOR
374
229
470
274
herd immunity
(word count healthies with [immune?] \" (\" precision (100 * (count healthies with [immune?] / count healthies)) 1 \"%)\")
17
1
11

MONITOR
360
281
464
326
tested positive
(word n-positives \"/\" n-tested \"  (\" precision (100 * (n-positives / n-tested)) 1 \"%)\")
17
1
11

MONITOR
272
281
357
326
doubling time
doubling-time
2
1
11

MONITOR
266
229
371
274
lethality CFR (IFR)
(word (precision (100 * (count deaths / n-confirmed)) 1 ) \"%  (\" (precision (100 * (count deaths / n-cases)) 1) \"%)\")
17
1
11

MONITOR
205
178
302
223
sicks ever (now)
(word n-cases \"    (\" count sicks \")\")
17
1
11

TEXTBOX
15
146
165
164
--- City settings ---
12
25.0
1

SLIDER
8
292
142
325
hospital-beds
hospital-beds
0
.1 * pop-size
12.0
1
1
NIL
HORIZONTAL

PLOT
1069
315
1394
435
Hospital and ICU demand
Hours since start
NIL
0.0
10.0
0.0
10.0
true
true
"set-plot-y-range 0 hospital-beds + 1" ""
PENS
"beds (H)" 1.0 0 -4079321 true "" "plotxy hour hospital-beds"
"demand (H)" 1.0 0 -955883 true "" "plotxy hour count sicks with [severe?]"
"beds (ICU)" 1.0 0 -8990512 true "" "plotxy hour ICU-beds"
"demand (ICU)" 1.0 0 -14070903 true "" "plotxy hour count sicks with [deadly?]"

SLIDER
121
28
154
139
end-day
end-day
10
300
60.0
10
1
NIL
VERTICAL

PLOT
1234
448
1394
568
Doubling Time
Hours
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -955883 true "" "plotxy hour doubling-time"

SLIDER
160
558
305
591
%-mask-users
%-mask-users
0
100
50.0
5
1
NIL
HORIZONTAL

SLIDER
313
558
463
591
%-handwashers
%-handwashers
0
100
50.0
5
1
NIL
HORIZONTAL

MONITOR
9
553
143
598
last contagion risk
p-contagion
17
1
11

SLIDER
314
435
406
468
pick-zone
pick-zone
1
zones
5.0
1
1
NIL
HORIZONTAL

BUTTON
408
354
463
387
NIL
lock3X4\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This is an extension of the "COVID-19 epidemics with Non-Pharmaceutical Interventions and zonal restraints" model (http://modelingcommons.org/browse/one_model/6374), that incorporates some additional interventions including personal hygiene habits (mask-using and hand-washing) as an individual choice, plus an intermittent 3x4 or "accordion lockdown" (the lockdown is periodically enforced 3 days and lifted the following 4 days), along with the possibility of unlocking a particular zone or set of zones. Such new changes implemented in this extended version are documented in the WHAT IS NEW and NEW THINGS TO NOTICE sections at the end of this page. The original model description is given below.
 
The model is an agent-based system to simulate the evolution of the COVID-19 epidemics and the mitigation impact of Non-Pharmaceutical Interventions (NPI) within a toy city. NPIs are the mechanisms that public health offices around the world are using to attempt to contain the spread of the virus. 

We consider a number of NPIs, including social distancing, case isolation, home quarantine, total lockdown, sentinel testing, mask protection and a proposed zonal restriction where these interventions can be applied to separated districts or zones of the city. The effect of these strategies are measured in terms of morbidity, mortality, lethality -infection fatality rate (IFR) and case fatality rate (CFR)-, doubling time, reproduction number and plots of infection, recovery and death during the simulation timeline. The main purpose of the tool is to let modelers assess which NPIs or combination of NPIs can help flattening or crushing the curve of spread of the disease (mitigate or suppress, respectively), and the effects on the corresponding epidemics indicators. 


## HOW IT WORKS

The model implements the NPIs within the epidemic SIRE+CARDS model (see [1]) as well as the agent behavior rules. We build upon the compartmental SIRE epidemic framework (Susceptible-Infectious-Recovered-Extinct), but we regarded the Infectious compartment as an extended-state consisting of a number of conditions: Confirmed or not (indicating the patient will remain isolated), Risky or not (meaning predisposition to develop severe or critical disease), Asymptomatic or not (meaning patient is unaware of being a virus carrier), Severe or not (meaning it requires to be hospitalized to avoid dead), and Deadly or not (indicating the patient requires Intensive Care Unit -ICU- assistance). The details of event transition rules and condition activation are described in [1].

The simulation tracks an individual disease path for every agent, from susceptible to recovery or death, according to the SIRE+CARDS epidemic model described above. Each agent is assigned a daily routine consisting of going outdoors and returning home with a commuting distance range randomly chosen from {25, 50, 100, 200} unit steps; the actual route the agent follows varies slightly due to random fluctuations in his orientation. In addition the length of each step can be set as a global parameter between 0.1 and 1 units. Similarly, the day length can be defined with a given number of ticks in the range between 600 and 2400. 

Individuals interact randomly with other agents around. Virus transmission occurs due to proximity of an infectious agent with other susceptible agent within a spatial radius of 0.5 units; the chance of contagion depends on what mask protection intervention is applied. On recovery from the disease, individuals acquire immunity to the virus so they cannot be re-infected. In this model all deaths are considered to be caused by COVID-19. 
The flow of events applied to each agent at each timestep are: lifestyle, epidemic, isolation, quarantine, distancing, sentinel, lockdown, illness, clock and indicators (see code and [1] for further details).

In the simulation view area, agents are represented with different shapes according to the zone where they reside. The color of the agent represents its extended state (healthy: same color as zone ground; immune: white; sick: red or yellow if asymptomatic; dead: black X). Special agents intended to implement some of the NPIs such as households for home-quarantine and ambulances for sentinel-testing can also be seen.

## HOW TO USE IT

The control panel is organised in sections related to general, city and COVID-19 settings, monitors of epidemic indicators, parameters, action commands to execute the simulation, and a dedicated section for NPI activation with their corresponding parameters. 

A typical setup for a simulation run is: POP-SIZE=400 (total number of simulated people or agents), ZONES=9 (number of residential zones), DAYS=60 (period of observation days or simulation length), %-HIGH-RISK=30 (percentage of population with co-morbidities), HOSPITAL-BEDS=12 (total number of hospital beds available), ICU-BEDS=2 (total number of ICU beds available), AMBULANCES-ZONE=1 (number of ambulances or sentinels per zone), HOUSING?=on (show households in the view area), AVG-DURATION=18 (average day period to recover from illness), %-ASYMPTOMATIC=50 (percentage of patients showing mild or no symptoms), TICKS-DAY=1200 (number of ticks after one day elapse), STEP-SIZE=0.1 (distance each agent move every tick), END-DAY=60 (simulation timeline). 

Once setup is done, activate or deactivate the NPIs you want to assess (the details of meaning and implementation of each NPI can be found in [1]), then press SETUP, then GO. After a few simulated hours (or ticks) you can seed a patient zero randomly (or many other, if you want) by pressing INFECT. Alternatively, using OUTBREAK you can randomly infect a percentage (%-SPREAD) of the population. Afterwards, you will be able to see the emergence of the epidemic SIRE curves in the plot area, as well as the other epidemic indicators in the monitor area, while the contagion, recover and death of people unfolds as a result of the development of the COVID-19 epidemics with such particular settings. 

## THINGS TO NOTICE

The NPIs control panel can be used to handle their application even while the epidemic unfolds. For example, you can activate/disactivate a total lockdown at any time, and it will take effect at the 00:00 of the next simulated day. Or you can isolate or release confirmed cases inmmediately, by switching it on or off. Similar thing occurs with the rest of NPIs. In contrast, you should not change any of the other parameters (general, city, or COVID-19 settings), as they are used to setup the model just and only before running the simulation.

There are two distinctive features in this model that allows the application of NPIs with a zonal scope. This means, firstly, you can restrict mobility of agents within their zone of residence; hence, you can try and see if this kind of restriction (ZONE-ENFORCING) is advantageous in flattening the curve when combined with other NPIs compared to the single application of said NPIs. Secondly, when enforcing a TOTAL-LOCKDOWN, you are able to lift this NPI in an arbitrary zone or zones using the PICK-ZONE slider and UNLOCK-ZONE button. Again, you can see how the resulting emergencies compare to maintaining the total lockdown in the entire city. These features may yield interesting insights about alternative implementation of these measures that usually have a hard impact on the economy of the city. Notice that TOTAL-LOCKDOWN has an associated parameter %-PERMITS that determines the amount of agents permitted to go out of their households.

The SENTINEL-TESTING intervention is also useful to assess the impact of mass-test campaigns in the population. Sentinels are shown as ambulances moving around the city while testing any agent they find in their ways; again, you can restrict sentinels mobility with a zonal scope (ZONAL?) and see the difference with respect to allowing city-wide mobility. Notice that since test kits are a finite resource, the model assigns an initial stock of tests equals to the population size. When ambulances run out of tests they stop and stay still. You can of course replenish the stock for each ambulance using the RESUPPLY button to re-activate sentinels' journeys.


## THINGS TO TRY

Start-off by defining the general, city, COVID-19 and NPI settings you want to simulate in a single run (a suggested set of settings was given above, see HOW TO USE IT). Execute the simulation to observe how the contagion spreads and how the epidemics evolve by inspecting the monitors, plots and view areas. Then try changing the NPI parameters to assess their mitigation impact by comparing with previous outputs (see also THINGS TO NOTICE). If you like, start running the simulation with only one zone (choose 1 in the ZONES pull down menu); this will be equivalent to have a single district in the entire city, so agents will to move around freely. Then experiment simulations with  many districts or multi-zone (choose ZONES in {4, 9, 16}), and try to apply ZONE-ENFORCING to see how agents mobility becomes restricted to their zones of residence.

The HOUSING? switch is used as a decoration and as a behaviour feature. When this switch is on, the model shows households in the simulation view area and assigns agents randomly to each one with a proportion of one house per four agents approximately. When this switch is off, the household of an agent would be the patch were he/she was created randomly at the beginning of the simulation; no houses will be seen. If you try the CASE-ISOLATION or TOTAL-LOCKDOWN interventions with HOUSING? on, agents will be sent home and stay confined in their corresponding households (they will be hidden as they stay inside the house); in contrast, if HOUSING? is off, they will move to their home patch and stay quiet there, being visible and showing their shape, color and state.

Finally, the SEE-SUNSET is a decoration feature. Try it and surprise yourself!

## WHAT IS NEW

This extended version includes two personal health interventions implemented as an individual choice, namely using face masks at all times and hand-washing regularly during the day. For this purpose, two sliders, %-MASK-USERS and %-HAND-WASHERS have been enabled. These two parameters will define the tendency of an agent to develop those two hygiene habits. Recent studies have suggested that these two personal interventions can have positive effect in mitigating the contagion of the coronavirus (see [2] and references within). This is reasonable, as they reduce the risk of getting infected when a healthy agent encounters an infectious agent, which may be even more effective when physical distancing to other agents is observed (try and see it!). 

Given that these three interventions depend mostly on individual decisions rather than government policy, it would be of great interest in the post-COVID19 scenario, to study strategies where they are combined with other interventions not affecting the mobility of the agents (such as SENTINEL-TESTING) to continue with epidemic mitigation while opening economy and social interaction.    

Regarding mobility restrictions, when the TOTAL-LOCKDOWN intervention is on, in this extended model it is possible to lift it independently on the agents residing in a particular zone. For this aim, the user can choose the desired zone number with the PICK-ZONE slider and then should press the UNLOCK button. Then, the agents residing in such zone will start immediately to get out of their households and move freely around, while the rest of the population stays at home, as it can be attested in the simulation view area. When the ZONE-ENFORCING intervention is simultaneously activated, its residents will move around but only within the zone limits.

In addition, this extended version enables the application of the TOTAL-LOCKDOWN intervention with an intermittent 3x4 schedule (the so called "accordion" scheme), where the mobility restriction is applied adhering to a 7-day loop consisting of enforcing stay-at-home order during 3 consecutive days, and lifting the rule the remaining 4 days (hoping in this way to ameliorate the negative impact of shutting down economy due to indefinite lockdowns). To apply the accordion intervention, the user can press the LOCK3x4 button to activate/inactivate this scheme; when activated, a "3x4 ON" sign will be displayed in the bottom-left corner of the simulation view area. We remark that once activated, this intervention, similarly to the TOTAL-LOCKDOWN takes effect starting at zero hours of the next day (you can check it on the CLOCK monitor).

## NEW THINGS TO NOTICE

Observe that now the chance of a contagion depends on the individual decisions regarding keeping physical distance, using mask and washing hands, which varies amply depending on the habits of the two agents involved in a contagion encounter. The actual risk on a particular encounter can be seen in the LAST CONTAGION RISK monitor. You can try and see how the risks fluctuate as you change the combination of sliders of %-WILLINGS to keep personal distance, and %-MASK-USERS and %-HAND-WASHERS with values between 0 and 100 (these parameters have effect only at the beginning of the simulation, when the population is created).

Now, when the ZONE-ENFORCING and TOTAL-LOCKDOWN interventions are simultaneously been applied, it would be interesting to verify if releasing a particular single zone or a couple of them having low contagion rates, would be advantageous to facilitate mobility (and encourage economic recovering) of a group of the population without worsening too much the overall epidemic indicators of the city.

Lastly, the impact of the "accordion" LOCK3x4 intervention is evident in the epidemic SIRE+CARDS curves, which adopt a "staircase" or "chainsaw" shape with discrete rising and falling steps, corresponding to the periods of intermittent enforce/lift of the lockdown. 

## EXTENDING THE MODEL

Given the complex nature of human behavior and virus infection, attempting to model every mechanism of the COVID-19 epidemic may prove difficult; necessary assumptions need to be taken to simplify the representation of the agents and their interactions. We have made a few of them, as described above, so evidently there are interesting aspects that can be addressed to extend the model. We mention just some ideas here (for a more comprehensive discussion we refer the reader to [1]):  

- consider virus incubation periods after infection, 
- model infectiousness, aggressiveness or symptomatic severity levels, 
- include age or gender structures, 
- account for births or deaths due to other causes, 
- expand further risk stratification for co-morbidities and age windows, 
- add hubs or attractor sites (i.e. mass transport, schools, cinemas, etc.),
- incorporate explicit sites for hospitals were severe or critical patients are moved instead of staying still, 
- set up daily schedules for agent routines (e.g. going forth and back to/from work, or leisure or shopping, etc.), 
- measure the economical impact of NPIs (model food or money supply at a population or individual level), 
- simulate cotagions not only by direct contact, but also indirect by surface contact (virus particles adhered to patches), 
- implement of other epidemic indicators (different estimates of R0) or measuring them within subpopulations corresponding to residents of the different zones,
- design mechanisms allowing tendencies to develop personal health habits (like wearing masks and routinely washing hands) to emerge as consequence of changes in the psychological biases or beliefs of the agents.

## CREDITS AND REFERENCES

Authors: 

Sergio Rojas-Galeano and Lindsay Alvarez 

Copyright (c) September 2020

email: srojas@udistrital.edu.co, lalvarez@udistrital.edu.co

Version 1.27

Licenses:

- The model code is licensed as GNU General Public License (GPLv3) 
  (see https://www.gnu.org/licenses/gpl-3.0.txt)

- This Info Tab document is licensed as CC BY-NC-ND 
  (see https://creativecommons.org/licenses/by-nc-nd/4.0/)

References:

[1] Alvarez, L. and Rojas-Galeano, S. "Simulation of Non-Pharmaceutical Interventions on      COVID-19 with an Agent-based Model of Zonal Restraint". medRxiv pre-print 2020/06/13;     https://www.medrxiv.org/content/10.1101/2020.06.13.20130542v1
    doi: 10.1101/2020.06.13.20130542

[2] Alvarez, L. and Rojas-Galeano, S. "Impact of Personal Care Habits on Post-Lockdown        COVID-19 Contagion: Insights from Agent-based Simulations". medRxiv pre-print 2020/10/01; https://www.medrxiv.org/content/10.1101/2020.09.23.20200212v2
doi: 10.1101/2020.09.23.20200212
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

ambulance
false
0
Rectangle -7500403 true true 30 90 210 195
Polygon -7500403 true true 296 190 296 150 259 134 244 104 210 105 210 190
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Circle -16777216 true false 69 174 42
Rectangle -1 true false 288 158 297 173
Rectangle -1184463 true false 289 180 298 172
Rectangle -2674135 true false 29 151 298 158
Line -16777216 false 210 90 210 195
Rectangle -16777216 true false 83 116 128 133
Rectangle -16777216 true false 153 111 176 134
Line -7500403 true 165 105 165 135
Rectangle -7500403 true true 14 186 33 195
Line -13345367 false 45 135 75 120
Line -13345367 false 75 135 45 120
Line -13345367 false 60 112 60 142

ambulance 2
false
0
Rectangle -7500403 true true 4 15 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42
Rectangle -2674135 true false 30 75 165 120
Rectangle -2674135 true false 75 30 120 165

ant 2
true
0
Polygon -7500403 true true 150 19 120 30 120 45 130 66 144 81 127 96 129 113 144 134 136 185 121 195 114 217 120 255 135 270 165 270 180 255 188 218 181 195 165 184 157 134 170 115 173 95 156 81 171 66 181 42 180 30
Polygon -7500403 true true 150 167 159 185 190 182 225 212 255 257 240 212 200 170 154 172
Polygon -7500403 true true 161 167 201 150 237 149 281 182 245 140 202 137 158 154
Polygon -7500403 true true 155 135 185 120 230 105 275 75 233 115 201 124 155 150
Line -7500403 true 120 36 75 45
Line -7500403 true 75 45 90 15
Line -7500403 true 180 35 225 45
Line -7500403 true 225 45 210 15
Polygon -7500403 true true 145 135 115 120 70 105 25 75 67 115 99 124 145 150
Polygon -7500403 true true 139 167 99 150 63 149 19 182 55 140 98 137 142 154
Polygon -7500403 true true 150 167 141 185 110 182 75 212 45 257 60 212 100 170 146 172

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee 2
true
0
Polygon -1184463 true false 195 150 105 150 90 165 90 225 105 270 135 300 165 300 195 270 210 225 210 165 195 150
Rectangle -16777216 true false 90 165 212 185
Polygon -16777216 true false 90 207 90 226 210 226 210 207
Polygon -16777216 true false 103 266 198 266 203 246 96 246
Polygon -6459832 true false 120 150 105 135 105 75 120 60 180 60 195 75 195 135 180 150
Polygon -6459832 true false 150 15 120 30 120 60 180 60 180 30
Circle -16777216 true false 105 30 30
Circle -16777216 true false 165 30 30
Polygon -7500403 true true 120 90 75 105 15 90 30 75 120 75
Polygon -16777216 false false 120 75 30 75 15 90 75 105 120 90
Polygon -7500403 true true 180 75 180 90 225 105 285 90 270 75
Polygon -16777216 false false 180 75 270 75 285 90 225 105 180 90
Polygon -7500403 true true 180 75 180 90 195 105 240 195 270 210 285 210 285 150 255 105
Polygon -16777216 false false 180 75 255 105 285 150 285 210 270 210 240 195 195 105 180 90
Polygon -7500403 true true 120 75 45 105 15 150 15 210 30 210 60 195 105 105 120 90
Polygon -16777216 false false 120 75 45 105 15 150 15 210 30 210 60 195 105 105 120 90
Polygon -16777216 true false 135 300 165 300 180 285 120 285

bird
false
0
Polygon -7500403 true true 135 165 90 270 120 300 180 300 210 270 165 165
Rectangle -7500403 true true 120 105 180 237
Polygon -7500403 true true 135 105 120 75 105 45 121 6 167 8 207 25 257 46 180 75 165 105
Circle -16777216 true false 128 21 42
Polygon -7500403 true true 163 116 194 92 212 86 230 86 250 90 265 98 279 111 290 126 296 143 298 158 298 166 296 183 286 204 272 219 259 227 235 240 241 223 250 207 251 192 245 180 232 168 216 162 200 162 186 166 175 173 171 180
Polygon -7500403 true true 137 116 106 92 88 86 70 86 50 90 35 98 21 111 10 126 4 143 2 158 2 166 4 183 14 204 28 219 41 227 65 240 59 223 50 207 49 192 55 180 68 168 84 162 100 162 114 166 125 173 129 180

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

butterfly 2
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7500403 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7500403 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7500403 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7500403 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7500403 true 167 47 159 82
Line -7500403 true 136 47 145 81
Circle -7500403 true true 165 45 8
Circle -7500403 true true 134 45 6
Circle -7500403 true true 133 44 7
Circle -7500403 true true 133 43 8

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

cat
false
0
Line -7500403 true 285 240 210 240
Line -7500403 true 195 300 165 255
Line -7500403 true 15 240 90 240
Line -7500403 true 285 285 195 240
Line -7500403 true 105 300 135 255
Line -16777216 false 150 270 150 285
Line -16777216 false 15 75 15 120
Polygon -7500403 true true 300 15 285 30 255 30 225 75 195 60 255 15
Polygon -7500403 true true 285 135 210 135 180 150 180 45 285 90
Polygon -7500403 true true 120 45 120 210 180 210 180 45
Polygon -7500403 true true 180 195 165 300 240 285 255 225 285 195
Polygon -7500403 true true 180 225 195 285 165 300 150 300 150 255 165 225
Polygon -7500403 true true 195 195 195 165 225 150 255 135 285 135 285 195
Polygon -7500403 true true 15 135 90 135 120 150 120 45 15 90
Polygon -7500403 true true 120 195 135 300 60 285 45 225 15 195
Polygon -7500403 true true 120 225 105 285 135 300 150 300 150 255 135 225
Polygon -7500403 true true 105 195 105 165 75 150 45 135 15 135 15 195
Polygon -7500403 true true 285 120 270 90 285 15 300 15
Line -7500403 true 15 285 105 240
Polygon -7500403 true true 15 120 30 90 15 15 0 15
Polygon -7500403 true true 0 15 15 30 45 30 75 75 105 60 45 15
Line -16777216 false 164 262 209 262
Line -16777216 false 223 231 208 261
Line -16777216 false 136 262 91 262
Line -16777216 false 77 231 92 261

chess knight
false
0
Line -16777216 false 75 255 225 255
Polygon -7500403 true true 90 255 60 255 60 225 75 180 75 165 60 135 45 90 60 75 60 45 90 30 120 30 135 45 240 60 255 75 255 90 255 105 240 120 225 105 180 120 210 150 225 195 225 210 210 255
Polygon -16777216 false false 210 255 60 255 60 225 75 180 75 165 60 135 45 90 60 75 60 45 90 30 120 30 135 45 240 60 255 75 255 90 255 105 240 120 225 105 180 120 210 150 225 195 225 210
Line -16777216 false 255 90 240 90
Circle -16777216 true false 134 63 24
Line -16777216 false 103 34 108 45
Line -16777216 false 80 41 88 49
Line -16777216 false 61 53 70 58
Line -16777216 false 64 75 79 75
Line -16777216 false 53 100 67 98
Line -16777216 false 63 126 69 123
Line -16777216 false 71 148 77 145
Rectangle -7500403 true true 90 255 210 300
Rectangle -16777216 false false 90 255 210 300

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cow skull
false
0
Polygon -7500403 true true 150 90 75 105 60 150 75 210 105 285 195 285 225 210 240 150 225 105
Polygon -16777216 true false 150 150 90 195 90 150
Polygon -16777216 true false 150 150 210 195 210 150
Polygon -16777216 true false 105 285 135 270 150 285 165 270 195 285
Polygon -7500403 true true 240 150 263 143 278 126 287 102 287 79 280 53 273 38 261 25 246 15 227 8 241 26 253 46 258 68 257 96 246 116 229 126
Polygon -7500403 true true 60 150 37 143 22 126 13 102 13 79 20 53 27 38 39 25 54 15 73 8 59 26 47 46 42 68 43 96 54 116 71 126

cylinder
false
0
Circle -7500403 true true 0 0 300

dog
false
0
Polygon -7500403 true true 300 165 300 195 270 210 183 204 180 240 165 270 165 300 120 300 0 240 45 165 75 90 75 45 105 15 135 45 165 45 180 15 225 15 255 30 225 30 210 60 225 90 225 105
Polygon -16777216 true false 0 240 120 300 165 300 165 285 120 285 10 221
Line -16777216 false 210 60 180 45
Line -16777216 false 90 45 90 90
Line -16777216 false 90 90 105 105
Line -16777216 false 105 105 135 60
Line -16777216 false 90 45 135 60
Line -16777216 false 135 60 135 45
Line -16777216 false 181 203 151 203
Line -16777216 false 150 201 105 171
Circle -16777216 true false 171 88 34
Circle -16777216 false false 261 162 30

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

fish 2
false
0
Polygon -1 true false 56 133 34 127 12 105 21 126 23 146 16 163 10 194 32 177 55 173
Polygon -7500403 true true 156 229 118 242 67 248 37 248 51 222 49 168
Polygon -7500403 true true 30 60 45 75 60 105 50 136 150 53 89 56
Polygon -7500403 true true 50 132 146 52 241 72 268 119 291 147 271 156 291 164 264 208 211 239 148 231 48 177
Circle -1 true false 237 116 30
Circle -16777216 true false 241 127 12
Polygon -1 true false 159 228 160 294 182 281 206 236
Polygon -7500403 true true 102 189 109 203
Polygon -1 true false 215 182 181 192 171 177 169 164 152 142 154 123 170 119 223 163
Line -16777216 false 240 77 162 71
Line -16777216 false 164 71 98 78
Line -16777216 false 96 79 62 105
Line -16777216 false 50 179 88 217
Line -16777216 false 88 217 149 230

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

ghost
false
0
Polygon -7500403 true true 30 165 13 164 -2 149 0 135 -2 119 0 105 15 75 30 75 58 104 43 119 43 134 58 134 73 134 88 104 73 44 78 14 103 -1 193 -1 223 29 208 89 208 119 238 134 253 119 240 105 238 89 240 75 255 60 270 60 283 74 300 90 298 104 298 119 300 135 285 135 285 150 268 164 238 179 208 164 208 194 238 209 253 224 268 239 268 269 238 299 178 299 148 284 103 269 58 284 43 299 58 269 103 254 148 254 193 254 163 239 118 209 88 179 73 179 58 164
Line -16777216 false 189 253 215 253
Circle -16777216 true false 102 30 30
Polygon -16777216 true false 165 105 135 105 120 120 105 105 135 75 165 75 195 105 180 120
Circle -16777216 true false 160 30 30

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

house colonial
false
0
Rectangle -7500403 true true 270 75 285 255
Rectangle -7500403 true true 45 135 270 255
Rectangle -16777216 true false 124 195 187 256
Rectangle -16777216 true false 60 195 105 240
Rectangle -16777216 true false 60 150 105 180
Rectangle -16777216 true false 210 150 255 180
Line -16777216 false 270 135 270 255
Polygon -7500403 true true 30 135 285 135 240 90 75 90
Line -16777216 false 30 135 285 135
Line -16777216 false 255 105 285 135
Line -7500403 true 154 195 154 255
Rectangle -16777216 true false 210 195 255 240
Rectangle -16777216 true false 135 150 180 180

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

monster
false
0
Polygon -7500403 true true 75 150 90 195 210 195 225 150 255 120 255 45 180 0 120 0 45 45 45 120
Circle -16777216 true false 165 60 60
Circle -16777216 true false 75 60 60
Polygon -7500403 true true 225 150 285 195 285 285 255 300 255 210 180 165
Polygon -7500403 true true 75 150 15 195 15 285 45 300 45 210 120 165
Polygon -7500403 true true 210 210 225 285 195 285 165 165
Polygon -7500403 true true 90 210 75 285 105 285 135 165
Rectangle -7500403 true true 135 165 165 270

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person doctor
false
0
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Polygon -13345367 true false 135 90 150 105 135 135 150 150 165 135 150 105 165 90
Polygon -7500403 true true 105 90 60 195 90 210 135 105
Polygon -7500403 true true 195 90 240 195 210 210 165 105
Circle -7500403 true true 110 5 80
Rectangle -7500403 true true 127 79 172 94
Polygon -1 true false 105 90 60 195 90 210 114 156 120 195 90 270 210 270 180 195 186 155 210 210 240 195 195 90 165 90 150 150 135 90
Line -16777216 false 150 148 150 270
Line -16777216 false 196 90 151 149
Line -16777216 false 104 90 149 149
Circle -1 true false 180 0 30
Line -16777216 false 180 15 120 15
Line -16777216 false 150 195 165 195
Line -16777216 false 150 240 165 240
Line -16777216 false 150 150 165 150

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

rabbit
false
0
Polygon -7500403 true true 61 150 76 180 91 195 103 214 91 240 76 255 61 270 76 270 106 255 132 209 151 210 181 210 211 240 196 255 181 255 166 247 151 255 166 270 211 270 241 255 240 210 270 225 285 165 256 135 226 105 166 90 91 105
Polygon -7500403 true true 75 164 94 104 70 82 45 89 19 104 4 149 19 164 37 162 59 153
Polygon -7500403 true true 64 98 96 87 138 26 130 15 97 36 54 86
Polygon -7500403 true true 49 89 57 47 78 4 89 20 70 88
Circle -16777216 true false 37 103 16
Line -16777216 false 44 150 104 150
Line -16777216 false 39 158 84 175
Line -16777216 false 29 159 57 195
Polygon -5825686 true false 0 150 15 165 15 150
Polygon -5825686 true false 76 90 97 47 130 32
Line -16777216 false 180 210 165 180
Line -16777216 false 165 180 180 165
Line -16777216 false 180 165 225 165
Line -16777216 false 180 210 210 240

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

spider
true
0
Polygon -7500403 true true 134 255 104 240 96 210 98 196 114 171 134 150 119 135 119 120 134 105 164 105 179 120 179 135 164 150 185 173 199 195 203 210 194 240 164 255
Line -7500403 true 167 109 170 90
Line -7500403 true 170 91 156 88
Line -7500403 true 130 91 144 88
Line -7500403 true 133 109 130 90
Polygon -7500403 true true 167 117 207 102 216 71 227 27 227 72 212 117 167 132
Polygon -7500403 true true 164 210 158 194 195 195 225 210 195 285 240 210 210 180 164 180
Polygon -7500403 true true 136 210 142 194 105 195 75 210 105 285 60 210 90 180 136 180
Polygon -7500403 true true 133 117 93 102 84 71 73 27 73 72 88 117 133 132
Polygon -7500403 true true 163 140 214 129 234 114 255 74 242 126 216 143 164 152
Polygon -7500403 true true 161 183 203 167 239 180 268 239 249 171 202 153 163 162
Polygon -7500403 true true 137 140 86 129 66 114 45 74 58 126 84 143 136 152
Polygon -7500403 true true 139 183 97 167 61 180 32 239 51 171 98 153 137 162

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

wolf 7
false
0
Circle -16777216 true false 183 138 24
Circle -16777216 true false 93 138 24
Polygon -7500403 true true 30 105 30 150 90 195 120 270 120 300 180 300 180 270 210 195 270 150 270 105 210 75 90 75
Polygon -7500403 true true 255 105 285 60 255 0 210 45 195 75
Polygon -7500403 true true 45 105 15 60 45 0 90 45 105 75
Circle -16777216 true false 90 135 30
Circle -16777216 true false 180 135 30
Polygon -16777216 true false 120 300 150 255 180 300

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Do Nothing (baseline)" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Social Distancing" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Case Isolation" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Total Lockdown" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Sentinel Testing" repetitions="30" runMetricsEveryStep="false">
    <setup>setup
supply-infinity</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Zone Enforcing" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Mask Protection: Everybody" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="SD+ZE" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="CI+ZE" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="TL+ZE" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="ST+ZE" repetitions="30" runMetricsEveryStep="false">
    <setup>setup
supply-infinity</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="MP Everybody+ZE" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Mask Protection: Sicks only" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Mask Protection: Healthy only" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="SD+MP" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-sicks?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mask-healthies?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Personal Habits" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <final>display</final>
    <metric>count healthies</metric>
    <metric>count deaths</metric>
    <metric>count healthies with [immune?]</metric>
    <metric>count healthies with [immune?] / count healthies</metric>
    <metric>count deaths / n-confirmed</metric>
    <metric>count deaths / n-cases</metric>
    <metric>R0</metric>
    <metric>R0_</metric>
    <metric>doubling-time</metric>
    <metric>n-cases</metric>
    <metric>n-confirmed</metric>
    <metric>n-tested</metric>
    <metric>n-positives</metric>
    <enumeratedValueSet variable="ticks-day">
      <value value="1200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="step-size">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="see-sunset?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-day">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zones">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pop-size">
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-high-risk">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="hospital-beds">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ICU-beds">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ambulances-zone">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="avg-duration">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-asymptomatics">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-spread">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-tests">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-willings">
      <value value="0"/>
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="case-isolation?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="home-quarantine?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-lockdown?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-permits">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sentinel-testing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zonal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="zone-enforcing?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-mask-users">
      <value value="0"/>
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%-handwashers">
      <value value="0"/>
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
