
// Software copyright  Anthony H. Dekker, 2013
// Java code for the tutorial at http://scientificgems.wordpress.com/2013/12/11/integrating-netlogo-and-java-part-1/

import org.nlogo.api.*;
import java.util.*;

/*======================*/
/* Main extension class */
/*======================*/

public class DiffusionExtension extends DefaultClassManager {
	public static final String VERSION = "1.2"; // these are not essential, but I'm using them in the "about" reporter
 	public static final String ABOUT = "Patch diffusion demo, copyright  Anthony H. Dekker, 2013";

 	public static final int PATCH_VALUE_INDEX = 5;

	public void load(PrimitiveManager primitiveManager) {
		
		// Usage: diffusion:about
		primitiveManager.addPrimitive("about", new AboutDiffusion());
		
		// Usage: diffusion:init-grid world-width world-height min-pxcor min-pycor
		primitiveManager.addPrimitive("init-grid", new InitDiffusionGrid());

		// Usage: diffusion:set-grid-cell grid pxcor pycor value
		primitiveManager.addPrimitive("set-grid-cell", new SetDiffusionGridCell());

		// Usage: diffusion:get-grid-cell grid pxcor pycor
		primitiveManager.addPrimitive("get-grid-cell", new GetDiffusionGridCell());

		// Usage: diffusion:diffusion-step grid factor
		primitiveManager.addPrimitive("diffusion-step", new DiffusionStep());

		// Usage: diffusion:copy-patches-to-grid grid patches
		primitiveManager.addPrimitive("copy-patches-to-grid", new SetGridFromPatches());

		// Usage: diffusion:copy-grid-to-patches grid patches
		primitiveManager.addPrimitive("copy-grid-to-patches", new SetPatchesFromGrid());

	}

	public org.nlogo.api.ExtensionObject readExtensionObject(org.nlogo.api.ExtensionManager reader, String typeName, String value) throws org.nlogo.api.ExtensionException {
		return DiffusionGrid.parse(value);
	}
	
	public static void main (String [] args) throws ExtensionException { // non-essential test code
		System.err.println (ABOUT + "; Version " + VERSION);
	}
}

/*==================*/
/* "about" reporter */
/*==================*/

class AboutDiffusion extends DefaultReporter { // this template defines a reporter
	public Syntax getSyntax() { // no arguments, report a String
		return Syntax.reporterSyntax(new int[] {}, Syntax.StringType());
	}
  
	public Object report(Argument args[], Context context) throws ExtensionException {
		return DiffusionExtension.ABOUT + "; Version " + DiffusionExtension.VERSION;
	}
}

/*=====================*/
/* DiffusionGrid class */
/*=====================*/

class DiffusionGrid implements org.nlogo.api.ExtensionObject { // new NetLogo data types defined by extensions must implement this interface
    public final int width;
    public final int height;
    public final int loX;
    public final int loY;
    private double [] [] grid;

	public DiffusionGrid (int w, int h, int loX, int loY) {
		this.width = w;
		this.height = h;
		this.loX = loX;
		this.loY = loY;
		this.grid = new double [w] [h];
    }
	
	public void setValue (int x, int y, double v) { // set one entry in the diffusion grid
		grid [x-loX] [y-loY] = v;
	}
	
	public void setRandom (Random r, double max) { // randomly set all entries in the diffusion grid
		for (int i = 0; i < width; i++) {
			for (int j = 0; j < height; j++) {
				grid [i] [j] = max * r.nextDouble ();
			}
		}
	}
	
	public double getValue (int x, int y) { // get one entry in the diffusion grid
		return grid [x-loX] [y-loY];
	}
	
	public void step (double diffusionFactor) { // perform one diffision step
		double [] [] temp = new double [width] [height];
		for (int i = 0; i < width; i++) {
			for (int j = 0; j < height; j++) {
				double a = (i == 0 ? 0 : grid [i-1] [j]);
				double b = (i == width-1 ? 0 : grid [i+1] [j]);
				double c = (j == 0 ? 0 : grid [i] [j-1]);
				double d = (j == height-1 ? 0 : grid [i] [j+1]);
				int n = (i == 0 ? 0 : 1) + (i == width-1 ? 0 : 1) + (j == 0 ? 0 : 1) + (j == height-1 ? 0 : 1);
				temp [i] [j] = (1 - diffusionFactor) * grid [i] [j] + diffusionFactor * (a + b + c + d) / n;
			}
		}
		for (int i = 0; i < width; i++) {
			for (int j = 0; j < height; j++) {
				grid [i] [j] = temp [i] [j];
			}
		}
	}

    public boolean equals(Object obj) { // equality test
      return this == obj;
    }

    public String getExtensionName() { // return extension name
		return "diffusion";
    }

    public String getNLTypeName() {
		// since this extension only defines one type, we don't need to give it a name (ok to just return empty string)
		return "DiffusionGrid";
    }
	
	public boolean recursivelyEqual(Object obj) { // should really do an item-by-item comparison here
		return this == obj;
	}
	
	public static DiffusionGrid parse (String value) { // decode a string into a grid
		int k = value.indexOf (":");
		String s = value.substring (0, k);
		String body = value.substring (k+1);
		k = s.indexOf ("x");
		int w = Integer.parseInt(s.substring (0, k));
		int h = Integer.parseInt (s.substring (k+1));
		int lox = -(w/2);
		int loy = -(h/2);
		DiffusionGrid grid = new DiffusionGrid (w, h, lox, loy);
		k = 0;
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				grid.setValue (i+lox, j+loy, decode(body.charAt (k++)));
			}
		}
		return grid;
	}
	
	private static int encode (double v) { // assume 0..9.9 range, and return a stupidly simplistic text equivalent of a number
		return 'A' + ((int) (v * 26.0 / 10));
	}
	
	private static double decode (int ch) {
		return (ch - 'A') * 10.0 / 26;
	}
	
	public String dump(boolean readable, boolean exportable, boolean reference) { // dump a grid to a string
		StringBuilder sb = new StringBuilder (width +  "x" + height + ":");
		for (int i = 0; i < width; i++) {
			for (int j = 0; j < height; j++) {
				sb.append ((char) encode(grid [i] [j]));
			}
		}
		return sb.toString ();
	}
}

/*======================================*/
/* DiffusionGrid reporters and commands */
/*======================================*/

class InitDiffusionGrid extends DefaultReporter {
	public Syntax getSyntax() {
		return Syntax.reporterSyntax(new int[] {Syntax.NumberType(), Syntax.NumberType(), Syntax.NumberType(), Syntax.NumberType()}, Syntax.WildcardType());
	}
  
	public Object report(Argument args[], Context context) throws ExtensionException, LogoException {
		int width = args[0].getIntValue();  
		int height = args[1].getIntValue();
		int lox = args[2].getIntValue();
		int loy = args[3].getIntValue();
		return new DiffusionGrid (width, height, lox, loy);
	}
}

class SetDiffusionGridCell extends DefaultCommand {
	public Syntax getSyntax() {
		return Syntax.commandSyntax(new int[] {Syntax.WildcardType(), Syntax.NumberType(), Syntax.NumberType(), Syntax.NumberType()});
	}
  
	public void perform(Argument args[], Context context) throws ExtensionException, LogoException {
		Object g = args[0].get();
		if (g instanceof DiffusionGrid) {
			int x = args[1].getIntValue();  
			int y = args[2].getIntValue();
			double v = args[3].getDoubleValue();
			((DiffusionGrid) g).setValue (x, y, v);
		} else throw new ExtensionException(g + " is not a DiffusionGrid") ;
	}
}

class GetDiffusionGridCell extends DefaultReporter {
	public Syntax getSyntax() {
		return Syntax.reporterSyntax(new int[] {Syntax.WildcardType(), Syntax.NumberType(), Syntax.NumberType()}, Syntax.NumberType());
	}
  
	public Object report(Argument args[], Context context) throws ExtensionException, LogoException {
		Object g = args[0].get();
		if (g instanceof DiffusionGrid) {
			int x = args[1].getIntValue();  
			int y = args[2].getIntValue();
			return new Double (((DiffusionGrid) g).getValue (x, y));
		} else throw new ExtensionException(g + " is not a DiffusionGrid") ;
	}
}

class DiffusionStep extends DefaultCommand {
	public Syntax getSyntax() {
		return Syntax.commandSyntax(new int[] {Syntax.WildcardType(), Syntax.NumberType()});
	}
  
	public void perform(Argument args[], Context context) throws ExtensionException, LogoException {
		Object g = args[0].get();
		if (g instanceof DiffusionGrid) {
			double diffusionFactor = args[1].getDoubleValue();  
			((DiffusionGrid) g).step (diffusionFactor);
		} else throw new ExtensionException(g + " is not a DiffusionGrid") ;
	}
}

class SetGridFromPatches extends DefaultCommand {
	public Syntax getSyntax() {
		return Syntax.commandSyntax(new int[] {Syntax.WildcardType(), Syntax.AgentsetType()});
	}
  
	public void perform(Argument args[], Context context) throws ExtensionException, LogoException {
		Object g = args[0].get();
		if (g instanceof DiffusionGrid) {
			DiffusionGrid dg = (DiffusionGrid) g;
			AgentSet aset = args[1].getAgentSet(); 
			for (Agent a : aset.agents()) {
				if (a instanceof Patch) {
					Patch p = (Patch) a;
					int x = p.pxcor();
					int y = p.pycor();
					Object v = p.getVariable(DiffusionExtension.PATCH_VALUE_INDEX);
					if (v instanceof Double) dg.setValue (x, y, ((Double) v).doubleValue ());
					else throw new ExtensionException("Did not find numeric patch variable in slot " + DiffusionExtension.PATCH_VALUE_INDEX);
				} else throw new ExtensionException("Non-patch agents passed into copy-patches-to-grid") ;
			}
		} else throw new ExtensionException(g + " is not a DiffusionGrid") ;
	}
}
class SetPatchesFromGrid extends DefaultCommand {
	public Syntax getSyntax() {
		return Syntax.commandSyntax(new int[] {Syntax.WildcardType(), Syntax.AgentsetType()});
	}
  
	public void perform(Argument args[], Context context) throws ExtensionException, LogoException {
		Object g = args[0].get();
		if (g instanceof DiffusionGrid) {
			DiffusionGrid dg = (DiffusionGrid) g;
			AgentSet aset = args[1].getAgentSet(); 
			for (Agent a : aset.agents()) {
				if (a instanceof Patch) {
					Patch p = (Patch) a;
					int x = p.pxcor();
					int y = p.pycor();
					double v = dg.getValue (x, y);
					try {
						p.setVariable(DiffusionExtension.PATCH_VALUE_INDEX, new Double (v));
					} catch (AgentException e) {
						throw new ExtensionException("Could not set variable in slot " + DiffusionExtension.PATCH_VALUE_INDEX + " because of " + e);
					}
				} else throw new ExtensionException("Non-patch agents passed into copy-grid-to-patches") ;
			}
		} else throw new ExtensionException(g + " is not a DiffusionGrid") ;
	}
}